from collections.abc import Generator
import json
from pathlib import Path
import shutil
from typing import TYPE_CHECKING
from typing import Any
from uuid import uuid4

from ase import Atoms
from ase.calculators.calculator import Calculator
import numpy as np
import pytest

from autojob import SETTINGS
from autojob.bases.task_base import TASK_GROUP_FIELDS
from autojob.next import create_next_step
from autojob.next import create_task_group_tree
from autojob.next import create_task_tree
from autojob.next import finalize_task
from autojob.next import initialize_task
from autojob.next import substitute_context
from autojob.next.restart import _to_exclude_on_restart
from autojob.parametrizations import VariableReference
from autojob.parametrizations import create_parametrization
from autojob.study import STUDY_FIELDS
from autojob.tasks.calculation import Calculation
from autojob.tasks.task import Task
from autojob.tasks.task import TaskInputs
from autojob.tasks.task import TaskMetadata
from autojob.tasks.task import TaskOutputs
from autojob.workflow import Step

if TYPE_CHECKING:
    from autojob.bases.task_base import TaskBase

# TODO: Unify fixtures


class TestFinalizeTask:
    @staticmethod
    @pytest.fixture(name="task_metadata")
    def fixture_task_metadata() -> "TaskMetadata":
        task_metadata: TaskMetadata = TaskMetadata(
            study_id=uuid4(), task_group_id=uuid4()
        )
        return task_metadata

    @staticmethod
    @pytest.fixture(name="input_atoms")
    def fixture_input_atoms() -> Atoms:
        return Atoms("C")

    @staticmethod
    @pytest.fixture(name="task_inputs")
    def fixture_task_inputs(input_atoms: Atoms) -> TaskInputs:
        return TaskInputs(atoms=input_atoms)

    @staticmethod
    @pytest.fixture(name="task_outputs")
    def fixture_task_outputs(input_atoms: Atoms) -> TaskOutputs:
        return TaskOutputs(atoms=input_atoms)

    @staticmethod
    @pytest.fixture(name="task", params=[Task, Calculation])
    def fixture_task(
        request: pytest.FixtureRequest,
        task_metadata: TaskMetadata,
        task_inputs: TaskInputs,
        task_outputs: TaskOutputs,
    ) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task = task_class(
            task_metadata=task_metadata,
            task_inputs=task_inputs,
            task_outputs=task_outputs,
        )
        return task

    @staticmethod
    @pytest.fixture(name="study_dir")
    def fixture_study_dir(
        tmp_path_factory: pytest.TempPathFactory,
        task_metadata: TaskMetadata,
    ) -> Path:
        study_dir: Path = tmp_path_factory.mktemp(str(task_metadata.study_id))
        Path(study_dir, SETTINGS.RECORD_FILE).touch()
        return study_dir

    @staticmethod
    @pytest.fixture(name="write_task_dir")
    def fixture_write_task_dir(study_dir: Path, task: "TaskBase") -> Path:
        task_dir = Path(
            study_dir,
            str(task.task_metadata.task_group_id),
            str(task.task_metadata.task_id),
        )
        task_dir.mkdir(parents=True)
        task.write_inputs(task_dir)
        task.task_outputs.atoms.write(
            Path(task_dir, SETTINGS.OUTPUT_ATOMS_FILE)
        )
        return task_dir

    @staticmethod
    def test_should_record_task_if_record_task_is_true(
        study_dir: Path,
        write_task_dir: Path,
        task: "TaskBase",
    ) -> None:
        finalize_task(write_task_dir, task, True)
        record_file = Path(study_dir, SETTINGS.RECORD_FILE)
        with record_file.open(mode="r", encoding="utf-8") as file:
            text = file.read()
        assert str(task.task_metadata.task_id) in text

    @staticmethod
    def test_should_not_record_task_if_record_task_is_false(
        study_dir: Path,
        write_task_dir: Path,
        task: "TaskBase",
    ) -> None:
        finalize_task(write_task_dir, task, False)
        record_file = Path(study_dir, SETTINGS.RECORD_FILE)
        with record_file.open(mode="r", encoding="utf-8") as file:
            text = file.read()
        assert str(task.task_metadata.task_id) not in text

    @staticmethod
    def test_should_archive_task_if_archive_file_not_present(
        write_task_dir: Path,
        task: "TaskBase",
    ) -> None:
        finalize_task(write_task_dir, task)
        archive_file = Path(write_task_dir, SETTINGS.ARCHIVE_FILE)
        assert archive_file.exists()

    @staticmethod
    def test_should_not_archive_task_if_archive_file_present(
        write_task_dir: Path,
        task: "TaskBase",
    ) -> None:
        archive_file = Path(write_task_dir, SETTINGS.ARCHIVE_FILE)
        archive_file.touch()
        created = archive_file.stat().st_mtime
        finalize_task(write_task_dir, task)
        assert created == archive_file.stat().st_mtime


class TestSubstituteContext:
    @staticmethod
    @pytest.fixture(name="context")
    def fixture_context() -> dict[str, Any]:
        return TaskMetadata().model_dump(mode="json")

    @staticmethod
    @pytest.fixture(name="key", params=["label", "task_id", "date_created"])
    def fixture_key(request: pytest.FixtureRequest) -> str:
        key: str = request.param
        return key

    @staticmethod
    def test_should_substitute_values(
        context: dict[str, Any], key: str
    ) -> None:
        placeholder = key
        template = "{" + placeholder + "}"
        mods = {key: template}
        new_mods = substitute_context(mods, context)
        assert new_mods[key] == context[key]

    @staticmethod
    @pytest.mark.parametrize("value", [1, 1.0, None])
    def test_should_not_substitute_non_strings(value: Any) -> None:
        key = "time"

        mods = {key: value}
        context = {}

        new_mods = substitute_context(mods, context)
        assert new_mods[key] == value


class TestInitializeTask:
    @staticmethod
    @pytest.fixture(name="task_class", params=[Task, Calculation])
    def fixture_task_class(request: pytest.FixtureRequest) -> "type[TaskBase]":
        task_class: type[TaskBase] = request.param
        return task_class

    @staticmethod
    @pytest.fixture(name="task")
    def fixture_task(task_class: "type[TaskBase]") -> "TaskBase":
        return task_class()

    @staticmethod
    @pytest.fixture(name="parametrization")
    def fixture_parametrization(task: "TaskBase") -> list[VariableReference]:
        parametrization = create_parametrization(task)
        return parametrization

    @staticmethod
    def test_should_create_equivalent_task_when_parametrization_inherits_all_parameters(
        task_class: "type[TaskBase]",
        parametrization: list[VariableReference],
        task: "TaskBase",
    ) -> None:
        new_task = initialize_task(
            task_class=task_class.__name__.lower(),
            parametrization=parametrization,
            previous_task=task,
        )

        assert task.task_metadata.model_dump(
            exclude={"tags"}
        ) == new_task.task_metadata.model_dump(exclude={"tags"})
        assert task.model_dump(
            exclude={"task_metadata"}
        ) == new_task.model_dump(exclude={"task_metadata"})

    @staticmethod
    def test_should_return_correct_task_type(
        task_class: "type[TaskBase]",
        parametrization: list[VariableReference],
        task: "TaskBase",
    ) -> None:
        new_task = initialize_task(
            task_class=task_class.__name__.lower(),
            parametrization=parametrization,
            previous_task=task,
        )
        assert isinstance(new_task, task_class)

    @staticmethod
    def test_should_include_previous_task_id_in_new_task_tags(
        task_class: "type[TaskBase]",
        parametrization: list[VariableReference],
        task: "TaskBase",
    ) -> None:
        new_task = initialize_task(
            task_class=task_class.__name__.lower(),
            parametrization=parametrization,
            previous_task=task,
        )
        assert str(task.task_metadata.task_id) in new_task.task_metadata.tags

    @staticmethod
    def test_should_include_previous_task_group_id_in_new_task_tags_if_restart_false(
        task_class: "type[TaskBase]",
        parametrization: list[VariableReference],
        task: "TaskBase",
    ) -> None:
        new_task = initialize_task(
            task_class=task_class.__name__.lower(),
            parametrization=parametrization,
            previous_task=task,
            restart=False,
        )
        assert (
            str(task.task_metadata.task_group_id)
            in new_task.task_metadata.tags
        )

    @staticmethod
    def test_should_not_include_previous_task_group_id_in_new_task_tags_if_restart_true(
        task_class: "type[TaskBase]",
        parametrization: list[VariableReference],
        task: "TaskBase",
    ) -> None:
        new_task = initialize_task(
            task_class=task_class.__name__.lower(),
            parametrization=parametrization,
            previous_task=task,
            restart=True,
        )
        assert (
            str(task.task_metadata.task_group_id)
            not in new_task.task_metadata.tags
        )


@pytest.mark.usefixtures("write_study_metadata")
class TestCreateTaskGroupTree:
    @staticmethod
    @pytest.fixture(name="task", params=[Task, Calculation])
    def fixture_task(request: pytest.FixtureRequest) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        return task

    @staticmethod
    @pytest.fixture(name="previous", params=[Task, Calculation])
    def fixture_previous(request: pytest.FixtureRequest) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        return task

    @staticmethod
    @pytest.fixture(name="src")
    def fixture_src(
        tmp_path: Path, previous: "TaskBase"
    ) -> Generator[Path, None, None]:
        task_group_id = str(previous.task_metadata.task_group_id)
        task_id = str(previous.task_metadata.task_id)
        src = Path(tmp_path, task_group_id, task_id)
        src.mkdir(parents=True)
        yield src
        shutil.rmtree(src)

    @staticmethod
    @pytest.fixture(name="dest")
    def fixture_dest(src: Path) -> Path:
        return Path(src.parents[1])

    @staticmethod
    @pytest.fixture(name="write_study_metadata")
    def fixture_write_study_metadata(
        previous: "TaskBase", dest: Path
    ) -> Generator[Path, None, None]:
        metadata_file = Path(dest, SETTINGS.STUDY_METADATA_FILE)
        metadata = previous.task_metadata.model_dump(
            mode="json", include=STUDY_FIELDS
        )
        metadata["task_groups"] = [str(previous.task_metadata.task_group_id)]
        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)
        yield metadata_file
        metadata_file.unlink()

    @staticmethod
    def test_should_create_task_group_directory_with_task_group_id_if_name_template_is_none(
        src: Path, task: "TaskBase", dest: Path
    ) -> None:
        created = create_task_group_tree(src=src, task=task, dest=dest)
        assert created.exists()
        assert created.is_dir()
        assert created == Path(dest, str(task.task_metadata.task_group_id))

    @staticmethod
    def test_should_create_task_group_metadata_file_in_new_task_group_directory(
        src: Path, task: "TaskBase", dest: Path
    ) -> None:
        created = create_task_group_tree(src=src, task=task, dest=dest)
        metadata_file = Path(created, SETTINGS.TASK_GROUP_METADATA_FILE)
        assert metadata_file.exists()

    @staticmethod
    def test_should_create_task_group_metadata_file_with_metadata_matching_new_task(
        src: Path, task: "TaskBase", dest: Path
    ) -> None:
        created = create_task_group_tree(src=src, task=task, dest=dest)
        metadata_file = Path(created, SETTINGS.TASK_GROUP_METADATA_FILE)
        with metadata_file.open(mode="r", encoding="utf-8") as file:
            metadata: dict[str, Any] = json.load(file)

        task_metadata = task.task_metadata.model_dump(
            mode="json",
            include=TASK_GROUP_FIELDS,
        )
        metadata_matching = []

        for key, value in task_metadata.items():
            metadata_matching.append(value == metadata[key])

        assert all(metadata_matching)
        assert str(task.task_metadata.task_id) in metadata["tasks"]

    @staticmethod
    @pytest.mark.parametrize("name_template", ["name"])
    def test_should_create_new_task_group_directory_with_name_template(
        src: Path,
        task: "TaskBase",
        dest: Path,
        name_template: str,
    ) -> None:
        created = create_task_group_tree(
            src=src,
            task=task,
            dest=dest,
            name_template=name_template,
        )
        task_group_dir = Path(dest, name_template)
        assert task_group_dir.exists()
        assert created == task_group_dir

    # This test is necessary to make sure that the contents temporary directory are not
    # used to check for naming conflicts conflicts
    @staticmethod
    @pytest.mark.parametrize("name_template", ["name_{i}"])
    def test_should_create_new_task_group_directory_with_conflict_free_name_template(
        src: Path,
        task: "TaskBase",
        dest: Path,
        name_template: str,
    ) -> None:
        Path(src.parent, name_template.format(i=1)).mkdir()
        created = create_task_group_tree(
            src=src,
            task=task,
            dest=dest,
            name_template=name_template,
        )
        task_group_dir = Path(dest, name_template.format(i=2))
        assert task_group_dir.exists()
        assert created == task_group_dir

    @staticmethod
    def test_should_raise_value_error_if_task_group_id_is_none(
        src: Path,
        task: "TaskBase",
        dest: Path,
    ) -> None:
        task.task_metadata.task_group_id = None
        with pytest.raises(
            ValueError,
            match=r"Cannot create new task group without task group ID*",
        ):
            _ = create_task_group_tree(
                src=src,
                task=task,
                dest=dest,
            )


@pytest.mark.usefixtures("write_task_group_metadata")
class TestCreateTaskTree:
    @staticmethod
    @pytest.fixture(name="task", params=[Task, Calculation])
    def fixture_task(request: pytest.FixtureRequest) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        return task

    @staticmethod
    @pytest.fixture(name="previous", params=[Task, Calculation])
    def fixture_previous(request: pytest.FixtureRequest) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        return task

    @staticmethod
    @pytest.fixture(name="src")
    def fixture_src(
        tmp_path: Path, previous: "TaskBase"
    ) -> Generator[Path, None, None]:
        task_id = str(previous.task_metadata.task_id)
        src = Path(tmp_path, task_id)
        src.mkdir(parents=True)
        yield src
        shutil.rmtree(src)

    @staticmethod
    @pytest.fixture(name="dest")
    def fixture_dest(src: Path) -> Path:
        return Path(src.parent)

    @staticmethod
    @pytest.fixture(name="write_task_group_metadata")
    def fixture_write_task_group_metadata(
        previous: "TaskBase", dest: Path
    ) -> Generator[Path, None, None]:
        metadata_file = Path(dest, SETTINGS.TASK_GROUP_METADATA_FILE)
        metadata = previous.task_metadata.model_dump(
            mode="json", include=TASK_GROUP_FIELDS
        )
        metadata["tasks"] = [str(previous.task_metadata.task_id)]
        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)
        yield metadata_file
        metadata_file.unlink()

    @staticmethod
    @pytest.fixture(name="files_to_carry_over")
    def fixture_files_to_carry_over(
        src: Path,
    ) -> Generator[list[str], None, None]:
        file = Path(src, "file.txt")
        directory = Path(src, "directory")
        file.touch()
        directory.mkdir()
        yield [file.name, directory.name]
        file.unlink()
        shutil.rmtree(directory)

    @staticmethod
    def test_should_create_task_directory_with_task_id_if_name_template_is_none(
        src: Path,
        task: "TaskBase",
        dest: Path,
    ) -> None:
        created = create_task_tree(
            task=task,
            dest=dest,
            src=src,
        )
        assert created.exists()
        assert created.is_dir()
        assert created == Path(dest, str(task.task_metadata.task_id))

    @staticmethod
    def test_should_create_task_metadata_file_in_new_task_directory(
        src: Path,
        task: "TaskBase",
        dest: Path,
    ) -> None:
        created = create_task_tree(
            task=task,
            dest=dest,
            src=src,
        )
        metadata_file = Path(created, SETTINGS.TASK_METADATA_FILE)
        assert metadata_file.exists()

    @staticmethod
    def test_should_create_task_metadata_file_with_metadata_matching_new_task(
        src: Path,
        task: "TaskBase",
        dest: Path,
    ) -> None:
        created = create_task_tree(
            task=task,
            dest=dest,
            src=src,
        )
        metadata_file = Path(created, SETTINGS.TASK_METADATA_FILE)
        with metadata_file.open(mode="r", encoding="utf-8") as file:
            metadata: dict[str, Any] = json.load(file)

        task_metadata = task.task_metadata.model_dump(mode="json")
        metadata_matching = []

        for key, value in task_metadata.items():
            metadata_matching.append(value == metadata[key])

        assert all(metadata_matching)

    @staticmethod
    @pytest.mark.parametrize("name_template", ["name"])
    def test_should_create_new_task_directory_with_name_template(
        src: Path,
        task: "TaskBase",
        dest: Path,
        name_template: str,
    ) -> None:
        created = create_task_tree(
            task=task,
            dest=dest,
            src=src,
            name_template=name_template,
        )
        task_dir = Path(dest, name_template)
        assert task_dir.exists()
        assert created == task_dir

    # This test is necessary to make sure that the contents temporary directory are not
    # used to check for naming conflicts conflicts
    @staticmethod
    @pytest.mark.parametrize("name_template", ["name_{i}"])
    def test_should_create_new_task_directory_with_conflict_free_name_template(
        src: Path,
        task: "TaskBase",
        dest: Path,
        name_template: str,
    ) -> None:
        Path(src.parent, name_template.format(i=1)).mkdir()
        created = create_task_tree(
            task=task,
            dest=dest,
            src=src,
            name_template=name_template,
        )
        task_group_dir = Path(dest, name_template.format(i=2))
        assert task_group_dir.exists()
        assert created == task_group_dir

    @staticmethod
    def test_should_copy_files_to_carry_over_to_new_task_directory(
        src: Path, task: "TaskBase", dest: Path, files_to_carry_over: list[str]
    ) -> None:
        created = create_task_tree(
            src=src,
            task=task,
            dest=dest,
            files_to_carry_over=files_to_carry_over,
        )
        files_carried_over = []
        for f in files_to_carry_over:
            files_carried_over.append(Path(created, f).exists())
        assert all(files_carried_over)

    @staticmethod
    def test_should_not_throw_error_for_missing_carry_over_files(
        src: Path, task: "TaskBase", dest: Path
    ) -> None:
        _ = create_task_tree(
            src=src,
            task=task,
            dest=dest,
            files_to_carry_over=["fake_file"],
        )


@pytest.mark.usefixtures("write_study_metadata", "write_task_group_metadata")
class TestCreateNextStep:
    @staticmethod
    @pytest.fixture(name="files_to_delete")
    def fixture_files_to_delete() -> list[str]:
        return []

    @staticmethod
    @pytest.fixture(name="task", params=[Task, Calculation])
    def fixture_task(request: pytest.FixtureRequest) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        return task

    # TODO: use calculate fixture
    @staticmethod
    @pytest.fixture(name="previous", params=[Task, Calculation])
    def fixture_previous(
        request: pytest.FixtureRequest, files_to_delete: list[str]
    ) -> "TaskBase":
        task_class: type[TaskBase] = request.param
        task: TaskBase = task_class()
        task.task_inputs.files_to_delete = files_to_delete

        # Set input/output atoms and calculator
        task.task_inputs.atoms = Atoms("C")
        calc = Calculator()
        calc.results = {"forces": np.zeros(3), "energy": 0.0}
        task.task_inputs.atoms.calc = calc
        task.task_outputs = TaskOutputs(atoms=task.task_inputs.atoms)
        if hasattr(task, "calculation_inputs"):
            task.calculation_inputs.calculator = "default"

        return task

    # TODO: use calculate fixture
    @staticmethod
    @pytest.fixture(name="src")
    def fixture_src(
        tmp_path: Path, previous: "TaskBase"
    ) -> Generator[Path, None, None]:
        task_group_id = str(previous.task_metadata.task_group_id)
        task_id = str(previous.task_metadata.task_id)
        src = Path(tmp_path, task_group_id, task_id)
        src.mkdir(parents=True)
        _ = previous.write_inputs(src)
        previous.task_outputs.atoms.write(
            Path(src, SETTINGS.OUTPUT_ATOMS_FILE)
        )
        yield src
        shutil.rmtree(tmp_path)

    @staticmethod
    @pytest.fixture(name="write_study_metadata")
    def fixture_write_study_metadata(
        previous: "TaskBase", src: Path
    ) -> Generator[Path, None, None]:
        metadata_file = Path(src.parents[1], SETTINGS.STUDY_METADATA_FILE)
        metadata = previous.task_metadata.model_dump(
            mode="json", include=STUDY_FIELDS
        )
        metadata["task_groups"] = [str(previous.task_metadata.task_group_id)]
        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)
        yield metadata_file
        metadata_file.unlink()

    @staticmethod
    @pytest.fixture(name="write_task_group_metadata")
    def fixture_write_task_group_metadata(
        previous: "TaskBase", src: Path
    ) -> Generator[Path, None, None]:
        metadata_file = Path(src.parent, SETTINGS.TASK_GROUP_METADATA_FILE)
        metadata = previous.task_metadata.model_dump(
            mode="json", include=TASK_GROUP_FIELDS
        )
        metadata["tasks"] = [str(previous.task_metadata.task_id)]
        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)
        yield metadata_file
        metadata_file.unlink()

    @staticmethod
    @pytest.fixture(name="restart")
    def fixture_restart(request: pytest.FixtureRequest) -> bool:
        return bool(request.param)

    @staticmethod
    @pytest.fixture(name="parametrization")
    def fixture_parametrization(
        task: "TaskBase", restart: bool
    ) -> list[VariableReference]:
        parametrization = create_parametrization(
            task, exclude_metadata=_to_exclude_on_restart if restart else []
        )
        return parametrization

    @staticmethod
    @pytest.fixture(name="step")
    def fixture_step(
        parametrization: list[VariableReference], task: "TaskBase"
    ) -> Step:
        step = Step(
            workflow_step_id=task.task_metadata.workflow_step_id,
            task_class=task.task_metadata.task_class,
            progression="independent",
            parametrizations=[parametrization],
        )
        return step

    @staticmethod
    @pytest.mark.parametrize("restart", [True], indirect=True)
    def test_should_create_new_task_in_parent_directory_of_source_on_restart(
        src: Path, step: Step, previous: "TaskBase", restart: bool
    ) -> None:
        _, created_task_dir = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                )
            )
        )
        assert created_task_dir.parent == src.parent

    @staticmethod
    @pytest.mark.parametrize("restart", [False], indirect=True)
    def test_should_create_new_task_group_in_second_parent_directory_of_source_on_not_restart(
        src: Path, step: Step, previous: "TaskBase", restart: bool
    ) -> None:
        _, created_task_dir = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                )
            )
        )
        assert created_task_dir.parent == src.parent.parent

    @staticmethod
    @pytest.mark.parametrize("restart", [True, False], indirect=True)
    def test_should_add_new_task_to_task_group_metadata_file(
        src: Path, step: Step, previous: "TaskBase", restart: bool
    ) -> None:
        created_task, created_task_dir = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                )
            )
        )
        root = created_task_dir.parent if restart else created_task_dir
        metadata_file = Path(root, SETTINGS.TASK_GROUP_METADATA_FILE)
        with metadata_file.open(mode="r", encoding="utf-8") as file:
            metadata = json.load(file)
        task_id = str(created_task.task_metadata.task_id)
        assert task_id in metadata["tasks"]

    @staticmethod
    @pytest.mark.parametrize("restart", [False], indirect=True)
    def test_should_add_new_task_group_to_study_metadata_file_if_restart_is_false(
        src: Path, step: Step, previous: "TaskBase", restart: bool
    ) -> None:
        created_task, created_task_dir = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                )
            )
        )
        root = created_task_dir.parent
        metadata_file = Path(root, SETTINGS.STUDY_METADATA_FILE)
        with metadata_file.open(mode="r", encoding="utf-8") as file:
            metadata = json.load(file)
        task_group_id = str(created_task.task_metadata.task_group_id)
        assert task_group_id in metadata["task_groups"]

    @staticmethod
    @pytest.mark.parametrize("file_size_limit", [-1.0, 0.0, 1e3])
    @pytest.mark.parametrize("restart", [True, False], indirect=True)
    def test_should_delete_files_over_file_limit(
        src: Path,
        step: Step,
        previous: "TaskBase",
        restart: bool,
        file_size_limit: float,
    ) -> None:
        _ = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                    file_size_limit=file_size_limit,
                )
            )
        )
        files_exceeding_size_limit = []
        for f in src.iterdir():
            files_exceeding_size_limit.append(
                f.stat().st_size > file_size_limit
            )

        assert not any(files_exceeding_size_limit)

    @staticmethod
    @pytest.mark.parametrize(
        "files_to_delete", [[], [SETTINGS.TASK_METADATA_FILE]]
    )
    @pytest.mark.parametrize("restart", [True, False], indirect=True)
    def test_should_delete_files_to_delete(
        src: Path,
        step: Step,
        previous: "TaskBase",
        restart: bool,
        files_to_delete: list[str],
    ) -> None:
        _ = next(
            iter(
                create_next_step(
                    src=src,
                    step=step,
                    previous_task=previous,
                    restart=restart,
                    submit=False,
                )
            )
        )
        files_to_delete_remaining = []
        for f in src.iterdir():
            files_to_delete_remaining.append(f.name in files_to_delete)

        assert not any(files_to_delete_remaining)

    # TODO: Pending: implementation of TaskInputs.task_script(_template)
    # TODO: write task script template that, when templated, is a script that
    # TODO: creates a directory ("task_dir") in the submission directory and
    # TODO: returns the test should confirm that the directory has been created
    @staticmethod
    @pytest.mark.xfail(reason="Not implemented")
    def test_should_submit_local_job() -> None:
        submit_local_job = False
        assert submit_local_job
