from datetime import UTC
from datetime import datetime
import math
from pathlib import Path

import pytest

from autojob._cli.run import execute_command


class TestRunScript:
    @staticmethod
    def test_should_execute_script(tmp_path: Path) -> None:
        to_create = Path(tmp_path, "testdir")
        text = ["from pathlib import Path\n", f"Path('{to_create}').mkdir()\n"]
        slurm_script = Path(tmp_path, "run.py")

        with slurm_script.open(mode="w", encoding="utf-8") as file:
            file.writelines(text)

        with pytest.raises(SystemExit) as exc_info:
            execute_command(
                command=["python", str(slurm_script)],
                time_limit=math.inf,
                frequency=1,
            )
        assert exc_info.value.code == 0
        assert to_create.exists()

    @staticmethod
    def test_should_stop_script_execution_at_time_limit_without_buffer(
        tmp_path: Path,
    ) -> None:
        text = ["import time\n", "time.sleep(15)\n"]
        slurm_script = Path(tmp_path, "run.py")

        with slurm_script.open(mode="w", encoding="utf-8") as file:
            file.writelines(text)

        start_time = datetime.now(tz=UTC)
        time_limit = 5

        with pytest.raises(SystemExit) as exc_info:
            execute_command(
                command=["python", str(slurm_script)],
                time_limit=time_limit,
                frequency=1,
            )

        end_time = datetime.now(tz=UTC)

        assert exc_info.value.code == 124
        run_time = (end_time - start_time).total_seconds()
        # Give 100ms-second (extra) buffer
        assert time_limit < run_time < time_limit + 0.1

    @staticmethod
    def test_should_stop_script_execution_before_time_limit_with_absolute_buffer(
        tmp_path: Path,
    ) -> None:
        text = ["import time\n", "time.sleep(15)\n"]
        slurm_script = Path(tmp_path, "run.py")

        with slurm_script.open(mode="w", encoding="utf-8") as file:
            file.writelines(text)

        start_time = datetime.now(tz=UTC)
        time_limit = 5
        buffer = 1

        with pytest.raises(SystemExit) as exc_info:
            execute_command(
                command=["python", str(slurm_script)],
                time_limit=time_limit,
                frequency=1,
                buffer=buffer,
            )

        end_time = datetime.now(tz=UTC)

        assert exc_info.value.code == 124
        run_time = (end_time - start_time).total_seconds()
        expected_run_time = time_limit - buffer
        # Give 100ms-second (extra) buffer
        assert abs(run_time - expected_run_time) < 0.1

    @staticmethod
    def test_should_stop_script_execution_before_time_limit_with_fractional_buffer(
        tmp_path: Path,
    ) -> None:
        text = ["import time\n", "time.sleep(15)\n"]
        slurm_script = Path(tmp_path, "run.py")

        with slurm_script.open(mode="w", encoding="utf-8") as file:
            file.writelines(text)

        start_time = datetime.now(tz=UTC)
        time_limit = 10
        buffer = 0.5

        with pytest.raises(SystemExit) as exc_info:
            execute_command(
                command=["python", str(slurm_script)],
                time_limit=time_limit,
                frequency=1,
                buffer=buffer,
            )

        end_time = datetime.now(tz=UTC)

        assert exc_info.value.code == 124
        run_time = (end_time - start_time).total_seconds()
        expected_run_time = time_limit * (1 - buffer)
        # Give 100ms-second (extra) buffer
        assert abs(run_time - expected_run_time) < 0.1
