from datetime import UTC
from datetime import datetime
from pathlib import Path

import pytest

from autojob import SETTINGS
from autojob.study import Study
from autojob.study_group import StudyGroup
from autojob.tasks.calculation import Calculation


class TestToDirectory:
    @staticmethod
    def test_should_create_directory_using_name_template(
        tmp_path: Path, study_group: StudyGroup
    ) -> None:
        study_group.name = "name"
        study_group_directory = study_group.to_directory(
            tmp_path, study_group_template="{name}"
        )
        expected = Path(tmp_path, study_group.name)
        assert study_group_directory == expected
        assert expected.exists()

    @staticmethod
    def test_should_create_study_directory_as_immediate_child_of_parent_directory(
        study_group: StudyGroup, tmp_path: Path
    ) -> None:
        study_group_directory = study_group.to_directory(tmp_path)

        studies_created = []
        files = [f.name for f in study_group_directory.iterdir()]

        for study in study_group.studies:
            study_id = str(study.study_id)
            study_directory = Path(
                study_group_directory, study_id, SETTINGS.STUDY_METADATA_FILE
            )
            studies_created.append(
                study_id in files
                # This check ensures that the directory is in fact a study
                # directory and not simply named as such
                and study_directory.exists()
            )

        assert all(studies_created)


class TestToDirectoryForCalculations(TestToDirectory):
    @staticmethod
    @pytest.fixture(name="study")
    def fixture_study(
        calculation: Calculation, study_id: str, study_group_id: str
    ) -> Study:
        study = Study(
            tasks=[calculation],
            date_created=datetime.now(tz=UTC),
            study_id=(study_id),
            study_group_id=study_group_id,
        )
        return study
