import json
from pathlib import Path
from typing import Any
from uuid import uuid4

from pydantic import UUID4
from pydantic import ValidationError
import pytest

from autojob import SETTINGS
from autojob.tasks.task import TaskMetadata
from autojob.utils.files import get_uri

LEGACY_KEYS = [
    "Name",
    "Notes",
    "Study Group ID",
    "Study ID",
    "Calculation ID",
    "Job ID",
]


@pytest.fixture(name="label", params=["label"])
def fixture_label(request: pytest.FixtureRequest) -> str:
    label: str = request.param
    return label


@pytest.fixture(name="tags", params=[["tag1", "tag2"]])
def fixture_tags(request: pytest.FixtureRequest) -> list[str]:
    tags: list[str] = request.param
    return tags


@pytest.fixture(name="uri", params=[True, False])
def fixture_uri(tmp_path: Path, request: pytest.FixtureRequest) -> str | None:
    if request.param:
        return get_uri(tmp_path)
    return None


@pytest.fixture(name="study_group_id", params=[True, False])
def fixture_study_group_id(
    id_stem: str, request: pytest.FixtureRequest
) -> UUID4 | str:
    if request.param:
        return uuid4()
    return f"g{id_stem}"


@pytest.fixture(name="study_id", params=[True, False])
def fixture_study_id(
    id_stem: str, request: pytest.FixtureRequest
) -> UUID4 | str:
    if request.param:
        return uuid4()
    return f"s{id_stem}"


@pytest.fixture(name="workflow_step_id", params=[True, False])
def fixture_workflow_step_id(request: pytest.FixtureRequest) -> UUID4 | None:
    if request.param:
        return uuid4()
    return None


@pytest.fixture(name="calculation_id")
def fixture_calculation_id(id_stem: str) -> str:
    return f"c{id_stem}"


@pytest.fixture(name="task_id")
def fixture_task_id() -> UUID4:
    return uuid4()


@pytest.fixture(name="job_id")
def fixture_job_id(id_stem: str) -> str:
    return f"j{id_stem}"


@pytest.fixture(name="calculator")
def fixture_calculator() -> str:
    return "gaussian"


@pytest.fixture(
    name="id_stem",
    params=[
        "111111111",
    ],
)
def fixture_id_stem(request: pytest.FixtureRequest) -> str:
    id_stem: str = request.param
    return id_stem


class TestTaskMetadataValidation:
    @staticmethod
    @pytest.fixture(
        name="id_name",
        params=["study_group_id", "study_id", "task_group_id", "task_id"],
    )
    def fixture_id_name(request: pytest.FixtureRequest) -> str:
        return str(request.param)

    @staticmethod
    @pytest.fixture(
        name="id_value", params=["123456789", "abcdefghi", "123abc456"]
    )
    def fixture_id_value(id_name: str, request: pytest.FixtureRequest) -> str:
        prefix = ""
        match id_name:
            case "study_group_id":
                prefix = "g"
            case "study_id":
                prefix = "s"
            case "task_group_id":
                prefix = "c"
            case "task_id":
                prefix = "j"

        return f"{prefix}{request.param}"

    @staticmethod
    def test_should_validate_legacy_ids(id_name: str, id_value: str) -> None:
        metadata = {id_name: id_value}
        assert TaskMetadata(**metadata)

    @staticmethod
    def test_should_validate_uuids(id_name: str) -> None:
        metadata = {id_name: "a586c920-92ee-401b-9289-6a8ed2860377"}
        assert TaskMetadata(**metadata)

    @staticmethod
    @pytest.mark.parametrize("bad_id_value", [1, "1", "a"])
    def test_should_fail_validation_on_bad_ids(
        id_name: str, bad_id_value: Any
    ) -> None:
        metadata = {id_name: bad_id_value}
        with pytest.raises(ValidationError):
            TaskMetadata(**metadata)


class TestFromDirectory:
    @staticmethod
    def test_should_load_equivalent_metadata_from_directory(
        tmp_path: Path,
    ) -> None:
        metadata_file = Path(tmp_path, SETTINGS.TASK_METADATA_FILE)
        # Don't check uri because uri is set in from_directory
        metadata = TaskMetadata().model_dump(mode="json", exclude={"uri"})

        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)

        loaded_metadata = TaskMetadata.from_directory(tmp_path).model_dump(
            mode="json", exclude={"uri"}
        )
        assert loaded_metadata == metadata

    @staticmethod
    def test_should_set_uri_to_source_directory_uri(
        tmp_path: Path,
    ) -> None:
        metadata_file = Path(tmp_path, SETTINGS.TASK_METADATA_FILE)
        metadata = TaskMetadata().model_dump(mode="json")

        with metadata_file.open(mode="w", encoding="utf-8") as file:
            json.dump(metadata, file)

        uri = get_uri(tmp_path)
        loaded_metadata = TaskMetadata.from_directory(tmp_path)
        assert uri == loaded_metadata.uri
