from datetime import timedelta
import logging
from pathlib import Path

import pytest

from autojob.utils.parsing import TimedeltaTuple
from autojob.utils.parsing import parse_job_stats_file
from autojob.utils.parsing import parse_time_from_slurm_script

logger = logging.getLogger(__name__)


def _fill(v: int) -> str:
    return f"0{v}" if v < 10 else str(v)


class TestTimedeltaTuple:
    @staticmethod
    @pytest.fixture(name="slurm_days", params=[1, 2])
    def fixture_slurm_days(request: pytest.FixtureRequest) -> int:
        slurm_day: int = request.param
        return slurm_day

    @staticmethod
    @pytest.fixture(name="slurm_hours", params=[0, 1])
    def fixture_slurm_hours(request: pytest.FixtureRequest) -> int:
        slurm_hours: int = request.param
        return slurm_hours

    @staticmethod
    @pytest.fixture(name="slurm_minutes", params=[0, 1])
    def fixture_slurm_minutes(request: pytest.FixtureRequest) -> int:
        slurm_minutes: int = request.param
        return slurm_minutes

    @staticmethod
    @pytest.fixture(name="slurm_seconds", params=[0, 1])
    def fixture_slurm_seconds(request: pytest.FixtureRequest) -> int:
        slurm_seconds: int = request.param
        return slurm_seconds

    @staticmethod
    def test_should_create_equivalent_timedelta(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
        elapsed: timedelta,
    ) -> None:
        assert (
            slurm_days,
            slurm_hours,
            slurm_minutes,
            slurm_seconds,
        ) == TimedeltaTuple.from_timedelta(delta=elapsed)

    @staticmethod
    def test_should_create_timedeltatuple_from_string1(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        string = f"{slurm_days}-{slurm_hours}:{slurm_minutes}:{slurm_seconds}"
        assert (
            slurm_days,
            slurm_hours,
            slurm_minutes,
            slurm_seconds,
        ) == TimedeltaTuple.from_string(string=string, time_format="slurm")

    @staticmethod
    def test_should_create_timedeltatuple_from_string2(
        slurm_days: int,
        slurm_hours: int,
    ) -> None:
        string = f"{slurm_days}-{slurm_hours}"
        assert (
            slurm_days,
            slurm_hours,
        ) == TimedeltaTuple.from_string(string=string, time_format="slurm")[:2]

    @staticmethod
    @pytest.mark.parametrize("invalid_string", ["a", "1.00.00", "$1 ", "\n"])
    def test_should_raise_value_error_from_string_with_invalid_string(
        invalid_string: str,
    ) -> None:
        with pytest.raises(ValueError):  # noqa: PT011
            _ = TimedeltaTuple.from_string(invalid_string)

    @staticmethod
    @pytest.mark.parametrize("invalid_string", ["a", "1.00.00", "$1 ", "\n"])
    def test_should_raise_value_error_from_slurm_time_with_invalid_string(
        invalid_string: str,
    ) -> None:
        with pytest.raises(ValueError):  # noqa: PT011
            _ = TimedeltaTuple.from_slurm_time(invalid_string)

    @staticmethod
    def test_should_accept_slurm_format_1(slurm_minutes: int) -> None:
        string = str(slurm_minutes)
        _, _, minutes, _ = TimedeltaTuple.from_slurm_time(time=string)
        assert minutes == slurm_minutes

    @staticmethod
    def test_should_accept_slurm_format_2(
        slurm_minutes: int, slurm_seconds: int
    ) -> None:
        string = f"{slurm_minutes}:{slurm_seconds}"
        _, _, minutes, seconds = TimedeltaTuple.from_slurm_time(time=string)
        assert (minutes, seconds) == (slurm_minutes, slurm_seconds)

    @staticmethod
    def test_should_accept_slurm_format_3(
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        string = f"{slurm_hours}:{slurm_minutes}:{slurm_seconds}"
        _, hours, minutes, seconds = TimedeltaTuple.from_slurm_time(
            time=string
        )
        assert (hours, minutes, seconds) == (
            slurm_hours,
            slurm_minutes,
            slurm_seconds,
        )

    @staticmethod
    def test_should_accept_slurm_format_4(
        slurm_days: int, slurm_hours: int
    ) -> None:
        string = f"{slurm_days}-{slurm_hours}"
        days, hours, _, _ = TimedeltaTuple.from_slurm_time(time=string)
        assert (days, hours) == (
            slurm_days,
            slurm_hours,
        )

    @staticmethod
    def test_should_accept_slurm_format_5(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
    ) -> None:
        string = f"{slurm_days}-{slurm_hours}:{slurm_minutes}"
        days, hours, minutes, _ = TimedeltaTuple.from_slurm_time(time=string)
        assert (days, hours, minutes) == (
            slurm_days,
            slurm_hours,
            slurm_minutes,
        )

    @staticmethod
    def test_should_accept_slurm_format_6(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        string = f"{slurm_days}-{slurm_hours}:{slurm_minutes}:{slurm_seconds}"
        days, hours, minutes, seconds = TimedeltaTuple.from_slurm_time(
            time=string
        )
        assert (days, hours, minutes, seconds) == (
            slurm_days,
            slurm_hours,
            slurm_minutes,
            slurm_seconds,
        )

    @staticmethod
    def test_should_reverse_from_timedelta(elapsed: timedelta) -> None:
        assert (
            elapsed
            == TimedeltaTuple.from_timedelta(delta=elapsed).to_timedelta()
        )

    @staticmethod
    def test_should_reverse_to_timedelta(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        delta = TimedeltaTuple(
            days=slurm_days,
            hours=slurm_hours,
            minutes=slurm_minutes,
            seconds=slurm_seconds,
        )
        assert delta == TimedeltaTuple.from_timedelta(delta.to_timedelta())

    @staticmethod
    def test_should_reverse_from_string(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        string = (
            f"{slurm_days}-{_fill(slurm_hours)}:{_fill(slurm_minutes)}:"
            f"{_fill(slurm_seconds)}"
        )
        assert (
            string == TimedeltaTuple.from_string(string=string).to_slurm_time()
        )

    @staticmethod
    def test_should_reverse_from_slurm_time(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        string = (
            f"{slurm_days}-{_fill(slurm_hours)}:{_fill(slurm_minutes)}:"
            f"{_fill(slurm_seconds)}"
        )
        assert (
            string
            == TimedeltaTuple.from_slurm_time(time=string).to_slurm_time()
        )

    @staticmethod
    def test_should_reverse_to_slurm_time(
        slurm_days: int,
        slurm_hours: int,
        slurm_minutes: int,
        slurm_seconds: int,
    ) -> None:
        delta = TimedeltaTuple(
            days=slurm_days,
            hours=slurm_hours,
            minutes=slurm_minutes,
            seconds=slurm_seconds,
        )
        assert delta == TimedeltaTuple.from_slurm_time(delta.to_slurm_time())


@pytest.mark.usefixtures("write_job_stats_file")
class TestParseJobStatsFile:
    @staticmethod
    def test_should_load_matching_dictionary(task_directory: Path) -> None:
        parsed_job_stats = parse_job_stats_file(task_directory)

        assert parsed_job_stats["elapsed"] == timedelta(seconds=4529)
        assert parsed_job_stats["idle_time"] == timedelta(seconds=72)
        assert parsed_job_stats["job_id"] == 14386652
        assert parsed_job_stats["max_rss"] == 7340000
        assert parsed_job_stats["state"] == "RUNNING"
        assert parsed_job_stats["nodes"] == ["fc30209"]
        assert parsed_job_stats["partition"] == "cpubase_bycore_b3"


class TestParseTimeFromSLURMScript:
    @staticmethod
    @pytest.mark.parametrize(
        ("slurm_time", "expected_time"),
        [
            ("1-00:00:00", timedelta(days=1)),
            ("1:00:00", timedelta(hours=1)),
            ("30:00", timedelta(minutes=30)),
        ],
    )
    def test_should_parse_time_from_slurm_script(
        tmp_path: Path, slurm_time: str, expected_time: timedelta
    ) -> None:
        text = ["#! /usr/bin/env bash\n", f"#SBATCH --time={slurm_time}\n"]

        slurm_script = Path(tmp_path, "run.sh")
        with slurm_script.open(mode="w", encoding="utf-8") as file:
            file.writelines(text)
        with slurm_script.open(mode="r", encoding="utf-8") as file:
            time = parse_time_from_slurm_script(file)

        assert time == expected_time
