import json
from pathlib import Path

from ase import Atoms
from pydantic import BaseModel
from pydantic import TypeAdapter

from autojob.utils.schemas import PydanticAtoms


class TestAtomsAnnotation:
    @staticmethod
    def test_should_validate_python_round_trip(input_atoms: Atoms) -> None:
        adapter = TypeAdapter(PydanticAtoms)
        assert input_atoms == adapter.validate_python(
            adapter.dump_python(input_atoms)
        )

    @staticmethod
    def test_should_validate_json_round_trip(input_atoms: Atoms) -> None:
        adapter = TypeAdapter(PydanticAtoms)
        assert input_atoms == adapter.validate_json(
            adapter.dump_json(input_atoms)
        )

    @staticmethod
    def test_should_serialize_atoms_to_atoms_object_in_python_mode(
        input_atoms: Atoms,
    ) -> None:
        adapter = TypeAdapter(PydanticAtoms)
        assert isinstance(adapter.dump_python(input_atoms), Atoms)

    @staticmethod
    def test_should_serialize_atoms_to_dictionary_in_json_mode(
        input_atoms: Atoms,
    ) -> None:
        class _Class(BaseModel):
            atoms: PydanticAtoms

        my_class = _Class(atoms=input_atoms)
        dumped_atoms = my_class.model_dump(mode="json")
        assert isinstance(dumped_atoms["atoms"], dict)
        assert isinstance(dumped_atoms["atoms"]["atoms_json"], str)

    @staticmethod
    def test_should_create_serializable_representation_in_json_mode(
        input_atoms: Atoms, tmp_path: Path
    ) -> None:
        class _Class(BaseModel):
            atoms: PydanticAtoms

        my_class = _Class(atoms=input_atoms)
        dumped_atoms = my_class.model_dump(mode="json")
        with Path(tmp_path, "atoms.json").open(
            mode="w", encoding="utf-8"
        ) as file:
            json.dump(dumped_atoms, file)
