# Copyright (C) 2018 The python-bitcointx developers
#
# This file is part of python-bitcointx.
#
# It is subject to the license terms in the LICENSE file found in the top-level
# directory of this distribution.
#
# No part of python-bitcointx, including this file, may be copied, modified,
# propagated, or distributed except according to the terms contained in the
# LICENSE file.
#
# Part of the code is directly ported from C++ code from Bitcoin Core.
# Original C++ code was Copyright (c) 2014-2017 The Bitcoin Core developers
# and licensed under MIT software license.

# pylama:ignore=E501

import unittest
import hashlib
from bitcointx.core import x
from bitcointx.core.sha256 import CSHA256


class Test_CSHA256(unittest.TestCase):
    def test_transform(self):
        hasher = CSHA256()
        in1 = b'\x80'+b'\x00'*63
        in2 = bytes([
            32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
            32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32,
            0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0
        ])
        init = [0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a, 0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19]
        out1 = [0xe3b0c442, 0x98fc1c14, 0x9afbf4c8, 0x996fb924, 0x27ae41e4, 0x649b934c, 0xa495991b, 0x7852b855]
        out2 = [0xce4153b0, 0x147c2a86, 0x3ed4298e, 0xe0676bc8, 0x79fc77a1, 0x2abe1f49, 0xb2b055df, 0x1069523e]

        self.assertEqual(hasher.s, init)

        # Process nothing, and check we remain in the initial state.
        hasher.Transform(b'', 0)
        self.assertEqual(hasher.s, init)

        # Process the padded empty string
        hasher.Transform(in1, 1)
        self.assertEqual(hasher.s, out1)

        hasher.s = init.copy()
        # Process 64 spaces
        hasher.Transform(in2, 2)

        self.assertEqual(hasher.s, out2)

    def test_randomhash(self):

        sample_data = [
            x('13f0f62bb370b1581e0b05e4ca25e482bdd79bd1f46dca986c33ba8a6d587be5c0c4a8f68efe7eaa42e3614c2b1116543899378ba15583c018c901dcc9ad2e96df1ad1d3b4aa141949ca62c7b7d8c0107a132f15fd8026a654f9a39fb567bdf3c5da003f031aef59fabbda5f365349dcce0e59a13ad97eebadc529df9d22d259d3988247c68b0e693d2ef49c3331e6954384221fd3d8747ae99e2392d600cf040211de53c0bb03871231be6522265ef31a7efaf7d683267b225c74a1116c1df058aed85ac23c35540554e2c2c54fdc17be6c5d6cfdc27c60fe601c82f26d248dce748a96207c5a37f31759318758a9a0b2b3bc113b280e4c6e8e99f0c5b718444fa09ff7779c1e375941cf8b8700bbdddf7f3f9bfcddb14bfdd5960eaaa305867acce45a388d602dc19bd0b0d7fea1d22af3f4296d41aa9d1e084fa4afefb4b9c9db49c5ede30e91f22939d0db111e8990d8277fe0edd98f8a42e6b44b1be744b7beef2d0995e27c7644cd6fcfb749aa0152b1e55b14a696caaed1e927b698786f05c8f7fca80a14b9b3c3e0a7e3764ba2b576e5f704c6a709b3586d3b7fdc3ac7b2e5b01fe74d952732b2086c0b6106575b6d3e4ed3ddbec9082cf838f8236d20fdde9eac4ef4bf307ee4b57769c857a5961146'),
            x('cdf1fc6bb786cc43f31653cdbf0aacd9e33c7766efd45582ba7a422e512363ee9974ac9c75e19392cb3c99551ccc5ae77dacfc62d067473b2432ee60d9a3e5341a1162b8c21f2a5b9b6e10ef3dd6374a0624046fc3a0402148aec75544d53c5ae9d42df174f70dbdad545391e9d2848f385099c4abfc8358297cd2f6d9e49ed56c577c9a1280b96e5debbc87cfe0f5b9c7a6ce93db05e40848163bf46dd0767c434e341fb53eb3eb2a792be01de14fab2ca658ae3001f5136c931e6b08d608548f34c838fceed1e804681161c6efa5077e180469468301bd30b577ed4fed8b955b6463766493e99a609f56f461919ddc4b16310bf3f5bff0ff9d766d524bc7a2165566e43066339b5c227a39e340aa98b9060ee667095f549454cd91d22d8cf417e592eb50c9576618e349f45060e54f6142ffabe8538df919da58d3b0c5d30ff8ccdd1d37476ede7a1e2f63ae574f07f0e1b60ab3514b71ea1b68e3fe179ac35066e43296256ebc10e35889004c03a8b33926ecdebadd2c2d7273b900c2471718246deac3adb2bf9afd9f3dc39320ce48564934a834ebacc1d0b45c88df3fc4a47985283a6397d5d7e18c657827f3362431ca6d33b52753c46418728586f3bae8e8ff468b7e2fe1222e993e33de91ef43c35d728df7df362db9cec9ef74c3ffcc49c1a74eb4b27ce2bfcc61916ca208b91b66a241a070930bce540178a804a006c656f4096d9ea226a2cc2df1635d0004bf75dd0c07c67ab9d64fbe95f2e031b98a7f724376d6685dadbe154190efe181e4da84778a6e6cd2bde49859ba5104fad6b79e7168fdae364aeab4ef44cf794afd4d636deef6e8554aed1c83384cb47ea99a4e3d8aefce16bd307c3c594434ea6b6674124610fb00f25afe7fee12ce9fcaf5b5'),
            x('b9788c38b1b3f271ab5460e920c942deee4343feda4355da44ef40ba627e93d4d040ccf4d30c0c4268f018c0656c4ff45c7d45c1a13491df2ed3304de41ecc3462a531da88d4fabb1c2126178ca9a1b93cda6d8c808591050e20a587618e77c19013fe990ca5f43a5f22e8d792ff5c684092a1ff073612260b838210ef2f63190ad5cd062eaac6ac42e18fd44b2e52c8e45e3ab208'),
            x('2f03ca2912f6876499dd3bf22e3ae887e1e218d21e29649f5b8ba98df9f9caf56cee2faaec98063dee39fe9dffa8eac1cbbe34275c1423a2248c3eaf06ea8330e76df63ef781b8bff4ce14bcbc17cedf100774567736aacb63a12c47cfa9e6da0437487b3b64893d4bda427fb71523'),
            x('a92482aa3a65ffb0a79e88d898e4d84bb0198f4daee938b47b26f588c4d029f0f5da0e8f78edc298204e2fa8ace86c7891d024fb4b05a30d80d88accdf49ad50d1440769dd491b677346d616a60b831953d6c4ddf61de737a117cd2fdf2c15766f3c7f087c4838029d0244b0aa484354dd4a944b1bdcd4c04c8a65e6d32d14e48a5f28e85ef5e9951df083f7001bd3b5861412918fdd6898391fdd44e59cd0f0b05841fa0b82e69549066ab53e2e2a46350c215896e0cfbf0d98845be806413fb09a6c6516304b81c39c601ec4b859d90abd3652fd13a8d2db31691e73bcda0ad0d22fc8392e5382899d4853ed46840b7a7a80241529d8ded22eeebd30f7cf6fc63d15f4559faf312caab893be2cad8d75bceadbc5b15b55af32610cf9f16dff56adad78e50f79630537a14599d7912c9262f00819d2e8c4293941a4069be87b969fcfddcdf5c2eb0c142f67721de309260ed70a7b520ad3a2f8fe48559378bb0a5c778c21e8743f5691788c08bb65dca35ea69e6c69510c4a7a3fb0e6454a047fbde7241f692daf343129c21015f28bc0cc78c7cfe7d2795a22215302da0b7dffb31e2bd3be7945cd3923d400e551f7f13c03f77d23702f1742f15bd9398674ded2eec41328bb096caa537c30d972818e449d5b8225bc97399f1d0048762e29bd89a724abd77788a8a38c7a5f0771ada4976446830d583f8fccd5e73138ef654f1fecf1dacac30510cc7a0bd919e3779026016fefbb60980008936d3fab5f359d17762e4129d1fdd76ca89377c3162d0505bf380e6cd4c343554dac401df1fc331ea544b99b90008ddcdebbb7497c1d7d7db61fd1d52e0b336820df93ed01'),
            x('59d2cef970a1a247d9890d79145740247bce2cdb020ac29f65b305f322a4983c1a894d1a213cd4f8fea2010b3e33dc90e8ff60f64e360cbd185f46dec08b26375930be71465d4f1c118c0c71a6c4fa86b69aab7fe973c001bd0fd2aca3ce6f27dc61cbebf66bfcdf181ac795ba416962e8b772f71777d5119d14460ccdae090d3b7e1edd236beac3373b7f0db29dcd3fdde3bbf1dad2b37c801155d729f6bebd46ec32a29c9b211aa9f8501455ca199d5091e73e708bf9b5451eae66e2ce0988a31b81d0ced7ff208b3bdf27930c5f20ea18b3d2899340b1ef36c4d5710ae65bf9d862e59c3a1f10eb3a438947efaf3db6b519db9ca50455fb73c933dec29b46ece393170c27467aa0a58c29c0e4bc30f2787d129bdbc18aab70f3b657d788c990a0660ec7c4acbfc2d77f428722a11cf3091d7d23b663ff2299d207200112b00a4ab0be47cfb55d7824763184101dc700b874fe67053aac855ff2d57d6f9fdf7d74131631d74b2c'),
            x('9c49080c7fb7e347161861295bb84cfb534864d6536c84939a970a11cd7d3c6b75f17c35cde3169d6ef0f755498bf2e29cea6b0611ebd4bc85ec61'),
            x('eaf13618d81010c6268a0e2e9e4aa2085863db4cdf5094faf6502de95219b93f14a0abd21971079f430b0c5d3dee09903ccc0ae050dfd1571bb5a6783afffc8f869b91bd61188d711eee7b27c283f39b8edfbf943b2b46cb0fd27256925d17101e349a128da0b635cc669641592f59e7506bee10084c514be29261de5dc9798590f575e771a6425b494c23093b3a2dfbaad19363796253c6ed52dd2c109c4481c4bbf5fbed5d08730ded2d7f899489114ebd8c7cb7b813f0be9f5510990a93bb2125a03207abb2eeb5397a6252e4804fd93919bfbc27de28e11302078b7e759ae079598e51e0b8f876c7ec0cea34434fa96738d63e1cf334983313af7f563c7a8ad951c14c054c9a915d23e296dbf7b0513699633ea89033b4c80521bab9dc1967b75a20a40488ddfa6eb3056b52e8fd68e332033db05b6bbd725a653d2dfa4121662e86616f2c75a14b5d522a250f3a3ab1be4ea5072a567fb86f5f2332f2ad3d70a20fd38281f5163adb6ef98fbd89bb717f29bb1f213e47ef57238fe72c3f39504a1af196b47f91'),
            x('776570ba516bb931c654eb839c11b02c337aa34c7cca7168d1c544dd8a3b16cd44951754610333a92a02b5a128e1b6085449b2469c95918262b1c9d840aca9b264a6bff1fe4f54855699a3203e0ced688b0cb9863096f4aceb4c3b5b5e5b4a4561cdf299935f35a859084a5dce948c51545e9cc5686b597c6f96d26b050479e3e8f82d0531223234b221ce4f5c8ec22da55016a04088c7c3541da229fb196df81682b1402d81b09acf3b977e6365abcaabb21e361915c2a3dc7a61f0bb352edcd3ff0d122ff377715f8a30d54c4dd3d8602a208612384a91a1af4724'),
            x('91369d9d519faf59044100364efa5d76b8857e64c654eaca232f3820e017944e0b0cf1da6d18ec58ec53bfbd50a7f4b230b11a4041c99401d781489825eaa3d3f57dc10c7abb681321b2fedb9af16db09e2af1c4dd25a6e8d0ae'),
            x('fd31337d1990652533633ff264dfc60ea8d89c3dd426e1dbaca2698410894596444bd73596c2079224985fed9ea0b445e8023f51c801a53173dfb5c6c98641a048bfb375a5924d2ef4272f5696cafb24b302302b53e2713b6723984fd1fb923fd14e78bed97cff296bb39bb15113f1f510'),
            x('227c5b324e25a34ca6b9f3dea158241e084e60ee6dd18117b46d015cdb08e94ab3d93cee855ee16c8fd8cbcc464138c7ed67f94d6a1e8e708165981c8824d6166e0d0dc900eed2d72d897364ab0c85fe15c74b3e441bb7472b5ccaf643fea5caf258c2d23e59ad8ddff67f130f5ff3fda7e9ff4743206001b515d051cc89d6809d34a7b2b49d5c26164e9e2ed5ca0d065ce6624f791866d70ee92209cfa7f47aec8e21f16616ba41a1b768074585a87e588474cf0753eb09ec0a921e46be5044305e5b1fd78b2afaa6c21ef7ec595d397973cfbe87182ba1846da900bb74a8a0ded36d7ead6b7d36da0a693cd42a81ebf511e572ef046514d1a6afdd074f4c2e9f6bd6f6478c7498f742d8c89f2f4658c3cab1e5d702e9b3fc1b1a04034605e9ab3fc2ec5592f8d974d9cba048fae1aaa0924bf050e9fb8ec2517dbaf1374448716b516937a31a7e0a719ceeb652e22ed8d0cdd9cade39fbf78672e6ad115f74abec3ef56290efdfa172d8e9d39e16dce1a10870f13683372b6d078954edcf6f875df6ed24a6e01af83a48f426edfc2550ef6976eb9b60d210bfee16f5d4f42dcd98fc582c5cf59a8268b10c90213a431623e22ef6787e0d63aee8d26299fff70421eac311e226fb648b0d2c17a978f020d807f91793f538b3ef44d0ebb71a6d2c2596e8614d2ac6eb743e397be4bcb48f3d8856a228d2753750e517fd6dc2ff711348ac169d050bb47c722cbc548c9d7f7feb8bf86dc7d938a38bcc77cc59170ca56ff4cd6ad4c345754f9ac6a301d2f33f2669dd417a7badb49f006e16ecfa2d64d4d36000eadddbedaa642612f6525990db158cfa7670619bff21a68f40029b4c31962a941879f839959a3ea51c1bf755652759494f7dd98423f55e4268758e123399f4aa774dbdf487891dbee03520cc066e833defb2148fcfee6b53ef5de73de7d0452f1886875b22ee1c3db0401a16b588cf4a0a717294a38312fd3643d6910aea7ac5d0b5a664d1e5e4e9dc55e6e6c4675b616d2246ede016cc2ff3b66dd368642de857df2489f24f5f274e9c5f2e280392df5398874c1b8ab82d04d3c540d4ff834765ccba9245bff8d186d10ad1f5a28916a993ad58cc2ae2d8678fb5ce5933bf6bac9d77ac019f43a95b'),
            x('5d8ede7a09a91849e208cd240b0503bf4152708fb2abaffa238bc988d6e0f558b524b34727697113c2452b2d9272e48a1fe8b6ac5e2a9db3dba8407654ac784a51d1c60e4790a27345c4a569491215d5d8c882914e41eea04b6b656da4b38a95a719854464e6bb2f981d389194d119bfcfda5ffa5eb7cc89f7a0ec71b1f9afb15a311b7d7d207cfe455a1a1a9a309b74832d720d94f2c179d5a0811f0a97b7a5b29c59a07c7fe5f48a895b1a69e0b217301f0f74fc85184e85a5e7cc131737c5af71dc01b1a9bf4a6b18f6f3b010f79017cb97b78a6c45522b11fc88ebef5bb4cce7518ac4fd7b92eb6f9d7ade642b4bd44379cc7ce69479db7847b8f316681830970e8654649aa21442d344a4bb6ff144420041465129fd58a6bad3fc9cdd95f2744ecfdf1b07bfa5a75b84a9e4bbcfd545c062b50325c829c496e952005c0833ea11a2e548d66fb6e55785c83b3598d3c7238325f309e3a66a31727357349b782df5caef215f9f23a6a46245294661f67993fa425389dcb0af261da98db84c06dc211df3257a721e77b298ed3a83446d0c113971212894a8cc66a35730a08e3511d7118db671c6a75801ddf78939fa5e34e2407fbd08e08fb9256b67a1a2c748fc888befa36cac3ceed15f9ce44afbe1d7ee59b1aec6af3a5f2b1252205db55e53f8109333cf354086d1b908516a1e8d0d3a6906247d013044dc33f06f7a95387475bcb0d2dd50dc9fcf0bcb8c0378e7a47e2bd1c87d87e00edfcfad87ae681e4304b29e2e17a90413419bf8e5b33e39f7a81647d1ca0a3587ae2e20f173d9ea88abe6380a73585da61243bc8520624105735fc7e13cc18019169a28989f5d2ba2b574612f15259d42c7b58c8a8f519f7ec90e6be9519f86a26262e337cdb8288c9aba642c18c654ce77c009192bf50d7b68b9f766d8f382b1e552285345834cd07349b92b92a45a97362f54c363fb'),
            x('fb6cbb46ec02a06420f84aeaff680e6fa672e4545b3350c0879be706491141b8d1541e3ab6fc87532a0ac707fafba1178c98a4f3a145a56aa42481f060fb53e581d109f8b8ca2450a4cde1bf92d6a7aed200588ae0d512c9e5bf0b28aeabadd25a8ddade6c1b399ae360941fe6b1b86155092a7ddb59a3561b86672dc575787afd8a195e08139f4726320ca60343849cf4e795177cc8d465d83379849754440c69cc59486e8c42e6c88ffe497167ad8ea5b4b4bb2cf24adf274180190ad873b9767399afe544006cf399695022fadcde0b43692b4dad7b85dce46349186d5df78af213e33e9fc8f111b656bbf280c7f0c581f4d8d52e6e6099f5bc72dbb408b475d2993161cc0ac7be631d51a342cf946d629c619c36904e7e5acdf7284e6479d8b030b91185252c75edd1177e117e71106b8592fdadf4b8f55e79dce40b49771aa4c34eb55b16dbc432737aec05ac7edf1606e4c4087c0813900e7b440375dd777d3b40d88f6874b8fa8986056bb36fd3f092b1c5336ec7cb479e5e21592aa46eb405c0828c086d1217aa281de7a868eebfe3bb0f5300310e07cc0814bef918e771a15ae90e6ee4492027edaf7c6a84782a9e998db1313d9ce7190372d8c1f7c1c8637339a575f02905d62e151fcf4b5335f81f2043762db8fc57f9151640dc514ba1160cab27703eb7237f64c60b604f7e42f617d702851998124134cac51d82c67e25c6c016b8f1127e885a58c51d49b98daa953740413df0646e36b86b3ed76a1e1c8a146a656d9bb28500d67a7fec0086290f7da0'),
            x('cecf0daf1604b74afb1ed75daeead9205a37af758e87687199affdbaab74c55183cfe44c56ade1552ca09499c8b994fc51e8afed45a0457f8bc440248eff37795beac497e03275cb08ee8b89b102c7c6e8537956f314af96e0d2b1e82ec96414f6e84212e3f9cd666e7cce97fa9af74e4d970c90d8144430fd4226d6eae02ec97fda0f3956d7aa3bdd5f62588e'),
            x('1508277e56802bad29d7387cdc1d29c3d0bd677249f07fcc88f807294378df481dfa27fd1452359fde9b3cfbeee7b3742549c53666b3d0c0deb4c9981b5f1bd0a7d3c43c829d24eb9b2391fa314b325e9e33a01bf80f837b55a163fb98efaa74cd4c45aa0f29b43ea3b4072233c6987b0c5d5aaaff529ea276b4573ded2c45f2f53473f26301b925ed70b62a04dcf56cf5836e8b0dc13e4c52e78e3a1a184e86364865b25d8322b2c140e5ed26dc839c4c82c23adbe150545fa14cd2bf'),
        ]
        for data in sample_data:
            hash1 = hashlib.sha256(data).digest()
            hash2 = CSHA256().Write(data).Finalize()
            self.assertEqual(hash1, hash2)
