"""Utility functions for the CLI.

If this grows into groups of utilities, move to a cli/utils/ dir, with more specific
filenames.
"""

import os
import sys
from pathlib import Path
import subprocess
import shlex
import shutil

import click

from py_bugger.cli import cli_messages
from py_bugger.cli.config import pb_config


def validate_config():
    """Make sure the CLI options are valid."""
    if not pb_config.exception_type:
        click.echo(cli_messages.msg_exception_type_required)
        sys.exit()

    if pb_config.target_dir and pb_config.target_file:
        click.echo(cli_messages.msg_target_file_dir)
        sys.exit()

    if pb_config.target_dir:
        _validate_target_dir()

    if pb_config.target_file:
        _validate_target_file()

    # Update all options before running Git status checks. Info like target_dir
    # is used for those checks.
    _update_options()

    _validate_git_status()


# --- Helper functions ___


def _update_options():
    """Make sure options are ready to use."""
    # Set an appropriate target directory.
    if pb_config.target_dir:
        pb_config.target_dir = Path(pb_config.target_dir)
    else:
        pb_config.target_dir = Path(os.getcwd())

    # Make sure target_file is a Path.
    if pb_config.target_file:
        pb_config.target_file = Path(pb_config.target_file)


def _validate_target_dir():
    """Make sure a valid directory was passed.

    Check for common mistakes, then verify it is a dir.
    """
    path_target_dir = Path(pb_config.target_dir)
    if path_target_dir.is_file():
        msg = cli_messages.msg_file_not_dir(path_target_dir)
        click.echo(msg)
        sys.exit()
    elif not path_target_dir.exists():
        msg = cli_messages.msg_nonexistent_dir(path_target_dir)
        click.echo(msg)
        sys.exit()
    elif not path_target_dir.is_dir():
        msg = cli_messages.msg_not_dir(path_target_dir)
        click.echo(msg)
        sys.exit()


def _validate_target_file():
    """Make sure an appropriate file was passed.

    Check for common mistakes, then verify it is a file.
    """
    path_target_file = Path(pb_config.target_file)
    if path_target_file.is_dir():
        msg = cli_messages.msg_dir_not_file(path_target_file)
        click.echo(msg)
        sys.exit()
    elif not path_target_file.exists():
        msg = cli_messages.msg_nonexistent_file(path_target_file)
        click.echo(msg)
        sys.exit()
    elif not path_target_file.is_file():
        msg = cli_messages.msg_not_file(path_target_file)
        click.echo(msg)
        sys.exit()
    elif path_target_file.suffix != ".py":
        msg = cli_messages.msg_file_not_py(path_target_file)
        click.echo(msg)
        sys.exit()


def _validate_git_status():
    """Look for a clean Git status before introducing bugs."""
    if pb_config.ignore_git_status:
        return

    _check_git_available()
    _check_git_status()


def _check_git_available():
    """Quit with appropriate message if Git not available."""
    if not shutil.which("git"):
        click.echo(cli_messages.msg_git_not_available)
        sys.exit()


def _check_git_status():
    """Make sure we're starting with a clean git status."""
    if pb_config.target_file:
        git_dir = pb_config.target_file.parent
    else:
        git_dir = pb_config.target_dir

    cmd = "git status --porcelain"
    cmd_parts = shlex.split(cmd)
    output = subprocess.run(cmd_parts, cwd=git_dir, capture_output=True, text=True)

    if "fatal: not a git repository" in output.stderr:
        msg = cli_messages.msg_git_not_used(pb_config)
        click.echo(msg)
        sys.exit()

    # `git status --porcelain` has no output when the status is clean.
    if output.stdout or output.stderr:
        msg = cli_messages.msg_unclean_git_status
        click.echo(msg)
        sys.exit()
