"""Contains the class representing tasks."""

from collections.abc import Generator
from itertools import chain
from pathlib import Path


class Task:  # pylint: disable=too-few-public-methods
    """
    Represents a SLURM batch task.

    Attributes
    ----------
    task_folder : Path
        Path to the folder containing the task files.
    output_folder : Path
        Path to the folder containing the rendered tasks.
    extra_input_files : list[Path]
        List of additional input files on which the task depends.
    output_files : list[Path]
        List of output files generated by the task.
    """

    def __init__(  # pylint: disable=too-many-arguments,too-many-positional-arguments
        self,
        name: str,
        input_folder: Path,
        output_folder: Path,
        output_files: list[Path],
        extra_input_files: list[Path] | None = None,
    ) -> None:
        """
        Initialize the task.

        Takes the additional input files beyond those in the task directory. Paths are
        relative to the input folder.

        Parameters
        ----------
        name : str
            Name of the task.
        input_folder : Path
            Path to the folder containing the task files.
        output_folder : Path
            Path to the folder containing the rendered task.
        output_files : list[Path]
            List of output files generated by the task.
        extra_input_files : list[Path]
            List of extra input files on which the task depends.
        """
        self.task_folder = input_folder / name
        self.extra_input_files = (
            extra_input_files if extra_input_files is not None else []
        )
        self.output_files = output_files
        self.output_folder = output_folder

    def _glob_non_empty(self, list_of_paths: list[Path], base: Path) -> bool:
        return all(list(Path().glob(str(base / glob))) for glob in list_of_paths)

    def _glob_list(
        self, list_of_paths: list[Path], base: Path
    ) -> Generator[Path, None, None]:
        return (
            path for glob in list_of_paths for path in Path().glob(str(base / glob))
        )

    def _transform_input_output(self, path: Path) -> Path:
        if path.suffix == '.j2':
            path = path.with_suffix('')
        return self.output_folder / path.relative_to(self.task_folder.parent)

    def run(self) -> bool:
        """
        Return True if the task needs to be run.

        Task needs to be run if the input files have changed since the last run or
        if the output files do not exist.

        Returns
        -------
        bool
            True if the task needs to be run, False otherwise.
        """
        if not self._glob_non_empty(self.output_files, self.output_folder):
            return True
        most_recent_input_change = max(
            (
                input_file.stat().st_mtime
                for input_file in chain(
                    map(self._transform_input_output, self.task_folder.rglob('*')),
                    self._glob_list(self.extra_input_files, self.task_folder.parent),
                )
            ),
            default=0,
        )
        most_recent_output_change = max(
            (
                output_file.stat().st_mtime
                for output_file in self._glob_list(
                    self.output_files, self.output_folder
                )
            ),
            default=0,
        )
        return most_recent_input_change >= most_recent_output_change
