"""
Comprehensive test suite for the auto_filename_generation module to improve test coverage.
"""

import pytest
import tempfile
import os
from unittest.mock import Mock, patch, MagicMock
from pathlib import Path
from datetime import datetime, date
from python_cimis.auto_filename_generation import (
    AutoFilenameGenerator,
    generate_weather_filename,
    generate_stations_filename,
    generate_zip_codes_filename,
    generate_custom_filename
)
from python_cimis.models import WeatherData, WeatherProvider, Station


class TestAutoFilenameGeneratorBasics:
    """Test basic AutoFilenameGenerator functionality."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_init_default(self):
        """Test generator initialization with default base directory."""
        generator = AutoFilenameGenerator()
        assert generator.base_directory == Path.cwd()
    
    def test_init_with_directory(self):
        """Test generator initialization with custom base directory."""
        test_dir = Path("/tmp/test")
        generator = AutoFilenameGenerator(test_dir)
        assert generator.base_directory == test_dir
    
    def test_set_base_directory(self):
        """Test setting base directory."""
        test_dir = Path("/tmp/test")
        self.generator.set_base_directory(test_dir)
        assert self.generator.base_directory == test_dir
    
    def test_set_base_directory_string(self):
        """Test setting base directory with string path."""
        test_dir = "/tmp/test"
        self.generator.set_base_directory(test_dir)
        assert self.generator.base_directory == Path(test_dir)


class TestWeatherDataFilenameGeneration:
    """Test weather data filename generation."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_weather_data_single_station(self):
        """Test weather data filename generation for single station."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 2, "Date": "2023-01-01"},
            {"Station": 2, "Date": "2023-01-02"}
        ]
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
        assert '2' in filename  # Station number
        assert '20230101' in filename  # Date
    
    def test_generate_for_weather_data_multiple_stations(self):
        """Test weather data filename generation for multiple stations."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 2, "Date": "2023-01-01"},
            {"Station": 6, "Date": "2023-01-01"},
            {"Station": 127, "Date": "2023-01-02"}
        ]
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_for_weather_data_date_range(self):
        """Test weather data filename generation with date range."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 2, "Date": "2023-01-01"},
            {"Station": 2, "Date": "2023-01-05"}
        ]
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert '20230101_to_20230105' in filename
    
    def test_generate_for_weather_data_empty(self):
        """Test weather data filename generation with empty data."""
        weather_data = WeatherData()
        weather_data.providers = []
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'unknown' in filename


class TestStationFilenameGeneration:
    """Test station filename generation."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_single_station(self):
        """Test filename generation for single station."""
        station = Station(station_nbr=2, name="Five Points")
        
        filename = self.generator.generate_for_stations([station])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_station_' in filename
        assert 'five_points' in filename
    
    def test_generate_for_multiple_stations(self):
        """Test filename generation for multiple stations."""
        stations = [
            Station(station_nbr=2, name="Five Points"),
            Station(station_nbr=6, name="Davis"),
            Station(station_nbr=127, name="Temecula")
        ]
        
        filename = self.generator.generate_for_stations(stations)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_stations_' in filename
    
    def test_generate_for_many_stations(self):
        """Test filename generation for many stations."""
        stations = []
        for i in range(10):
            stations.append(Station(station_nbr=i, name=f"Station {i}"))
        
        filename = self.generator.generate_for_stations(stations)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '10_stations' in filename
    
    def test_generate_for_empty_stations(self):
        """Test filename generation for empty station list."""
        filename = self.generator.generate_for_stations([])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_stations_' in filename
    
    def test_generate_for_stations_no_name(self):
        """Test filename generation for stations without names."""
        station = Station(station_nbr=2, name=None)
        
        filename = self.generator.generate_for_stations([station])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '2' in filename  # Should use station number


class TestZipCodeFilenameGeneration:
    """Test zip code filename generation."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_single_zip_code(self):
        """Test filename generation for single zip code."""
        filename = self.generator.generate_for_zip_codes(["93624"])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_93624' in filename
    
    def test_generate_for_multiple_zip_codes(self):
        """Test filename generation for multiple zip codes."""
        zip_codes = ["93624", "94503", "95667"]
        
        filename = self.generator.generate_for_zip_codes(zip_codes)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_zipcodes_' in filename
        assert '93624_94503_95667' in filename
    
    def test_generate_for_many_zip_codes(self):
        """Test filename generation for many zip codes."""
        zip_codes = [f"9{i:04d}" for i in range(10)]
        
        filename = self.generator.generate_for_zip_codes(zip_codes)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '10_codes' in filename
    
    def test_generate_for_empty_zip_codes(self):
        """Test filename generation for empty zip code list."""
        filename = self.generator.generate_for_zip_codes([])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_zipcodes_' in filename


class TestCustomFilenameGeneration:
    """Test custom filename generation."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_custom_basic(self):
        """Test basic custom filename generation."""
        filename = self.generator.generate_custom("weather")
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_weather_' in filename
    
    def test_generate_custom_with_identifiers(self):
        """Test custom filename generation with identifiers."""
        filename = self.generator.generate_custom("hourly", ["station1", "station2"])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_hourly_' in filename
        assert 'station1_station2' in filename
    
    def test_generate_custom_with_date_range(self):
        """Test custom filename generation with date range."""
        filename = self.generator.generate_custom("daily", date_range="20230101_to_20230105")
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '20230101_to_20230105' in filename
    
    def test_generate_custom_many_identifiers(self):
        """Test custom filename generation with many identifiers."""
        identifiers = [f"id{i}" for i in range(10)]
        
        filename = self.generator.generate_custom("test", identifiers)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '10_items' in filename


class TestPrivateHelperMethods:
    """Test private helper methods."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_format_station_names_single(self):
        """Test station names formatting for single station."""
        result = self.generator._format_station_names(["Five Points"])
        assert result == "five_points"
    
    def test_format_station_names_multiple(self):
        """Test station names formatting for multiple stations."""
        result = self.generator._format_station_names(["Five Points", "Davis", "Temecula"])
        assert result == "five_points_davis_temecula"
    
    def test_format_station_names_many(self):
        """Test station names formatting for many stations."""
        stations = [f"Station {i}" for i in range(10)]
        result = self.generator._format_station_names(stations)
        assert 'plus' in result and 'more' in result
    
    def test_format_station_names_empty(self):
        """Test station names formatting for empty list."""
        result = self.generator._format_station_names([])
        assert result == "unknown"
    
    def test_format_date_range_single(self):
        """Test date range formatting for single date."""
        result = self.generator._format_date_range(["2023-01-01"])
        assert result == "20230101"
    
    def test_format_date_range_multiple(self):
        """Test date range formatting for multiple dates."""
        result = self.generator._format_date_range(["2023-01-01", "2023-01-05"])
        assert result == "20230101_to_20230105"
    
    def test_format_date_range_empty(self):
        """Test date range formatting for empty list."""
        result = self.generator._format_date_range([])
        assert len(result) == 8  # YYYYMMDD format
    
    def test_format_date_range_invalid(self):
        """Test date range formatting with invalid dates."""
        result = self.generator._format_date_range(["invalid-date"])
        assert len(result) == 8  # Should fallback to current date
    
    def test_sanitize_name_normal(self):
        """Test name sanitization for normal names."""
        result = self.generator._sanitize_name("Five Points Weather Station")
        assert result == "five_points_weather_station"
    
    def test_sanitize_name_special_chars(self):
        """Test name sanitization with special characters."""
        result = self.generator._sanitize_name("Test/Station:Name|With*Special?Chars")
        assert "/" not in result
        assert ":" not in result
        assert "|" not in result
        assert "*" not in result
        assert "?" not in result
    
    def test_sanitize_name_empty(self):
        """Test name sanitization with empty string."""
        result = self.generator._sanitize_name("")
        assert result == "unknown"
    
    def test_sanitize_name_none(self):
        """Test name sanitization with None."""
        result = self.generator._sanitize_name(None)
        assert result == "unknown"
    
    def test_sanitize_name_long(self):
        """Test name sanitization with very long name."""
        long_name = "A" * 100
        result = self.generator._sanitize_name(long_name)
        assert len(result) <= 30
    
    def test_sanitize_filename_normal(self):
        """Test filename sanitization for normal filename."""
        result = self.generator._sanitize_filename("test_file.csv")
        assert result == "test_file.csv"
    
    def test_sanitize_filename_special_chars(self):
        """Test filename sanitization with special characters."""
        result = self.generator._sanitize_filename("test<file>name:with|special*chars?.txt")
        assert "<" not in result
        assert ">" not in result
        assert ":" not in result
        assert "|" not in result
        assert "*" not in result
        assert "?" not in result
        assert result.endswith(".csv")  # Should force .csv extension
    
    def test_sanitize_filename_no_extension(self):
        """Test filename sanitization without extension."""
        result = self.generator._sanitize_filename("test_file")
        assert result.endswith(".csv")
    
    def test_sanitize_filename_empty(self):
        """Test filename sanitization with empty string."""
        result = self.generator._sanitize_filename("")
        assert result.startswith("cimis_data_")
        assert result.endswith(".csv")
    
    def test_sanitize_filename_long(self):
        """Test filename sanitization with very long filename."""
        long_name = "A" * 300 + ".csv"
        result = self.generator._sanitize_filename(long_name)
        assert len(result) <= 204  # 200 + ".csv"


class TestModuleFunctions:
    """Test module-level convenience functions."""
    
    def test_generate_weather_filename(self):
        """Test generate_weather_filename function."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [{"Station": 2, "Date": "2023-01-01"}]
        weather_data.providers = [provider]
        
        filename = generate_weather_filename(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_weather_filename_with_base_dir(self):
        """Test generate_weather_filename with base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            weather_data = WeatherData()
            provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
            provider.records = [{"Station": 2, "Date": "2023-01-01"}]
            weather_data.providers = [provider]
            
            filename = generate_weather_filename(weather_data, temp_dir)
            
            assert temp_dir in filename
            assert filename.endswith('.csv')
    
    def test_generate_stations_filename(self):
        """Test generate_stations_filename function."""
        stations = [Station(station_nbr=2, name="Five Points")]
        
        filename = generate_stations_filename(stations)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_station_' in filename
    
    def test_generate_zip_codes_filename(self):
        """Test generate_zip_codes_filename function."""
        zip_codes = ["93624", "94503"]
        
        filename = generate_zip_codes_filename(zip_codes)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_zipcodes_' in filename
    
    def test_generate_custom_filename(self):
        """Test generate_custom_filename function."""
        filename = generate_custom_filename("test", ["id1", "id2"])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis_test_' in filename


class TestTimestampGeneration:
    """Test timestamp generation in filenames."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    @patch('python_cimis.auto_filename_generation.datetime')
    def test_timestamp_generation(self, mock_datetime):
        """Test timestamp generation."""
        # Mock datetime to return a fixed time
        mock_datetime.now.return_value.strftime.return_value = "20230101_120000"
        
        filename = self.generator.generate_custom("test", ["id1"])
        
        assert "20230101_120000" in filename
        mock_datetime.now.assert_called()
    
    def test_timestamp_format(self):
        """Test timestamp format in generated filenames."""
        filename = self.generator.generate_custom("test")
        
        # Extract timestamp (should be at the end before .csv)
        timestamp_part = filename.split('_')[-1].replace('.csv', '')
        
        # Should be in format YYYYMMDD_HHMMSS
        assert len(timestamp_part) == 15  # YYYYMMDD_HHMMSS
        assert timestamp_part[8] == '_'  # Separator


class TestEdgeCasesAndErrorHandling:
    """Test edge cases and error handling."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_weather_data_no_providers(self):
        """Test weather data with no providers."""
        weather_data = WeatherData()
        weather_data.providers = None
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
    
    def test_weather_data_providers_no_records(self):
        """Test weather data with providers but no records."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = []
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
    
    def test_station_with_invalid_data(self):
        """Test station filename generation with invalid data."""
        station = Station()  # Empty station
        
        filename = self.generator.generate_for_stations([station])
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
    
    def test_zip_codes_with_none(self):
        """Test zip code filename generation with None."""
        filename = self.generator.generate_for_zip_codes(None)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
    
    def test_custom_with_none_values(self):
        """Test custom filename generation with None values."""
        filename = self.generator.generate_custom(None)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
    
    def test_directory_creation(self):
        """Test automatic directory creation."""
        with tempfile.TemporaryDirectory() as temp_dir:
            test_path = os.path.join(temp_dir, "subdir", "subsubdir")
            
            generator = AutoFilenameGenerator()
            generator.set_base_directory(test_path)
            
            # Directory should be created automatically
            assert os.path.exists(test_path)
    
    def test_unicode_in_names(self):
        """Test handling of unicode characters in names."""
        unicode_name = "Statiön with Spëcial Chårs"
        result = self.generator._sanitize_name(unicode_name)
        
        assert isinstance(result, str)
        # Should handle unicode gracefully
    
    def test_very_long_paths(self):
        """Test handling of very long file paths."""
        long_identifiers = [f"very_long_identifier_number_{i:03d}" for i in range(20)]
        
        filename = self.generator.generate_custom("test", long_identifiers)
        
        # Should not create overly long filenames
        base_name = os.path.basename(filename)
        assert len(base_name) < 255  # Typical filesystem limit
    
    def test_invalid_characters_in_base_directory(self):
        """Test handling of invalid characters in base directory."""
        # This should not crash the generator
        try:
            generator = AutoFilenameGenerator("test<dir>with:invalid|chars")
            filename = generator.generate_custom("test")
            assert isinstance(filename, str)
        except Exception:
            # If the OS doesn't allow the directory, that's acceptable
            pass


class TestPerformanceAndMemory:
    """Test performance and memory efficiency."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_large_number_of_stations(self):
        """Test filename generation with large number of stations."""
        stations = []
        for i in range(1000):
            stations.append(Station(station_nbr=i, name=f"Station {i}"))
        
        filename = self.generator.generate_for_stations(stations)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '1000_stations' in filename
    
    def test_large_number_of_zip_codes(self):
        """Test filename generation with large number of zip codes."""
        zip_codes = [f"{i:05d}" for i in range(1000)]
        
        filename = self.generator.generate_for_zip_codes(zip_codes)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert '1000_codes' in filename
    
    def test_repeated_generation(self):
        """Test repeated filename generation for memory leaks."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [{"Station": 2, "Date": "2023-01-01"}]
        weather_data.providers = [provider]
        
        # Generate many filenames
        for _ in range(100):
            filename = self.generator.generate_for_weather_data(weather_data)
            assert isinstance(filename, str)
    
    def test_complex_data_structures(self):
        """Test with complex data structures."""
        weather_data = WeatherData()
        
        # Multiple providers with many records
        for provider_idx in range(5):
            provider = WeatherProvider(name=f"Provider {provider_idx}", type="station", owner="CA DWR")
            provider.records = []
            for record_idx in range(100):
                provider.records.append({
                    "Station": record_idx % 10,
                    "Date": f"2023-01-{(record_idx % 30) + 1:02d}",
                    "Value": record_idx * 0.1
                })
            weather_data.providers.append(provider)
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
