"""
Main client class for the Python CIMIS library.
"""

import csv
import json
import requests
from datetime import datetime, date
from pathlib import Path
from typing import Dict, List, Optional, Union, Any
from urllib.parse import urlencode

from .exceptions import (
    CimisAPIError, 
    CimisDataError, 
    CimisConnectionError, 
    CimisAuthenticationError
)
from .models import (
    WeatherData, 
    WeatherProvider, 
    WeatherRecord, 
    DataValue,
    Station, 
    ZipCode, 
    SpatialZipCode
)
from .endpoints import CimisEndpoints
from .utils import FilenameGenerator


class CimisClient:
    """
    Main client for accessing the California Irrigation Management Information System (CIMIS) API.
    
    This client provides methods to:
    - Fetch weather data by station, zip code, coordinates, or address
    - Retrieve station information
    - Get zip code information
    - Export data to CSV format with all available columns
    - Auto-generate filenames based on station names and dates
    """
    
    def __init__(self, app_key: str, timeout: int = 30):
        """
        Initialize the CIMIS client.
        
        Args:
            app_key: Your CIMIS API application key
            timeout: Request timeout in seconds (default: 30)
        """
        self.app_key = app_key
        self.timeout = timeout
        self.session = requests.Session()
        self.session.headers.update({
            'User-Agent': 'python-CIMIS/1.0.0',
            'Accept': 'application/json'
        })
        
        # Use centralized endpoints
        self.endpoints = CimisEndpoints()
        
        # Use filename generator for automatic CSV file naming
        self.filename_generator = FilenameGenerator()
    
    def _make_request(self, endpoint_key: str, params: Dict[str, Any], **endpoint_kwargs) -> Dict[str, Any]:
        """
        Make a request to the CIMIS API using centralized endpoint management.
        
        Args:
            endpoint_key: Key for the endpoint in CimisEndpoints
            params: Query parameters
            **endpoint_kwargs: Parameters for endpoint URL formatting
            
        Returns:
            Parsed JSON response
            
        Raises:
            CimisConnectionError: For connection issues
            CimisAuthenticationError: For authentication issues
            CimisAPIError: For API errors
        """
        # Add app key to parameters
        params['appKey'] = self.app_key
        
        # Get URL from centralized endpoints
        url = self.endpoints.get_url(endpoint_key, **endpoint_kwargs)
        
        try:
            response = self.session.get(url, params=params, timeout=self.timeout)
        except requests.exceptions.Timeout:
            raise CimisConnectionError("Request timeout")
        except requests.exceptions.ConnectionError as e:
            raise CimisConnectionError(f"Connection error: {e}")
        except requests.exceptions.RequestException as e:
            raise CimisConnectionError(f"Request error: {e}")
        
        # Handle HTTP errors
        if response.status_code == 403:
            raise CimisAuthenticationError("Invalid API key", "ERR1006", 403)
        elif response.status_code == 404:
            try:
                error_data = response.json()
                error_msg = error_data.get('Message', 'Resource not found')
                raise CimisAPIError(error_msg, http_code=404)
            except ValueError:
                raise CimisAPIError("Resource not found", http_code=404)
        elif response.status_code != 200:
            raise CimisAPIError(f"HTTP {response.status_code}: {response.reason}", 
                              http_code=response.status_code)
        
        try:
            return response.json()
        except ValueError as e:
            raise CimisDataError(f"Invalid JSON response: {e}")
    
    def _parse_data_response(self, data: Dict[str, Any]) -> WeatherData:
        """Parse weather data response into WeatherData object."""
        weather_data = WeatherData()
        
        if 'Data' not in data or 'Providers' not in data['Data']:
            return weather_data
        
        for provider_data in data['Data']['Providers']:
            provider = WeatherProvider(
                name=provider_data.get('Name', ''),
                type=provider_data.get('Type', ''),
                owner=provider_data.get('Owner', '')
            )
            
            for record_data in provider_data.get('Records', []):
                record = WeatherRecord(
                    date=record_data.get('Date', ''),
                    julian=record_data.get('Julian', ''),
                    station=record_data.get('Station'),
                    standard=record_data.get('Standard', 'english'),
                    zip_codes=record_data.get('ZipCodes', ''),
                    scope=record_data.get('Scope', 'daily'),
                    hour=record_data.get('Hour')
                )
                
                # Parse data values
                for key, value in record_data.items():
                    if isinstance(value, dict) and 'Value' in value:
                        data_value = DataValue(
                            value=value.get('Value'),
                            qc=value.get('Qc', ' '),
                            unit=value.get('Unit', '')
                        )
                        record.data_values[key] = data_value
                
                provider.records.append(record)
            
            weather_data.providers.append(provider)
        
        return weather_data
    
    def _parse_stations_response(self, data: Dict[str, Any]) -> List[Station]:
        """Parse stations response into list of Station objects."""
        stations = []
        
        for station_data in data.get('Stations', []):
            station = Station(
                station_nbr=station_data.get('StationNbr', ''),
                name=station_data.get('Name', ''),
                city=station_data.get('City', ''),
                regional_office=station_data.get('RegionalOffice'),
                county=station_data.get('County'),
                connect_date=station_data.get('ConnectDate', ''),
                disconnect_date=station_data.get('DisconnectDate', ''),
                is_active=station_data.get('IsActive', 'True').lower() == 'true',
                is_eto_station=station_data.get('IsEtoStation', 'True').lower() == 'true',
                elevation=station_data.get('Elevation', ''),
                ground_cover=station_data.get('GroundCover', ''),
                hms_latitude=station_data.get('HmsLatitude', ''),
                hms_longitude=station_data.get('HmsLongitude', ''),
                zip_codes=station_data.get('ZipCodes', []),
                siting_desc=station_data.get('SitingDesc', '')
            )
            stations.append(station)
        
        return stations
    
    def get_data(self, 
                 targets: Union[str, List[str]], 
                 start_date: Union[str, date, datetime],
                 end_date: Union[str, date, datetime],
                 data_items: Optional[List[str]] = None,
                 unit_of_measure: str = 'E',
                 prioritize_scs: bool = True) -> WeatherData:
        """
        Get weather data from CIMIS.
        
        Args:
            targets: Station numbers, zip codes, coordinates, or addresses
            start_date: Start date (YYYY-MM-DD format, date, or datetime)
            end_date: End date (YYYY-MM-DD format, date, or datetime)
            data_items: List of data items to retrieve (uses default if None)
            unit_of_measure: 'E' for English or 'M' for Metric
            prioritize_scs: Whether to prioritize SCS data for zip codes
            
        Returns:
            WeatherData object containing the response
        """
        # Use data_items if provided, otherwise use all available items
        if data_items is None:
            data_items = []  # Empty list will get all available data items
            
        params = self.endpoints.prepare_data_params(
            targets=targets,
            start_date=start_date,
            end_date=end_date,
            items=data_items,
            measure_unit=unit_of_measure,
            prioritize_sri=(unit_of_measure == 'M'),  # Use SRI for metric
            prioritize_scs=prioritize_scs
        )
        
        response_data = self._make_request('data', params)
        return self.endpoints.parse_data_response(response_data)
    
    def get_daily_data(self, 
                       targets: Union[str, List[str]], 
                       start_date: Union[str, date, datetime],
                       end_date: Union[str, date, datetime],
                       data_items: Optional[List[str]] = None,
                       unit_of_measure: str = 'E',
                       prioritize_scs: bool = True,
                       csv: bool = False,
                       filename: Optional[Union[str, Path]] = None) -> Union[WeatherData, tuple[WeatherData, str]]:
        """
        Get daily weather data from CIMIS.
        
        This is a convenience method that uses default daily data items.
        
        Args:
            targets: Station numbers, zip codes, coordinates, or addresses
            start_date: Start date for data retrieval
            end_date: End date for data retrieval
            data_items: List of data items to retrieve (uses default if None)
            unit_of_measure: 'E' for English or 'M' for Metric
            prioritize_scs: Whether to prioritize SCS data for zip codes
            csv: If True, automatically export to CSV with auto-generated filename
            filename: Custom filename for CSV export (only used if csv=True)
            
        Returns:
            WeatherData object if csv=False, or tuple of (WeatherData, csv_filename) if csv=True
        """
        if data_items is None:
            data_items = []  # Empty list will get all available data items
        
        weather_data = self.get_data(targets, start_date, end_date, data_items, 
                                   unit_of_measure, prioritize_scs)
        
        if csv:
            csv_filename = self.export_to_csv(weather_data, filename)
            return weather_data, csv_filename
        
        return weather_data
    
    def get_hourly_data(self, 
                        targets: Union[str, List[str]], 
                        start_date: Union[str, date, datetime],
                        end_date: Union[str, date, datetime],
                        data_items: Optional[List[str]] = None,
                        unit_of_measure: str = 'E',
                        csv: bool = False,
                        filename: Optional[Union[str, Path]] = None) -> Union[WeatherData, tuple[WeatherData, str]]:
        """
        Get hourly weather data from CIMIS.
        
        Note: Hourly data is only available from WSN stations, not SCS.
        
        Args:
            targets: Station numbers, zip codes, coordinates, or addresses
            start_date: Start date for data retrieval
            end_date: End date for data retrieval
            data_items: List of data items to retrieve (uses default if None)
            unit_of_measure: 'E' for English or 'M' for Metric
            csv: If True, automatically export to CSV with auto-generated filename
            filename: Custom filename for CSV export (only used if csv=True)
            
        Returns:
            WeatherData object if csv=False, or tuple of (WeatherData, csv_filename) if csv=True
        """
        if data_items is None:
            data_items = []  # Empty list will get all available data items
        
        weather_data = self.get_data(targets, start_date, end_date, data_items, 
                                   unit_of_measure, prioritize_scs=False)
        
        if csv:
            csv_filename = self.export_to_csv(weather_data, filename)
            return weather_data, csv_filename
        
        return weather_data
    
    def get_stations(self, station_number: Optional[str] = None) -> List[Station]:
        """
        Get station information.
        
        Args:
            station_number: Specific station number (gets all stations if None)
            
        Returns:
            List of Station objects
        """
        params = {}
        endpoint_kwargs = {}
        
        if station_number:
            endpoint_kwargs['station_id'] = station_number
            response_data = self._make_request('station', params, **endpoint_kwargs)
        else:
            response_data = self._make_request('stations', params)
        
        return self.endpoints.parse_stations_response(response_data)
    
    def get_station_zip_codes(self, zip_code: Optional[str] = None) -> List[ZipCode]:
        """
        Get station zip code information.
        
        Args:
            zip_code: Specific zip code (gets all zip codes if None)
            
        Returns:
            List of ZipCode objects
        """
        params = {}
        endpoint_kwargs = {}
        
        if zip_code:
            endpoint_kwargs['zip_code'] = zip_code
            response_data = self._make_request('zip_code', params, **endpoint_kwargs)
        else:
            response_data = self._make_request('zip_codes', params)
        
        return self.endpoints.parse_zip_codes_response(response_data)
    
    def get_spatial_zip_codes(self, zip_code: Optional[str] = None) -> List[SpatialZipCode]:
        """
        Get spatial zip code information.
        
        Args:
            zip_code: Specific zip code (gets all zip codes if None)
            
        Returns:
            List of SpatialZipCode objects
        """
        params = {}
        endpoint_kwargs = {}
        
        if zip_code:
            endpoint_kwargs['zip_code'] = zip_code
            response_data = self._make_request('spatial_zip_code', params, **endpoint_kwargs)
        else:
            response_data = self._make_request('spatial_zip_codes', params)
        
        return self.endpoints.parse_spatial_zip_codes_response(response_data)
    
    def export_to_csv(self, 
                      weather_data: WeatherData, 
                      filename: Optional[Union[str, Path]] = None,
                      include_all_columns: bool = True,
                      separate_daily_hourly: bool = True) -> str:
        """
        Export weather data to CSV file with properly formatted data columns.
        Uses automatic filename generation based on station names and dates by default.
        
        Args:
            weather_data: WeatherData object to export
            filename: Output CSV filename (auto-generated if None)
            include_all_columns: Whether to include all possible data columns
            separate_daily_hourly: Whether to separate daily and hourly data into different files
            
        Returns:
            Path to the created CSV file(s)
        """
        # Generate filename automatically if not provided
        if filename is None:
            filename = self.filename_generator.generate_for_weather_data(weather_data)
        
        filename = Path(filename)
        
        all_records = weather_data.get_all_records()
        if not all_records:
            raise CimisDataError("No data records to export")
        
        # Separate records by scope if requested
        if separate_daily_hourly:
            daily_records = [r for r in all_records if r.scope == 'daily']
            hourly_records = [r for r in all_records if r.scope == 'hourly']
            
            if daily_records and hourly_records:
                # Create separate files for daily and hourly data
                daily_filename = filename.with_name(filename.stem + '_daily' + filename.suffix)
                hourly_filename = filename.with_name(filename.stem + '_hourly' + filename.suffix)
                
                self._export_records_to_csv(weather_data, daily_records, daily_filename, 'daily')
                self._export_records_to_csv(weather_data, hourly_records, hourly_filename, 'hourly')
                
                return f"Daily: {daily_filename}, Hourly: {hourly_filename}"
            elif daily_records:
                return self._export_records_to_csv(weather_data, daily_records, filename, 'daily')
            elif hourly_records:
                return self._export_records_to_csv(weather_data, hourly_records, filename, 'hourly')
        
        # If not separating or only one type, export all together with scope-specific columns
        return self._export_records_to_csv(weather_data, all_records, filename, 'mixed')
    
    def _export_records_to_csv(self, weather_data: WeatherData, records: List, 
                              filename: Path, scope_type: str) -> str:
        """Helper method to export records to CSV with appropriate columns."""
        # Collect data items relevant to the scope type
        all_data_items = set()
        for record in records:
            all_data_items.update(record.data_values.keys())
        
        # Filter data items based on scope type
        if scope_type == 'daily':
            # Only include daily data items
            filtered_data_items = {item for item in all_data_items 
                                 if item.startswith('Day') or not item.startswith(('Day', 'Hly'))}
        elif scope_type == 'hourly':
            # Only include hourly data items
            filtered_data_items = {item for item in all_data_items 
                                 if item.startswith('Hly') or not item.startswith(('Day', 'Hly'))}
        else:
            # Mixed - include all
            filtered_data_items = all_data_items
        
        # Sort data items for consistent column ordering
        sorted_data_items = sorted(filtered_data_items)
        
        # Base columns
        base_columns = [
            'Provider_Name', 'Provider_Type', 'Date', 'Julian', 'Station', 
            'Standard', 'ZipCodes', 'Scope'
        ]
        
        # Add Hour column only if we have hourly data
        if scope_type in ['hourly', 'mixed']:
            base_columns.append('Hour')
        
        # Data value columns (value, qc, unit for each data item)
        data_columns = []
        for item in sorted_data_items:
            data_columns.extend([
                f"{item}_Value",
                f"{item}_QC", 
                f"{item}_Unit"
            ])
        
        all_columns = base_columns + data_columns
        
        # Ensure directory exists
        filename.parent.mkdir(parents=True, exist_ok=True)
        
        # Write CSV
        with open(filename, 'w', newline='', encoding='utf-8') as csvfile:
            writer = csv.DictWriter(csvfile, fieldnames=all_columns)
            writer.writeheader()
            
            for provider in weather_data.providers:
                for record in provider.records:
                    # Skip records not in our filtered list
                    if record not in records:
                        continue
                        
                    row = {
                        'Provider_Name': provider.name,
                        'Provider_Type': provider.type,
                        'Date': record.date,
                        'Julian': record.julian,
                        'Station': record.station or '',
                        'Standard': record.standard,
                        'ZipCodes': record.zip_codes,
                        'Scope': record.scope
                    }
                    
                    # Add Hour column only if needed
                    if 'Hour' in all_columns:
                        row['Hour'] = record.hour or ''
                    
                    # Add data values - only include items that are in our filtered set
                    for item in sorted_data_items:
                        data_value = record.data_values.get(item)
                        if data_value:
                            row[f"{item}_Value"] = data_value.value or ''
                            row[f"{item}_QC"] = data_value.qc
                            row[f"{item}_Unit"] = data_value.unit
                        else:
                            row[f"{item}_Value"] = ''
                            row[f"{item}_QC"] = ''
                            row[f"{item}_Unit"] = ''
                    
                    writer.writerow(row)
        
        return str(filename)
    
    def export_stations_to_csv(self, 
                               stations: List[Station], 
                               filename: Optional[Union[str, Path]] = None) -> str:
        """
        Export station information to CSV file.
        Uses automatic filename generation based on stations data by default.
        
        Args:
            stations: List of Station objects to export
            filename: Output CSV filename (auto-generated if None)
            
        Returns:
            Path to the created CSV file
        """
        # Generate filename automatically if not provided
        if filename is None:
            filename = self.filename_generator.generate_for_stations(stations)
        
        filename = Path(filename)
        
        if not stations:
            raise CimisDataError("No station data to export")
        
        columns = [
            'StationNbr', 'Name', 'City', 'RegionalOffice', 'County',
            'ConnectDate', 'DisconnectDate', 'IsActive', 'IsEtoStation',
            'Elevation', 'GroundCover', 'HmsLatitude', 'HmsLongitude',
            'Latitude', 'Longitude', 'ZipCodes', 'SitingDesc'
        ]
        
        # Ensure directory exists
        filename.parent.mkdir(parents=True, exist_ok=True)
        
        with open(filename, 'w', newline='', encoding='utf-8') as csvfile:
            writer = csv.DictWriter(csvfile, fieldnames=columns)
            writer.writeheader()
            
            for station in stations:
                row = {
                    'StationNbr': station.station_nbr,
                    'Name': station.name,
                    'City': station.city,
                    'RegionalOffice': station.regional_office or '',
                    'County': station.county or '',
                    'ConnectDate': station.connect_date,
                    'DisconnectDate': station.disconnect_date,
                    'IsActive': station.is_active,
                    'IsEtoStation': station.is_eto_station,
                    'Elevation': station.elevation,
                    'GroundCover': station.ground_cover,
                    'HmsLatitude': station.hms_latitude,
                    'HmsLongitude': station.hms_longitude,
                    'Latitude': station.latitude or '',
                    'Longitude': station.longitude or '',
                    'ZipCodes': ', '.join(station.zip_codes),
                    'SitingDesc': station.siting_desc
                }
                writer.writerow(row)
        
        return str(filename)
    
    def get_data_and_export_csv(self,
                                targets: Union[str, List[str]], 
                                start_date: Union[str, date, datetime],
                                end_date: Union[str, date, datetime],
                                filename: Optional[Union[str, Path]] = None,
                                data_items: Optional[List[str]] = None,
                                unit_of_measure: str = 'E',
                                prioritize_scs: bool = True) -> tuple[WeatherData, str]:
        """
        Convenience method to get data and immediately export to CSV.
        Uses automatic filename generation based on station names and dates by default.
        
        Args:
            targets: Station numbers, zip codes, coordinates, or addresses
            start_date: Start date
            end_date: End date
            filename: Output CSV filename (auto-generated if None)
            data_items: List of data items to retrieve (uses default if None)
            unit_of_measure: 'E' for English or 'M' for Metric
            prioritize_scs: Whether to prioritize SCS data for zip codes
            
        Returns:
            Tuple of (WeatherData object, path to created CSV file)
        """
        weather_data = self.get_data(targets, start_date, end_date, data_items,
                                   unit_of_measure, prioritize_scs)
        csv_path = self.export_to_csv(weather_data, filename)
        return weather_data, csv_path
