# Python CIMIS Client

A comprehensive Python client library for the California Irrigation Management Information System (CIMIS) API. This library provides easy access to CIMIS weather station data, spatial data, and station information with built-in CSV export functionality.

[![PyPI version](https://badge.fury.io/py/python-CIMIS.svg)](https://badge.fury.io/py/python-CIMIS)
[![Python versions](https://img.shields.io/pypi/pyversions/python-CIMIS.svg)](https://pypi.org/project/python-CIMIS/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

## Features

- **Comprehensive API Coverage**: Access all CIMIS API endpoints including weather data, station information, and zip code data
- **Multiple Data Sources**: Support for both Weather Station Network (WSN) and Spatial CIMIS System (SCS) data
- **Flexible Data Retrieval**: Get data by station numbers, zip codes, coordinates, or street addresses
- **Built-in CSV Export**: Export all data with comprehensive column coverage by default
- **Easy to Use**: Simple, Pythonic interface with sensible defaults
- **Error Handling**: Comprehensive exception handling with descriptive error messages
- **Type Hints**: Full type hint support for better IDE integration

## Installation

Install from PyPI:

```bash
pip install python-CIMIS
```

## Quick Start

```python
from python_cimis import CimisClient
from datetime import date

# Initialize the client with your API key
client = CimisClient(app_key="your-api-key-here")

# Get daily weather data for specific stations
weather_data = client.get_daily_data(
    targets=[2, 8, 127],  # Station numbers
    start_date="2023-01-01",
    end_date="2023-01-05"
)

# Export to CSV with all available columns
client.export_to_csv(weather_data, "weather_data.csv")

# Or get data and export in one step
weather_data = client.get_data_and_export_csv(
    targets=[2, 8, 127],
    start_date=date(2023, 1, 1),
    end_date=date(2023, 1, 5),
    filename="comprehensive_weather_data.csv"
)
```

## API Key

You need a CIMIS API key to use this library. You can obtain one by registering at the [CIMIS website](https://cimis.water.ca.gov/Default.aspx).

## Usage Examples

### Getting Weather Data

#### By Station Numbers

```python
# Get data for specific weather stations
weather_data = client.get_daily_data(
    targets=[2, 8, 127],
    start_date="2023-01-01",
    end_date="2023-01-31",
    unit_of_measure="E"  # English units (default)
)
```

#### By Zip Codes

```python
# Get data for specific zip codes
weather_data = client.get_daily_data(
    targets=["95823", "94503", "93624"],
    start_date="2023-01-01",
    end_date="2023-01-31",
    prioritize_scs=True  # Prioritize Spatial CIMIS System data
)
```

#### By Coordinates

```python
# Get data for specific coordinates
weather_data = client.get_data(
    targets=["lat=39.36,lng=-121.74", "lat=38.22,lng=-122.82"],
    start_date="2023-01-01",
    end_date="2023-01-31",
    data_items=["day-asce-eto", "day-sol-rad-avg"]  # Only ETo and solar radiation
)
```

#### By Addresses

```python
# Get data for specific addresses
weather_data = client.get_data(
    targets=[
        "addr-name=State Capitol,addr=1315 10th Street Sacramento, CA 95814",
        "addr-name=SF City Hall,addr=1 Dr Carlton B Goodlett Pl, San Francisco, CA 94102"
    ],
    start_date="2023-01-01",
    end_date="2023-01-31",
    data_items=["day-asce-eto", "day-sol-rad-avg"]
)
```

### Getting Station Information

```python
# Get all stations
all_stations = client.get_stations()

# Get specific station
station = client.get_stations(station_number="2")

# Export stations to CSV
client.export_stations_to_csv(all_stations, "all_stations.csv")
```

### Getting Zip Code Information

```python
# Get station zip codes
station_zip_codes = client.get_station_zip_codes()

# Get spatial zip codes
spatial_zip_codes = client.get_spatial_zip_codes()
```

### Custom Data Items

```python
# Specify custom data items
custom_data_items = [
    "day-air-tmp-avg", "day-air-tmp-max", "day-air-tmp-min",
    "day-precip", "day-eto", "day-asce-eto"
]

weather_data = client.get_daily_data(
    targets=[2, 8, 127],
    start_date="2023-01-01",
    end_date="2023-01-31",
    data_items=custom_data_items
)
```

### Hourly Data

```python
# Get hourly data (only available from WSN stations)
hourly_data = client.get_hourly_data(
    targets=[2, 8, 127],
    start_date="2023-01-01",
    end_date="2023-01-01"  # Note: Hourly data requests should be for shorter periods
)
```

## CSV Export Features

The library provides comprehensive CSV export functionality:

- **All Columns by Default**: Exports all available data columns automatically
- **Quality Control Information**: Includes QC flags and units for each data point
- **Provider Information**: Identifies data source (WSN or SCS)
- **Flexible Output**: Customizable column selection and formatting

### CSV Export Example

```python
# Export with all available columns (default behavior)
client.export_to_csv(weather_data, "complete_data.csv")

# The CSV will include columns like:
# Provider_Name, Provider_Type, Date, Julian, Station, Standard, ZipCodes, Scope, Hour,
# day-air-tmp-avg_Value, day-air-tmp-avg_QC, day-air-tmp-avg_Unit,
# day-air-tmp-max_Value, day-air-tmp-max_QC, day-air-tmp-max_Unit,
# ... (all available data items)
```

## Available Data Items

### Daily Data Items (WSN + SCS)

- Temperature: `day-air-tmp-avg`, `day-air-tmp-max`, `day-air-tmp-min`
- Humidity: `day-rel-hum-avg`, `day-rel-hum-max`, `day-rel-hum-min`
- Evapotranspiration: `day-eto`, `day-asce-eto`, `day-asce-etr`
- Solar Radiation: `day-sol-rad-avg`, `day-sol-rad-net`
- Wind: `day-wind-spd-avg`, `day-wind-run`, directional wind components
- Soil: `day-soil-tmp-avg`, `day-soil-tmp-max`, `day-soil-tmp-min`
- Other: `day-precip`, `day-dew-pnt`, `day-vap-pres-avg`, `day-vap-pres-max`

### Hourly Data Items (WSN only)

- Temperature: `hly-air-tmp`, `hly-soil-tmp`
- Humidity: `hly-rel-hum`
- Evapotranspiration: `hly-eto`, `hly-asce-eto`, `hly-asce-etr`
- Solar: `hly-sol-rad`, `hly-net-rad`
- Wind: `hly-wind-spd`, `hly-wind-dir`, `hly-res-wind`
- Other: `hly-precip`, `hly-dew-pnt`, `hly-vap-pres`

## Error Handling

The library provides comprehensive error handling:

```python
from python_cimis import CimisClient, CimisAPIError, CimisAuthenticationError

client = CimisClient("your-api-key")

try:
    weather_data = client.get_daily_data(
        targets=[999999],  # Invalid station
        start_date="2023-01-01",
        end_date="2023-01-01"
    )
except CimisAuthenticationError:
    print("Invalid API key")
except CimisAPIError as e:
    print(f"API Error: {e}")
```

## Data Models

The library includes comprehensive data models:

- `WeatherData`: Container for weather data responses
- `WeatherRecord`: Individual weather data records
- `Station`: Weather station information
- `ZipCode`: Zip code information for WSN stations
- `SpatialZipCode`: Zip code information for SCS data

## Requirements

- Python 3.8+
- requests >= 2.25.0

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## Disclaimer

This library is not affiliated with the California Department of Water Resources or the CIMIS program. It is an independent client library for accessing the public CIMIS API.

## Links

- [CIMIS Website](https://cimis.water.ca.gov/)
- [CIMIS API Documentation](https://et.water.ca.gov/Rest/Index)
- [California Department of Water Resources](https://water.ca.gov/)
