"""
Advanced integration tests based on CIMIS API sample requests.
These tests cover all the sample scenarios from the CIMIS API documentation.
"""

import pytest
from unittest.mock import Mock, patch
from datetime import date, datetime
from python_cimis import CimisClient
from python_cimis.exceptions import (
    CimisAPIError, 
    CimisAuthenticationError, 
    CimisConnectionError,
    CimisDataError
)


class TestSampleDataRequests:
    """Test cases based on official CIMIS API sample requests."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_daily_data_by_station_number_english(self, mock_request):
        """Test daily data by station number in English units."""
        # Mock response similar to the API sample
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "Station": 2,
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Standard": "english",
                        "DayAirTmpAvg": {"Value": 45.2, "Unit": "F", "QC": " "},
                        "DayAirTmpMax": {"Value": 58.3, "Unit": "F", "QC": " "},
                        "DayAirTmpMin": {"Value": 32.1, "Unit": "F", "QC": " "},
                        "DayEto": {"Value": 0.05, "Unit": "in", "QC": " "},
                        "DayPrecip": {"Value": 0.0, "Unit": "in", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=[2, 8, 127],
            start_date="2010-01-01",
            end_date="2010-01-05",
            data_items=[
                "day-air-tmp-avg", "day-air-tmp-max", "day-air-tmp-min",
                "day-dew-pnt", "day-eto", "day-asce-eto", "day-asce-etr", "day-precip"
            ],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()
        # Verify the request parameters
        call_args = mock_request.call_args
        params = call_args[0][1]
        assert 'targets' in params or any('2' in str(v) for v in params.values())
        assert 'startDate' in params or any('2010-01-01' in str(v) for v in params.values())
        assert 'unitOfMeasure' in params or any('E' in str(v) for v in params.values())
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_daily_data_by_station_number_metric(self, mock_request):
        """Test daily data by station number in Metric units."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "Station": 2,
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Standard": "metric",
                        "DayAirTmpAvg": {"Value": 7.3, "Unit": "C", "QC": " "},
                        "DayAirTmpMax": {"Value": 14.6, "Unit": "C", "QC": " "},
                        "DayAirTmpMin": {"Value": 0.1, "Unit": "C", "QC": " "},
                        "DayEto": {"Value": 1.3, "Unit": "mm", "QC": " "},
                        "DayPrecip": {"Value": 0.0, "Unit": "mm", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=[2, 8, 127],
            start_date="2010-01-01",
            end_date="2010-01-05",
            data_items=[
                "day-air-tmp-avg", "day-air-tmp-max", "day-air-tmp-min",
                "day-dew-pnt", "day-eto", "day-asce-eto", "day-asce-etr", "day-precip"
            ],
            unit_of_measure="M"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_daily_data_by_zip_code_english(self, mock_request):
        """Test daily data by zip code in English units."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "ZipCode": "93624",
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Standard": "english",
                        "DayAirTmpAvg": {"Value": 42.8, "Unit": "F", "QC": " "},
                        "DayEto": {"Value": 0.04, "Unit": "in", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=["93624", "94503", "95667"],
            start_date="2010-01-01",
            end_date="2010-01-05",
            data_items=[
                "day-air-tmp-avg", "day-air-tmp-max", "day-air-tmp-min",
                "day-dew-pnt", "day-eto", "day-asce-eto", "day-asce-etr", "day-precip"
            ],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_daily_data_by_coordinates_english(self, mock_request):
        """Test daily data by coordinates in English units."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "Latitude": 39.36,
                        "Longitude": -121.74,
                        "Date": "2010-03-15",
                        "Julian": "74",
                        "Standard": "english",
                        "DaySolRadAvg": {"Value": 520.2, "Unit": "ly", "QC": " "},
                        "DayAsceEto": {"Value": 0.15, "Unit": "in", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=["lat=39.36,lng=-121.74", "lat=38.22,lng=-122.82"],
            start_date="2010-03-15",
            end_date="2010-03-17",
            data_items=["day-sol-rad-avg", "day-asce-eto"],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_daily_data_by_address_english(self, mock_request):
        """Test daily data by address in English units."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "AddressName": "State Capitol",
                        "Address": "1315 10th Street Sacramento, CA 95814",
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Standard": "english",
                        "DaySolRadAvg": {"Value": 180.5, "Unit": "ly", "QC": " "},
                        "DayAsceEto": {"Value": 0.02, "Unit": "in", "QC": " "}
                    }
                ]
            }
        }
        
        # Note: Addresses need to be URL encoded in actual use
        data = self.client.get_data(
            targets=[
                "addr-name=State Capitol,addr=1315 10th Street Sacramento, CA 95814",
                "addr-name=Grauman's Chinese Theatre,addr=6925 Hollywood Boulevard Los Angeles, CA 90028"
            ],
            start_date="2010-01-01",
            end_date="2010-01-05",
            data_items=["day-sol-rad-avg", "day-asce-eto"],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_hourly_data_by_station_number_english(self, mock_request):
        """Test hourly data by station number in English units."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "Station": 2,
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Hour": "0800",
                        "Standard": "english",
                        "HlyAirTmp": {"Value": 38.5, "Unit": "F", "QC": " "},
                        "HlyRelHum": {"Value": 85.2, "Unit": "%", "QC": " "},
                        "HlyWindSpd": {"Value": 2.1, "Unit": "mph", "QC": " "},
                        "HlyEto": {"Value": 0.001, "Unit": "in", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_hourly_data(
            targets=[2, 8, 127],
            start_date="2010-01-01",
            end_date="2010-01-05",
            data_items=[
                "hly-air-tmp", "hly-dew-pnt", "hly-eto", "hly-net-rad",
                "hly-asce-eto", "hly-asce-etr", "hly-precip", "hly-rel-hum",
                "hly-res-wind", "hly-soil-tmp", "hly-sol-rad", "hly-vap-pres",
                "hly-wind-dir", "hly-wind-spd"
            ],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_mixed_daily_and_hourly_data(self, mock_request):
        """Test mixed daily and hourly data request."""
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": [
                    {
                        "Station": 2,
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Standard": "english",
                        "DayAirTmpAvg": {"Value": 45.2, "Unit": "F", "QC": " "},
                        "DayEto": {"Value": 0.05, "Unit": "in", "QC": " "}
                    },
                    {
                        "Station": 2,
                        "Date": "2010-01-01",
                        "Julian": "1",
                        "Hour": "0800",
                        "Standard": "english",
                        "HlySoilTmp": {"Value": 42.1, "Unit": "F", "QC": " "},
                        "HlyWindSpd": {"Value": 3.2, "Unit": "mph", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=[2, 8, 127],
            start_date="2010-01-01",
            end_date="2010-01-01",
            data_items=[
                "day-air-tmp-avg", "day-air-tmp-max", "day-air-tmp-min",
                "day-dew-pnt", "day-eto", "day-asce-eto", "day-asce-etr", "day-precip",
                "hly-soil-tmp", "hly-sol-rad", "hly-vap-pres", "hly-wind-dir", "hly-wind-spd"
            ],
            unit_of_measure="E"
        )
        
        assert data is not None
        mock_request.assert_called_once()


class TestSampleStationRequests:
    """Test cases for station information requests."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_get_all_stations(self, mock_request):
        """Test getting all stations."""
        mock_request.return_value = {
            "Stations": [
                {
                    "StationNbr": 2,
                    "Name": "Five Points",
                    "City": "Five Points",
                    "RegionalOffice": "Fresno",
                    "County": "Fresno",
                    "ConnectDate": "1982-06-07",
                    "DisconnectDate": "",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": 283,
                    "GroundCover": "Grass",
                    "HmsLatitude": "36°20'10N / 36.3360",
                    "HmsLongitude": "120°6'47W / -120.1130"
                },
                {
                    "StationNbr": 6,
                    "Name": "Davis",
                    "City": "Davis",
                    "RegionalOffice": "Sacramento",
                    "County": "Yolo",
                    "ConnectDate": "1982-06-07",
                    "DisconnectDate": "",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": 61,
                    "GroundCover": "Grass",
                    "HmsLatitude": "38°32'8N / 38.5355",
                    "HmsLongitude": "121°46'24W / -121.7733"
                }
            ]
        }
        
        stations = self.client.get_stations()
        
        assert stations is not None
        assert len(stations) >= 0
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_get_specific_station(self, mock_request):
        """Test getting a specific station."""
        mock_request.return_value = {
            "Stations": [
                {
                    "StationNbr": 2,
                    "Name": "Five Points",
                    "City": "Five Points",
                    "RegionalOffice": "Fresno",
                    "County": "Fresno",
                    "ConnectDate": "1982-06-07",
                    "DisconnectDate": "",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": 283,
                    "GroundCover": "Grass",
                    "HmsLatitude": "36°20'10N / 36.3360",
                    "HmsLongitude": "120°6'47W / -120.1130"
                }
            ]
        }
        
        stations = self.client.get_stations("2")
        
        assert stations is not None
        mock_request.assert_called_once()


class TestExceptionScenarios:
    """Test exception scenarios based on CIMIS API error examples."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_invalid_app_key(self, mock_request):
        """Test invalid application key error."""
        mock_request.side_effect = CimisAuthenticationError(
            "Invalid application key [FAKE-APP-KEY]",
            "ERR1006",
            403
        )
        
        with pytest.raises(CimisAuthenticationError) as excinfo:
            self.client.get_data([2], "2010-01-01", "2010-01-01")
        
        assert "Invalid application key" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1006"
        assert excinfo.value.http_code == 403
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_station_not_found(self, mock_request):
        """Test station not found error."""
        mock_request.side_effect = CimisAPIError(
            "Station number 376 does not exist in the CIMIS system",
            "ERR1019",
            404
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data([376], "2010-01-01", "2010-01-01")
        
        assert "Station number 376 does not exist" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1019"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_unsupported_zip_code(self, mock_request):
        """Test unsupported zip code error."""
        mock_request.side_effect = CimisAPIError(
            "Zip code 99923 is currently unsupported by CIMIS",
            "ERR1031",
            404
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(["99923"], "2010-01-01", "2010-01-01")
        
        assert "Zip code 99923 is currently unsupported" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1031"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_coordinates_not_in_california(self, mock_request):
        """Test coordinates not in California error."""
        mock_request.side_effect = CimisAPIError(
            "The coordinates provided are not recognized by GoogleMaps as being within the state of California",
            "ERR1034",
            404
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            # Paris coordinates
            self.client.get_data(
                ["lat=48.89364,lng=2.33739"],
                "2010-01-01",
                "2010-01-01",
                data_items=["day-asce-eto"]
            )
        
        assert "not recognized by GoogleMaps" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1034"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_invalid_coordinate_format(self, mock_request):
        """Test invalid coordinate format error."""
        mock_request.side_effect = CimisAPIError(
            "Numeric latitude or longitude values cannot be parsed: lat=abc,lng=def",
            "ERR1025",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                ["lat=abc,lng=def"],
                "2010-01-01",
                "2010-01-01",
                data_items=["day-asce-eto"]
            )
        
        assert "cannot be parsed" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1025"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_invalid_target_mix(self, mock_request):
        """Test invalid target mix error."""
        mock_request.side_effect = CimisAPIError(
            "Invalid target specified. The target must be either a list of stations, California zip codes, or California coordinates",
            "ERR2006",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            # Mixing station numbers and zip codes
            self.client.get_data(
                [2, "95823"],
                "2010-01-01",
                "2010-01-01",
                data_items=["day-asce-eto"]
            )
        
        assert "Invalid target specified" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR2006"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_invalid_data_item(self, mock_request):
        """Test invalid data item error."""
        mock_request.side_effect = CimisAPIError(
            "Invalid data item: 'day-fake'. See API documentation for a list of valid data items",
            "ERR1035",
            404
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2],
                "2010-01-01",
                "2010-01-01",
                data_items=["day-fake"]
            )
        
        assert "Invalid data item" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1035"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_future_date_fault(self, mock_request):
        """Test future date fault error."""
        mock_request.side_effect = CimisAPIError(
            "Dates must be less than or equal to today's date",
            "ERR1010",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2],
                "2030-01-01",
                "2030-01-01",
                data_items=["day-asce-eto"]
            )
        
        assert "less than or equal to today" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1010"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_origin_date_fault(self, mock_request):
        """Test origin date fault error."""
        mock_request.side_effect = CimisAPIError(
            "Dates must be greater than or equal to June 7st, 1982",
            "ERR1011",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2],
                "1955-01-01",
                "1955-01-01",
                data_items=["day-asce-eto"]
            )
        
        assert "greater than or equal to June 7" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1011"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_date_order_fault(self, mock_request):
        """Test date order fault error."""
        mock_request.side_effect = CimisAPIError(
            "The start date may not be greater than the end date",
            "ERR1012",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2],
                "2010-01-02",
                "2010-01-01",  # End date before start date
                data_items=["day-asce-eto"]
            )
        
        assert "start date may not be greater" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1012"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_invalid_unit_of_measure(self, mock_request):
        """Test invalid unit of measure error."""
        mock_request.side_effect = CimisAPIError(
            "Invalid unit of measure specified. The value must be either 'E' for English or 'M' for Metric",
            "ERR1032",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2],
                "2010-01-01",
                "2010-01-01",
                data_items=["day-asce-eto"],
                unit_of_measure="Z"  # Invalid unit
            )
        
        assert "Invalid unit of measure" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR1032"
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_data_volume_violation(self, mock_request):
        """Test data volume violation error."""
        mock_request.side_effect = CimisAPIError(
            "The daily report request exceeds the maximum data limit of 1,750 records. Your request: 5 stations * 1,004 days = 5,020 records",
            "ERR2112",
            400
        )
        
        with pytest.raises(CimisAPIError) as excinfo:
            self.client.get_data(
                [2, 8, 80, 127, 128],
                "2010-01-01",
                "2012-10-01",  # Very long date range
                data_items=["day-asce-eto"]
            )
        
        assert "exceeds the maximum data limit" in str(excinfo.value)
        assert excinfo.value.error_code == "ERR2112"


class TestAdvancedIntegrationScenarios:
    """Advanced integration test scenarios."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_large_data_request_and_export(self, mock_request):
        """Test large data request and CSV export."""
        # Mock a large dataset
        records = []
        for day in range(1, 31):  # 30 days
            for station in [2, 6, 8]:  # 3 stations
                records.append({
                    "Station": station,
                    "Date": f"2023-01-{day:02d}",
                    "Julian": str(day),
                    "Standard": "english",
                    "DayAirTmpAvg": {"Value": 45.0 + day, "Unit": "F", "QC": " "},
                    "DayEto": {"Value": 0.05 + (day * 0.001), "Unit": "in", "QC": " "}
                })
        
        mock_request.return_value = {
            "Data": {
                "Providers": [{"Name": "CIMIS"}],
                "Records": records
            }
        }
        
        with patch('builtins.open'), patch('csv.DictWriter'):
            weather_data, filename = self.client.get_data_and_export_csv(
                targets=[2, 6, 8],
                start_date="2023-01-01",
                end_date="2023-01-30",
                data_items=["day-air-tmp-avg", "day-eto"]
            )
        
        assert filename is not None
        assert weather_data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_multi_provider_data_handling(self, mock_request):
        """Test handling of data from multiple providers."""
        mock_request.return_value = {
            "Data": {
                "Providers": [
                    {"Name": "CIMIS", "Type": "station"},
                    {"Name": "CIMIS", "Type": "spatial"}
                ],
                "Records": [
                    {
                        "Provider": "cimis",
                        "ProviderType": "station",
                        "Station": 2,
                        "Date": "2023-01-01",
                        "DayAirTmpAvg": {"Value": 45.0, "Unit": "F", "QC": " "}
                    },
                    {
                        "Provider": "cimis",
                        "ProviderType": "spatial",
                        "ZipCode": "93624",
                        "Date": "2023-01-01",
                        "DayAirTmpAvg": {"Value": 46.0, "Unit": "F", "QC": " "}
                    }
                ]
            }
        }
        
        data = self.client.get_data(
            targets=[2, "93624"],  # Mixed targets
            start_date="2023-01-01",
            end_date="2023-01-01"
        )
        
        assert data is not None
        mock_request.assert_called_once()
    
    @pytest.mark.integration 
    def test_date_object_handling(self):
        """Test proper handling of Python date and datetime objects."""
        with patch.object(self.client, '_make_request') as mock_request:
            mock_request.return_value = {"Data": {"Providers": [], "Records": []}}
            
            # Test with date objects
            start_date = date(2023, 1, 1)
            end_date = date(2023, 1, 5)
            
            self.client.get_data([2], start_date, end_date)
            
            # Test with datetime objects
            start_datetime = datetime(2023, 1, 1, 10, 30)
            end_datetime = datetime(2023, 1, 5, 15, 45)
            
            self.client.get_data([2], start_datetime, end_datetime)
            
            assert mock_request.call_count == 2
    
    @pytest.mark.integration
    @patch('python_cimis.client.CimisClient._make_request')
    def test_error_recovery_and_retry_simulation(self, mock_request):
        """Test simulated error recovery scenarios."""
        # First call fails, second succeeds
        mock_request.side_effect = [
            CimisConnectionError("Temporary network error"),
            {"Data": {"Providers": [{"Name": "CIMIS"}], "Records": []}}
        ]
        
        # First call should fail
        with pytest.raises(CimisConnectionError):
            self.client.get_data([2], "2023-01-01", "2023-01-01")
        
        # Second call should succeed
        data = self.client.get_data([2], "2023-01-01", "2023-01-01")
        assert data is not None
        assert mock_request.call_count == 2
