"""
Test suite for auto_filename_generation module.
"""

import pytest
import tempfile
import os
from pathlib import Path
from datetime import datetime
from unittest.mock import Mock, patch

from python_cimis.auto_filename_generation import (
    AutoFilenameGenerator,
    generate_weather_filename,
    generate_stations_filename,
    generate_zip_codes_filename,
    generate_custom_filename
)
from python_cimis.models import WeatherData, Station


class TestAutoFilenameGenerator:
    """Test cases for AutoFilenameGenerator class."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_init_default(self):
        """Test generator initialization with default base directory."""
        generator = AutoFilenameGenerator()
        assert generator.base_directory is None
    
    def test_init_with_directory(self):
        """Test generator initialization with custom base directory."""
        test_dir = "/tmp/test"
        generator = AutoFilenameGenerator(test_dir)
        assert str(generator.base_directory) == test_dir
    
    def test_set_base_directory(self):
        """Test setting base directory."""
        test_dir = "/tmp/test"
        self.generator.set_base_directory(test_dir)
        assert str(self.generator.base_directory) == test_dir
    
    def test_generate_for_weather_data(self):
        """Test weather data filename generation."""
        # Create mock weather data
        weather_data = Mock()
        weather_data.get_stations.return_value = ["Davis", "Five Points"]
        weather_data.get_date_range.return_value = ["2023-01-01", "2023-01-05"]
        weather_data.providers = [{"Name": "cimis"}]
        weather_data.records = [
            {"Station": 6, "Date": "2023-01-01"},
            {"Station": 2, "Date": "2023-01-05"}
        ]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert "cimis_weather" in filename
        assert "2023-01-01_to_2023-01-05" in filename
        assert filename.endswith(".csv")
        assert "Davis_FivePoints" in filename
    
    def test_generate_for_weather_data_single_station(self):
        """Test weather data filename generation for single station."""
        weather_data = Mock()
        weather_data.get_stations.return_value = ["Davis"]
        weather_data.get_date_range.return_value = ["2023-01-01", "2023-01-01"]
        weather_data.providers = [{"Name": "cimis"}]
        weather_data.records = [{"Station": 6, "Date": "2023-01-01"}]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        
        assert "cimis_weather" in filename
        assert "Davis" in filename
        assert "2023-01-01" in filename
        assert filename.endswith(".csv")
    
    def test_generate_for_stations(self):
        """Test stations filename generation."""
        # Create mock stations
        station1 = Mock()
        station1.name = "Davis"
        station1.station_nbr = "6"
        
        station2 = Mock()
        station2.name = "Five Points"
        station2.station_nbr = "2"
        
        stations = [station1, station2]
        
        filename = self.generator.generate_for_stations(stations)
        
        assert "cimis_stations" in filename
        assert "Davis_FivePoints" in filename
        assert filename.endswith(".csv")
        # Should include timestamp
        assert len(filename.split("_")) >= 4
    
    def test_generate_for_zip_codes(self):
        """Test zip codes filename generation."""
        zip_codes = ["93624", "94503", "95667"]
        
        filename = self.generator.generate_for_zip_codes(zip_codes)
        
        assert "cimis_zipcodes" in filename
        assert "93624_94503_95667" in filename
        assert filename.endswith(".csv")
    
    def test_generate_for_zip_codes_single(self):
        """Test zip codes filename generation for single zip code."""
        zip_codes = ["93624"]
        
        filename = self.generator.generate_for_zip_codes(zip_codes)
        
        assert "cimis_zipcodes" in filename
        assert "93624" in filename
        assert filename.endswith(".csv")
    
    def test_generate_custom(self):
        """Test custom filename generation."""
        filename = self.generator.generate_custom(
            data_type="test_data",
            identifiers=["id1", "id2"],
            date_range=["2023-01-01", "2023-01-05"],
            provider="test_provider"
        )
        
        assert "test_provider_test_data" in filename
        assert "id1_id2" in filename
        assert "2023-01-01_to_2023-01-05" in filename
        assert filename.endswith(".csv")
    
    def test_generate_custom_with_timestamp(self):
        """Test custom filename generation with timestamp."""
        filename = self.generator.generate_custom(
            data_type="test_data",
            identifiers=["id1"],
            include_timestamp=True
        )
        
        assert "cimis_test_data" in filename
        assert "id1" in filename
        assert filename.endswith(".csv")
        # Should include timestamp
        assert len(filename.split("_")) >= 4
    
    def test_format_station_names(self):
        """Test station names formatting."""
        stations = ["Davis Weather Station", "Five Points"]
        result = self.generator._format_station_names(stations)
        assert result == "DavisWeatherStation_FivePoints"
    
    def test_format_station_names_long_list(self):
        """Test station names formatting for long list."""
        stations = ["Station1", "Station2", "Station3", "Station4", "Station5", "Station6"]
        result = self.generator._format_station_names(stations)
        # Should limit to 5 stations and add "Plus"
        expected_parts = result.split("_")
        assert "Plus" in expected_parts[-1] or len(expected_parts) <= 5
    
    def test_format_date_range_single_date(self):
        """Test date range formatting for single date."""
        dates = ["2023-01-01"]
        result = self.generator._format_date_range(dates)
        assert result == "2023-01-01"
    
    def test_format_date_range_multiple_dates(self):
        """Test date range formatting for multiple dates."""
        dates = ["2023-01-01", "2023-01-05"]
        result = self.generator._format_date_range(dates)
        assert result == "2023-01-01_to_2023-01-05"
    
    def test_format_date_range_many_dates(self):
        """Test date range formatting for many dates."""
        dates = ["2023-01-01", "2023-01-02", "2023-01-03", "2023-01-04", "2023-01-05"]
        result = self.generator._format_date_range(dates)
        assert result == "2023-01-01_to_2023-01-05"
    
    def test_sanitize_name(self):
        """Test name sanitization."""
        assert self.generator._sanitize_name("Test Name") == "TestName"
        assert self.generator._sanitize_name("Test-Name_123") == "Test-Name_123"
        assert self.generator._sanitize_name("Test@Name#123!") == "TestName123"
    
    def test_sanitize_filename(self):
        """Test filename sanitization."""
        assert self.generator._sanitize_filename("test<file>name.csv") == "testfilename.csv"
        assert self.generator._sanitize_filename("test|file?name") == "testfilename"
        assert self.generator._sanitize_filename("test:file*name") == "testfilename"
    
    def test_generate_with_base_directory(self):
        """Test filename generation with base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            self.generator.set_base_directory(temp_dir)
            
            zip_codes = ["93624"]
            filename = self.generator.generate_for_zip_codes(zip_codes)
            
            assert filename.startswith(temp_dir)
            assert "93624" in filename


class TestModuleFunctions:
    """Test module-level convenience functions."""
    
    def test_generate_weather_filename(self):
        """Test generate_weather_filename function."""
        weather_data = Mock()
        weather_data.get_stations.return_value = ["Davis"]
        weather_data.get_date_range.return_value = ["2023-01-01"]
        weather_data.providers = [{"Name": "cimis"}]
        weather_data.records = [{"Station": 6, "Date": "2023-01-01"}]
        
        filename = generate_weather_filename(weather_data)
        
        assert "cimis_weather" in filename
        assert "Davis" in filename
        assert filename.endswith(".csv")
    
    def test_generate_weather_filename_with_directory(self):
        """Test generate_weather_filename function with base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            weather_data = Mock()
            weather_data.get_stations.return_value = ["Davis"]
            weather_data.get_date_range.return_value = ["2023-01-01"]
            weather_data.providers = [{"Name": "cimis"}]
            weather_data.records = [{"Station": 6, "Date": "2023-01-01"}]
            
            filename = generate_weather_filename(weather_data, base_directory=temp_dir)
            
            assert filename.startswith(temp_dir)
            assert "Davis" in filename
    
    def test_generate_stations_filename(self):
        """Test generate_stations_filename function."""
        station = Mock()
        station.name = "Davis"
        station.station_nbr = "6"
        
        filename = generate_stations_filename([station])
        
        assert "cimis_stations" in filename
        assert "Davis" in filename
        assert filename.endswith(".csv")
    
    def test_generate_zip_codes_filename(self):
        """Test generate_zip_codes_filename function."""
        filename = generate_zip_codes_filename(["93624", "94503"])
        
        assert "cimis_zipcodes" in filename
        assert "93624_94503" in filename
        assert filename.endswith(".csv")
    
    def test_generate_custom_filename(self):
        """Test generate_custom_filename function."""
        filename = generate_custom_filename(
            data_type="test",
            identifiers=["id1"],
            date_range=["2023-01-01"],
            provider="test_provider"
        )
        
        assert "test_provider_test" in filename
        assert "id1" in filename
        assert "2023-01-01" in filename
        assert filename.endswith(".csv")


class TestEdgeCases:
    """Test edge cases and error conditions."""
    
    def setup_method(self):
        """Set up test generator."""
        self.generator = AutoFilenameGenerator()
    
    def test_empty_station_list(self):
        """Test handling of empty station list."""
        filename = self.generator.generate_for_stations([])
        
        assert "cimis_stations" in filename
        assert filename.endswith(".csv")
    
    def test_empty_zip_codes_list(self):
        """Test handling of empty zip codes list."""
        filename = self.generator.generate_for_zip_codes([])
        
        assert "cimis_zipcodes" in filename
        assert filename.endswith(".csv")
    
    def test_none_values_in_custom(self):
        """Test handling of None values in custom generation."""
        filename = self.generator.generate_custom(
            data_type="test",
            identifiers=None,
            date_range=None,
            provider=None
        )
        
        assert "cimis_test" in filename
        assert filename.endswith(".csv")
    
    def test_special_characters_in_station_names(self):
        """Test handling of special characters in station names."""
        station = Mock()
        station.name = "Davis/Five-Points@Weather#Station!"
        station.station_nbr = "6"
        
        filename = self.generator.generate_for_stations([station])
        
        # Should sanitize special characters
        assert "/" not in filename
        assert "@" not in filename
        assert "#" not in filename
        assert "!" not in filename
        assert "cimis_stations" in filename
    
    def test_very_long_station_names(self):
        """Test handling of very long station names."""
        station = Mock()
        station.name = "A" * 100  # Very long name
        station.station_nbr = "6"
        
        filename = self.generator.generate_for_stations([station])
        
        # Should handle long names gracefully
        assert len(filename) < 255  # Typical max filename length
        assert "cimis_stations" in filename
    
    @patch('python_cimis.auto_filename_generation.datetime')
    def test_timestamp_generation(self, mock_datetime):
        """Test timestamp generation."""
        # Mock datetime to return a fixed time
        mock_datetime.now.return_value.strftime.return_value = "20230101_120000"
        
        filename = self.generator.generate_custom(
            data_type="test",
            identifiers=["id1"],
            include_timestamp=True
        )
        
        assert "20230101_120000" in filename
        mock_datetime.now.assert_called_once()
