"""
Test suite for the Python CIMIS Client library.
"""

import pytest
import json
import requests
from datetime import date, datetime
from unittest.mock import Mock, patch, MagicMock
from python_cimis import CimisClient, CimisAPIError, CimisAuthenticationError
from python_cimis.models import WeatherData, Station
from python_cimis.exceptions import CimisDataError, CimisConnectionError


class TestCimisClient:
    """Test cases for CimisClient class."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    def test_init(self):
        """Test client initialization."""
        assert self.client.app_key == "test-api-key"
        assert self.client.timeout == 30
        assert self.client.session is not None
    
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    @patch('requests.Session.get')
    def test_make_request_success(self, mock_get, mock_get_url):
        """Test successful API request."""
        # Mock URL generation
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        
        # Mock response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {"test": "data"}
        mock_get.return_value = mock_response
        
        result = self.client._make_request("data", {"param": "value"})
        
        assert result == {"test": "data"}
        mock_get.assert_called_once()
        
        # Check that appKey was added to params
        call_args = mock_get.call_args
        params = call_args[1]['params']
        assert params['appKey'] == "test-api-key"
        assert params['param'] == "value"
    
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    @patch('requests.Session.get')
    def test_make_request_auth_error(self, mock_get, mock_get_url):
        """Test authentication error."""
        # Mock URL generation
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        
        mock_response = Mock()
        mock_response.status_code = 403
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisAuthenticationError):
            self.client._make_request("data", {})
    
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    @patch('requests.Session.get')
    def test_make_request_api_error(self, mock_get, mock_get_url):
        """Test API error."""
        # Mock URL generation
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        
        mock_response = Mock()
        mock_response.status_code = 404
        mock_response.json.return_value = {"Message": "Not found"}
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisAPIError):
            self.client._make_request("data", {})
    
    def test_parse_data_response(self):
        """Test parsing of weather data response."""
        sample_response = {
            "Data": {
                "Providers": [
                    {
                        "Name": "cimis",
                        "Type": "station",
                        "Owner": "water.ca.gov",
                        "Records": [
                            {
                                "Date": "2023-01-01",
                                "Julian": "1",
                                "Station": "2",
                                "Standard": "english",
                                "ZipCodes": "93624",
                                "Scope": "daily",
                                "DayAirTmpAvg": {
                                    "Value": "39",
                                    "Qc": " ",
                                    "Unit": "(F)"
                                }
                            }
                        ]
                    }
                ]
            }
        }
        
        weather_data = self.client._parse_data_response(sample_response)
        
        assert isinstance(weather_data, WeatherData)
        assert len(weather_data.providers) == 1
        
        provider = weather_data.providers[0]
        assert provider.name == "cimis"
        assert provider.type == "station"
        assert len(provider.records) == 1
        
        record = provider.records[0]
        assert record.date == "2023-01-01"
        assert record.station == "2"
        assert "DayAirTmpAvg" in record.data_values
        
        data_value = record.data_values["DayAirTmpAvg"]
        assert data_value.value == "39"
        assert data_value.unit == "(F)"
    
    def test_parse_stations_response(self):
        """Test parsing of stations response."""
        sample_response = {
            "Stations": [
                {
                    "StationNbr": "2",
                    "Name": "FivePoints",
                    "City": "Five Points",
                    "ConnectDate": "6/7/1982",
                    "DisconnectDate": "12/31/2030",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": "285",
                    "GroundCover": "Grass",
                    "HmsLatitude": "36°20'10N / 36.3360",
                    "HmsLongitude": "-120°6'47W / -120.1130",
                    "ZipCodes": ["93624"],
                    "SitingDesc": ""
                }
            ]
        }
        
        stations = self.client._parse_stations_response(sample_response)
        
        assert len(stations) == 1
        assert isinstance(stations[0], Station)
        
        station = stations[0]
        assert station.station_nbr == "2"
        assert station.name == "FivePoints"
        assert station.city == "Five Points"
        assert station.is_active is True
        assert station.is_eto_station is True
        assert station.latitude == 36.3360
        assert station.longitude == -120.1130
    
    def test_is_coordinate_list(self):
        """Test coordinate detection."""
        coord_targets = ["lat=39.36,lng=-121.74", "lat=38.22,lng=-122.82"]
        non_coord_targets = ["95823", "94503"]
        
        assert self.client.endpoints._is_coordinate_list(coord_targets) is True
        assert self.client.endpoints._is_coordinate_list(non_coord_targets) is False
    
    def test_is_address_list(self):
        """Test address detection."""
        addr_targets = ["addr-name=Test,addr=123 Main St"]
        non_addr_targets = ["95823", "94503"]
        
        assert self.client.endpoints._is_address_list(addr_targets) is True
        assert self.client.endpoints._is_address_list(non_addr_targets) is False
    
    @patch.object(CimisClient, '_make_request')
    @patch('python_cimis.endpoints.CimisEndpoints.parse_data_response')
    def test_get_data(self, mock_parse, mock_request):
        """Test get_data method."""
        mock_request.return_value = {"test": "data"}
        mock_parse.return_value = WeatherData()
        
        result = self.client.get_data(
            targets=[2, 8],
            start_date="2023-01-01",
            end_date="2023-01-02",
            data_items=["day-air-tmp-avg"]
        )
        
        assert isinstance(result, WeatherData)
        mock_request.assert_called_once()
        mock_parse.assert_called_once_with({"test": "data"})
        
        # Check parameters - the method uses prepare_data_params internally
        call_args = mock_request.call_args[0]
        params = call_args[1]  # Second argument should be params
        
        assert call_args[0] == "data"
        # params are prepared by prepare_data_params(), so check for expected keys
        assert 'targets' in params
        assert 'startDate' in params
        assert 'endDate' in params
    
    def test_get_data_with_dates(self):
        """Test get_data with date objects."""
        with patch.object(self.client, '_make_request') as mock_request, \
             patch.object(self.client.endpoints, 'parse_data_response') as mock_parse:
            
            mock_request.return_value = {"test": "data"}
            mock_parse.return_value = WeatherData()
            
            start_date = date(2023, 1, 1)
            end_date = date(2023, 1, 2)
            
            self.client.get_data(
                targets=[2],
                start_date=start_date,
                end_date=end_date
            )
            
            # Check that the request was made with proper endpoint and parameters
            mock_request.assert_called_once()
            call_args = mock_request.call_args
            assert call_args[0][0] == 'data'  # endpoint name
            params = call_args[0][1]  # parameters dict
            assert 'startDate' in params
            assert 'endDate' in params


class TestWeatherData:
    """Test cases for WeatherData class."""
    
    def test_get_all_records(self):
        """Test getting all records from weather data."""
        from python_cimis.models import WeatherProvider, WeatherRecord
        
        weather_data = WeatherData()
        provider1 = WeatherProvider(name="p1", type="station", owner="test")
        provider2 = WeatherProvider(name="p2", type="spatial", owner="test")
        
        record1 = WeatherRecord(date="2023-01-01", julian="1")
        record2 = WeatherRecord(date="2023-01-02", julian="2")
        record3 = WeatherRecord(date="2023-01-03", julian="3")
        
        provider1.records = [record1, record2]
        provider2.records = [record3]
        
        weather_data.providers = [provider1, provider2]
        
        all_records = weather_data.get_all_records()
        assert len(all_records) == 3
        assert record1 in all_records
        assert record2 in all_records
        assert record3 in all_records


class TestStation:
    """Test cases for Station class."""
    
    def test_latitude_property(self):
        """Test latitude extraction from HMS format."""
        station = Station(
            station_nbr="1",
            name="Test",
            city="Test City",
            hms_latitude="36°20'10N / 36.3360"
        )
        assert station.latitude == 36.3360
    
    def test_longitude_property(self):
        """Test longitude extraction from HMS format."""
        station = Station(
            station_nbr="1", 
            name="Test",
            city="Test City",
            hms_longitude="-120°6'47W / -120.1130"
        )
        assert station.longitude == -120.1130
    
    def test_invalid_coordinates(self):
        """Test handling of invalid coordinates."""
        station = Station(
            station_nbr="1",
            name="Test", 
            city="Test City",
            hms_latitude="invalid",
            hms_longitude="invalid"
        )
        assert station.latitude is None
        assert station.longitude is None


class TestClientAdditionalMethods:
    """Test additional client methods for better coverage."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @patch.object(CimisClient, '_make_request')
    def test_get_stations_with_specific_station(self, mock_request):
        """Test get_stations with specific station number."""
        mock_request.return_value = {"Stations": [{"StationNbr": 6, "Name": "Davis"}]}
        
        with patch.object(self.client.endpoints, 'parse_stations_response') as mock_parse:
            mock_parse.return_value = [Station(station_nbr="6", name="Davis", city="Davis")]
            
            result = self.client.get_stations("6")
            
            assert isinstance(result, list)
            mock_request.assert_called_once()
    
    @patch.object(CimisClient, '_make_request')
    def test_get_station_zip_codes(self, mock_request):
        """Test get_station_zip_codes method."""
        mock_request.return_value = {"ZipCodes": [{"ZipCode": "93624"}]}
        
        with patch.object(self.client.endpoints, 'parse_zip_codes_response') as mock_parse:
            mock_parse.return_value = []
            
            result = self.client.get_station_zip_codes()
            
            assert isinstance(result, list)
            mock_request.assert_called_once()
    
    @patch.object(CimisClient, '_make_request')
    def test_get_station_zip_codes_specific(self, mock_request):
        """Test get_station_zip_codes with specific zip code."""
        mock_request.return_value = {"ZipCodes": [{"ZipCode": "93624"}]}
        
        with patch.object(self.client.endpoints, 'parse_zip_codes_response') as mock_parse:
            mock_parse.return_value = []
            
            result = self.client.get_station_zip_codes("93624")
            
            assert isinstance(result, list)
            mock_request.assert_called_once()
    
    @patch.object(CimisClient, '_make_request')
    def test_get_spatial_zip_codes(self, mock_request):
        """Test get_spatial_zip_codes method."""
        mock_request.return_value = {"ZipCodes": [{"ZipCode": "93624"}]}
        
        with patch.object(self.client.endpoints, 'parse_spatial_zip_codes_response') as mock_parse:
            mock_parse.return_value = []
            
            result = self.client.get_spatial_zip_codes()
            
            assert isinstance(result, list)
            mock_request.assert_called_once()
    
    @patch.object(CimisClient, '_make_request')
    def test_get_spatial_zip_codes_specific(self, mock_request):
        """Test get_spatial_zip_codes with specific zip code."""
        mock_request.return_value = {"ZipCodes": [{"ZipCode": "93624"}]}
        
        with patch.object(self.client.endpoints, 'parse_spatial_zip_codes_response') as mock_parse:
            mock_parse.return_value = []
            
            result = self.client.get_spatial_zip_codes("93624")
            
            assert isinstance(result, list)
            mock_request.assert_called_once()
    
    def test_get_daily_data(self):
        """Test get_daily_data convenience method."""
        with patch.object(self.client, 'get_data') as mock_get_data:
            mock_get_data.return_value = WeatherData()
            
            result = self.client.get_daily_data(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02"
            )
            
            assert isinstance(result, WeatherData)
            mock_get_data.assert_called_once()
    
    def test_get_hourly_data(self):
        """Test get_hourly_data convenience method."""
        with patch.object(self.client, 'get_data') as mock_get_data:
            mock_get_data.return_value = WeatherData()
            
            result = self.client.get_hourly_data(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02"
            )
            
            assert isinstance(result, WeatherData)
            mock_get_data.assert_called_once()
    
    @patch('builtins.open', new_callable=MagicMock)
    @patch('csv.DictWriter')
    def test_export_to_csv(self, mock_csv_writer, mock_open):
        """Test export_to_csv method."""
        # Create mock weather data with proper structure
        weather_data = WeatherData()
        # Create a provider with records
        from python_cimis.models import WeatherProvider
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 6, "Date": "2023-01-01", "DayAirTmpAvg": {"Value": 20.0}},
            {"Station": 2, "Date": "2023-01-02", "DayAirTmpAvg": {"Value": 22.0}}
        ]
        weather_data.providers = [provider]
        
        mock_writer_instance = Mock()
        mock_csv_writer.return_value = mock_writer_instance
        mock_file = MagicMock()
        mock_open.return_value.__enter__.return_value = mock_file
        
        filename = self.client.export_to_csv(weather_data, "test.csv")
        
        assert filename == "test.csv"
        mock_open.assert_called_once()
        mock_writer_instance.writeheader.assert_called_once()
        # Check that writerow was called (at least once for the records)
        assert mock_writer_instance.writerow.call_count >= 1
    
    @patch('builtins.open', new_callable=MagicMock)
    @patch('csv.DictWriter')
    def test_export_to_csv_with_auto_filename(self, mock_csv_writer, mock_open):
        """Test export_to_csv with automatic filename generation."""
        # Create mock weather data with proper structure
        weather_data = WeatherData()
        from python_cimis.models import WeatherProvider
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 6, "Date": "2023-01-01", "DayAirTmpAvg": {"Value": 20.0}}
        ]
        weather_data.providers = [provider]
        
        mock_writer_instance = Mock()
        mock_csv_writer.return_value = mock_writer_instance
        mock_file = MagicMock()
        mock_open.return_value.__enter__.return_value = mock_file
        
        # Mock the auto filename generation
        with patch.object(self.client.filename_generator, 'generate_for_weather_data') as mock_gen:
            mock_gen.return_value = "auto_generated.csv"
            
            filename = self.client.export_to_csv(weather_data)
            
            assert filename == "auto_generated.csv"
            mock_gen.assert_called_once()
    
    def test_get_data_and_export_csv(self):
        """Test get_data_and_export_csv method."""
        with patch.object(self.client, 'get_data') as mock_get_data, \
             patch.object(self.client, 'export_to_csv') as mock_export:
            
            mock_weather_data = WeatherData()
            mock_get_data.return_value = mock_weather_data
            mock_export.return_value = "test.csv"
            
            weather_data, filename = self.client.get_data_and_export_csv(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02",
                filename="test.csv"
            )
            
            assert filename == "test.csv"
            assert weather_data == mock_weather_data
            mock_get_data.assert_called_once()
            mock_export.assert_called_once_with(mock_weather_data, "test.csv")
    
    def test_get_data_and_export_csv_without_filename(self):
        """Test get_data_and_export_csv without specifying filename."""
        with patch.object(self.client, 'get_data') as mock_get_data, \
             patch.object(self.client, 'export_to_csv') as mock_export:
            
            mock_weather_data = WeatherData()
            mock_get_data.return_value = mock_weather_data
            mock_export.return_value = "auto_generated.csv"
            
            weather_data, filename = self.client.get_data_and_export_csv(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02"
            )
            
            assert filename == "auto_generated.csv"
            assert weather_data == mock_weather_data
            mock_get_data.assert_called_once()
            mock_export.assert_called_once_with(mock_weather_data, None)
    
    def test_get_data_with_datetime_objects(self):
        """Test get_data with datetime objects."""
        with patch.object(self.client, '_make_request') as mock_request, \
             patch.object(self.client.endpoints, 'parse_data_response') as mock_parse:
            
            mock_request.return_value = {"test": "data"}
            mock_parse.return_value = WeatherData()
            
            start_datetime = datetime(2023, 1, 1, 10, 30)
            end_datetime = datetime(2023, 1, 2, 15, 45)
            
            self.client.get_data(
                targets=[2],
                start_date=start_datetime,
                end_date=end_datetime
            )
            
            mock_request.assert_called_once()
            call_args = mock_request.call_args
            params = call_args[0][1]
            assert 'startDate' in params
            assert 'endDate' in params
    
    def test_get_data_with_all_parameters(self):
        """Test get_data with all optional parameters."""
        with patch.object(self.client, '_make_request') as mock_request, \
             patch.object(self.client.endpoints, 'parse_data_response') as mock_parse:
            
            mock_request.return_value = {"test": "data"}
            mock_parse.return_value = WeatherData()
            
            self.client.get_data(
                targets=[2, 8],
                start_date="2023-01-01",
                end_date="2023-01-02",
                data_items=["day-air-tmp-avg", "day-eto"],
                unit_of_measure="M",
                prioritize_scs=False
            )
            
            mock_request.assert_called_once()
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_timeout_configuration(self, mock_get_url, mock_get):
        """Test that timeout is properly configured."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = {"test": "data"}
        mock_get.return_value = mock_response
        
        # Test custom timeout
        client = CimisClient(app_key="test-key", timeout=60)
        client._make_request("data", {})
        
        # Check that timeout was passed to requests
        call_args = mock_get.call_args
        assert call_args[1]['timeout'] == 60
    
    def test_session_reuse(self):
        """Test that session is reused between requests."""
        session1 = self.client.session
        session2 = self.client.session
        assert session1 is session2


class TestClientErrorHandling:
    """Test client error handling scenarios."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_timeout_error(self, mock_get_url, mock_get):
        """Test timeout error handling."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_get.side_effect = requests.exceptions.Timeout()
        
        with pytest.raises(CimisConnectionError, match="Request timeout"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_connection_error(self, mock_get_url, mock_get):
        """Test connection error handling."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_get.side_effect = requests.exceptions.ConnectionError("Network error")
        
        with pytest.raises(CimisConnectionError, match="Connection error"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_generic_request_error(self, mock_get_url, mock_get):
        """Test generic request error handling."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_get.side_effect = requests.exceptions.RequestException("Generic error")
        
        with pytest.raises(CimisConnectionError, match="Request error"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_404_with_json_response(self, mock_get_url, mock_get):
        """Test 404 error with JSON response."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_response = Mock()
        mock_response.status_code = 404
        mock_response.json.return_value = {"Message": "Station not found"}
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisAPIError, match="Station not found"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_404_without_json_response(self, mock_get_url, mock_get):
        """Test 404 error without valid JSON response."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_response = Mock()
        mock_response.status_code = 404
        mock_response.json.side_effect = ValueError("No JSON")
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisAPIError, match="Resource not found"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_other_http_errors(self, mock_get_url, mock_get):
        """Test other HTTP errors."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_response = Mock()
        mock_response.status_code = 500
        mock_response.reason = "Internal Server Error"
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisAPIError, match="HTTP 500"):
            self.client._make_request("data", {})
    
    @patch('requests.Session.get')
    @patch('python_cimis.endpoints.CimisEndpoints.get_url')
    def test_invalid_json_response(self, mock_get_url, mock_get):
        """Test invalid JSON response."""
        mock_get_url.return_value = "https://et.water.ca.gov/api/data"
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.side_effect = ValueError("Invalid JSON")
        mock_get.return_value = mock_response
        
        with pytest.raises(CimisDataError, match="Invalid JSON response"):
            self.client._make_request("data", {})


class TestClientCSVParameterSupport:
    """Test CSV parameter support in get_daily_data and get_hourly_data methods."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_daily_data_with_csv_true(self, mock_export, mock_get_data):
        """Test get_daily_data with csv=True parameter."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Mock the CSV export - updated to match new return format
        mock_export.return_value = "auto_generated_daily.csv"
        
        # Call with csv=True
        result = self.client.get_daily_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-02",
            csv=True
        )
        
        # Should return tuple of (weather_data, csv_filename)
        assert isinstance(result, tuple)
        assert len(result) == 2
        weather_data, csv_filename = result
        assert weather_data == mock_weather_data
        assert csv_filename == "auto_generated_daily.csv"
        
        # Verify methods were called correctly
        mock_get_data.assert_called_once_with(
            [2], "2023-01-01", "2023-01-02", [], 'E', True
        )
        mock_export.assert_called_once_with(mock_weather_data, None)
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_daily_data_with_csv_false(self, mock_export, mock_get_data):
        """Test get_daily_data with csv=False (default)."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Call with csv=False (default)
        result = self.client.get_daily_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-02"
        )
        
        # Should return just the weather_data object
        assert result == mock_weather_data
        
        # Verify get_data was called but export_to_csv was not
        mock_get_data.assert_called_once()
        mock_export.assert_not_called()
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_daily_data_with_custom_filename(self, mock_export, mock_get_data):
        """Test get_daily_data with csv=True and custom filename."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Mock the CSV export
        mock_export.return_value = "custom_daily.csv"
        
        # Call with csv=True and custom filename
        result = self.client.get_daily_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-02",
            csv=True,
            filename="custom_daily.csv"
        )
        
        # Should return tuple with custom filename
        weather_data, csv_filename = result
        assert csv_filename == "custom_daily.csv"
        
        # Verify export was called with custom filename
        mock_export.assert_called_once_with(mock_weather_data, "custom_daily.csv")
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_hourly_data_with_csv_true(self, mock_export, mock_get_data):
        """Test get_hourly_data with csv=True parameter."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Mock the CSV export
        mock_export.return_value = "auto_generated_hourly.csv"
        
        # Call with csv=True
        result = self.client.get_hourly_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-01",  # Shorter period for hourly
            csv=True
        )
        
        # Should return tuple of (weather_data, csv_filename)
        assert isinstance(result, tuple)
        assert len(result) == 2
        weather_data, csv_filename = result
        assert weather_data == mock_weather_data
        assert csv_filename == "auto_generated_hourly.csv"
        
        # Verify methods were called correctly (prioritize_scs=False for hourly)
        mock_get_data.assert_called_once_with(
            [2], "2023-01-01", "2023-01-01", [], 'E', prioritize_scs=False
        )
        mock_export.assert_called_once_with(mock_weather_data, None, separate_daily_hourly=False)
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_hourly_data_with_csv_false(self, mock_export, mock_get_data):
        """Test get_hourly_data with csv=False (default)."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Call with csv=False (default)
        result = self.client.get_hourly_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-01"
        )
        
        # Should return just the weather_data object
        assert result == mock_weather_data
        
        # Verify get_data was called but export_to_csv was not
        mock_get_data.assert_called_once()
        mock_export.assert_not_called()
    
    @patch('python_cimis.client.CimisClient.get_data')
    @patch('python_cimis.client.CimisClient.export_to_csv')
    def test_get_hourly_data_with_all_parameters(self, mock_export, mock_get_data):
        """Test get_hourly_data with all parameters including csv=True."""
        # Mock the get_data response
        mock_weather_data = Mock()
        mock_get_data.return_value = mock_weather_data
        
        # Mock the CSV export
        mock_export.return_value = "hourly_detailed.csv"
        
        # Call with all parameters
        result = self.client.get_hourly_data(
            targets=[2, 6],
            start_date="2023-01-01",
            end_date="2023-01-01",
            data_items=["hly-air-tmp", "hly-rel-hum"],
            unit_of_measure="M",
            csv=True,
            filename="hourly_detailed.csv"
        )
        
        # Should return tuple
        weather_data, csv_filename = result
        assert csv_filename == "hourly_detailed.csv"
        
        # Verify all parameters were passed correctly
        mock_get_data.assert_called_once_with(
            [2, 6], "2023-01-01", "2023-01-01", 
            ["hly-air-tmp", "hly-rel-hum"], 'M', prioritize_scs=False
        )
        mock_export.assert_called_once_with(mock_weather_data, "hourly_detailed.csv", separate_daily_hourly=False)


if __name__ == "__main__":
    pytest.main([__file__])
