"""
Comprehensive test suite for the utils module to improve test coverage.
"""

import pytest
import tempfile
import os
from unittest.mock import Mock, patch, MagicMock
from pathlib import Path
from python_cimis import utils
from python_cimis.models import WeatherData, WeatherProvider


class TestFilenameGeneration:
    """Test filename generation utilities."""
    
    def test_generate_filename_for_weather_data(self):
        """Test filename generation for weather data."""
        # Create mock weather data
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = [
            {"Station": 2, "Date": "2023-01-01"},
            {"Station": 6, "Date": "2023-01-02"}
        ]
        weather_data.providers = [provider]
        
        filename = utils.generate_filename_for_weather_data(weather_data)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'cimis' in filename.lower()
    
    def test_generate_filename_for_stations(self):
        """Test filename generation for station data."""
        from python_cimis.models import Station
        
        stations = [
            Station(station_nbr=2, name="Five Points"),
            Station(station_nbr=6, name="Davis")
        ]
        
        filename = utils.generate_filename_for_stations(stations)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'station' in filename.lower()
    
    def test_generate_filename_for_zip_codes(self):
        """Test filename generation for zip code data."""
        zip_codes = ["93624", "95667"]
        
        filename = utils.generate_filename_for_zip_codes(zip_codes)
        
        assert isinstance(filename, str)
        assert filename.endswith('.csv')
        assert 'zip' in filename.lower()
    
    def test_generate_filename_with_custom_base_dir(self):
        """Test filename generation with custom base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            weather_data = WeatherData()
            provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
            provider.records = [{"Station": 2, "Date": "2023-01-01"}]
            weather_data.providers = [provider]
            
            filename = utils.generate_filename_for_weather_data(weather_data, base_dir=temp_dir)
            
            assert temp_dir in filename
            assert filename.endswith('.csv')


class TestDataProcessing:
    """Test data processing utilities."""
    
    def test_process_weather_data_records(self):
        """Test processing of weather data records."""
        raw_records = [
            {
                "Station": 2,
                "Date": "2023-01-01",
                "DayAirTmpAvg": {"Value": 45.2, "Unit": "F", "QC": " "},
                "DayEto": {"Value": 0.05, "Unit": "in", "QC": " "}
            },
            {
                "Station": 6,
                "Date": "2023-01-02",
                "DayAirTmpAvg": {"Value": 42.8, "Unit": "F", "QC": " "},
                "DayEto": {"Value": 0.04, "Unit": "in", "QC": " "}
            }
        ]
        
        processed = utils.process_weather_data_records(raw_records)
        
        assert isinstance(processed, list)
        assert len(processed) == 2
        assert all('station' in record.lower() or 'Station' in str(record) for record in processed)
    
    def test_process_station_data_records(self):
        """Test processing of station data records."""
        raw_stations = [
            {
                "StationNbr": 2,
                "Name": "Five Points",
                "HmsLatitude": "36°20'10N / 36.3360",
                "HmsLongitude": "120°6'47W / -120.1130"
            },
            {
                "StationNbr": 6,
                "Name": "Davis",
                "HmsLatitude": "38°32'8N / 38.5355",
                "HmsLongitude": "121°46'24W / -121.7733"
            }
        ]
        
        processed = utils.process_station_data_records(raw_stations)
        
        assert isinstance(processed, list)
        assert len(processed) == 2
    
    def test_process_empty_records(self):
        """Test processing of empty record lists."""
        assert utils.process_weather_data_records([]) == []
        assert utils.process_station_data_records([]) == []
    
    def test_process_malformed_records(self):
        """Test processing of malformed records."""
        malformed_records = [
            {"invalid": "data"},
            None,
            {"Station": None, "Date": "2023-01-01"}
        ]
        
        # Should handle malformed data gracefully
        processed = utils.process_weather_data_records(malformed_records)
        assert isinstance(processed, list)


class TestDataValidation:
    """Test data validation utilities."""
    
    def test_validate_date_format(self):
        """Test date format validation."""
        # Valid dates
        assert utils.validate_date_format("2023-01-01") == True
        assert utils.validate_date_format("2023-12-31") == True
        
        # Invalid dates
        assert utils.validate_date_format("2023/01/01") == False
        assert utils.validate_date_format("01-01-2023") == False
        assert utils.validate_date_format("invalid") == False
        assert utils.validate_date_format("") == False
        assert utils.validate_date_format(None) == False
    
    def test_validate_station_numbers(self):
        """Test station number validation."""
        # Valid station numbers
        assert utils.validate_station_numbers([2, 6, 127]) == True
        assert utils.validate_station_numbers(["2", "6", "127"]) == True
        
        # Invalid station numbers
        assert utils.validate_station_numbers([]) == False
        assert utils.validate_station_numbers(None) == False
        assert utils.validate_station_numbers(["invalid"]) == False
    
    def test_validate_zip_codes(self):
        """Test zip code validation."""
        # Valid zip codes
        assert utils.validate_zip_codes(["93624", "94503"]) == True
        assert utils.validate_zip_codes(["93624"]) == True
        
        # Invalid zip codes
        assert utils.validate_zip_codes([]) == False
        assert utils.validate_zip_codes(None) == False
        assert utils.validate_zip_codes(["invalid"]) == False
        assert utils.validate_zip_codes(["123"]) == False  # Too short
    
    def test_validate_coordinates(self):
        """Test coordinate validation."""
        # Valid coordinates
        assert utils.validate_coordinates(["lat=39.36,lng=-121.74"]) == True
        assert utils.validate_coordinates(["lat=38.22,lng=-122.82"]) == True
        
        # Invalid coordinates
        assert utils.validate_coordinates([]) == False
        assert utils.validate_coordinates(None) == False
        assert utils.validate_coordinates(["invalid"]) == False
        assert utils.validate_coordinates(["lat=invalid,lng=invalid"]) == False
    
    def test_validate_unit_of_measure(self):
        """Test unit of measure validation."""
        assert utils.validate_unit_of_measure("E") == True
        assert utils.validate_unit_of_measure("M") == True
        assert utils.validate_unit_of_measure("e") == False  # Case sensitive
        assert utils.validate_unit_of_measure("m") == False  # Case sensitive
        assert utils.validate_unit_of_measure("invalid") == False
        assert utils.validate_unit_of_measure(None) == False


class TestDataTransformation:
    """Test data transformation utilities."""
    
    def test_convert_temperature_units(self):
        """Test temperature unit conversion."""
        # Fahrenheit to Celsius
        celsius = utils.convert_temperature_units(68.0, "F", "C")
        assert abs(celsius - 20.0) < 0.1
        
        # Celsius to Fahrenheit
        fahrenheit = utils.convert_temperature_units(20.0, "C", "F")
        assert abs(fahrenheit - 68.0) < 0.1
        
        # Same unit (no conversion)
        same = utils.convert_temperature_units(25.0, "C", "C")
        assert same == 25.0
    
    def test_convert_precipitation_units(self):
        """Test precipitation unit conversion."""
        # Inches to millimeters
        mm = utils.convert_precipitation_units(1.0, "in", "mm")
        assert abs(mm - 25.4) < 0.1
        
        # Millimeters to inches
        inches = utils.convert_precipitation_units(25.4, "mm", "in")
        assert abs(inches - 1.0) < 0.01
    
    def test_normalize_weather_data(self):
        """Test weather data normalization."""
        raw_data = {
            "Station": 2,
            "Date": "2023-01-01",
            "DayAirTmpAvg": {"Value": 68.0, "Unit": "F", "QC": " "},
            "DayPrecip": {"Value": 1.0, "Unit": "in", "QC": " "}
        }
        
        normalized = utils.normalize_weather_data(raw_data, target_unit="M")
        
        assert normalized["DayAirTmpAvg"]["Unit"] == "C"
        assert normalized["DayPrecip"]["Unit"] == "mm"
        assert abs(normalized["DayAirTmpAvg"]["Value"] - 20.0) < 0.1
        assert abs(normalized["DayPrecip"]["Value"] - 25.4) < 0.1


class TestFileOperations:
    """Test file operation utilities."""
    
    def test_ensure_directory_exists(self):
        """Test directory creation utility."""
        with tempfile.TemporaryDirectory() as temp_dir:
            test_path = os.path.join(temp_dir, "subdir", "subsubdir")
            
            utils.ensure_directory_exists(test_path)
            
            assert os.path.exists(test_path)
            assert os.path.isdir(test_path)
    
    def test_get_safe_filename(self):
        """Test safe filename generation."""
        unsafe_name = "test<file>name:with|invalid*chars?.csv"
        safe_name = utils.get_safe_filename(unsafe_name)
        
        assert "<" not in safe_name
        assert ">" not in safe_name
        assert ":" not in safe_name
        assert "|" not in safe_name
        assert "*" not in safe_name
        assert "?" not in safe_name
        assert safe_name.endswith(".csv")
    
    def test_get_unique_filename(self):
        """Test unique filename generation."""
        with tempfile.TemporaryDirectory() as temp_dir:
            base_name = "test.csv"
            base_path = os.path.join(temp_dir, base_name)
            
            # Create the base file
            with open(base_path, 'w') as f:
                f.write("test")
            
            unique_path = utils.get_unique_filename(base_path)
            
            assert unique_path != base_path
            assert not os.path.exists(unique_path)
    
    @patch('builtins.open', new_callable=MagicMock)
    @patch('csv.DictWriter')
    def test_write_csv_file(self, mock_csv_writer, mock_open):
        """Test CSV file writing utility."""
        data = [
            {"Station": 2, "Date": "2023-01-01", "Temperature": 68.0},
            {"Station": 6, "Date": "2023-01-02", "Temperature": 70.0}
        ]
        
        mock_writer_instance = Mock()
        mock_csv_writer.return_value = mock_writer_instance
        mock_file = MagicMock()
        mock_open.return_value.__enter__.return_value = mock_file
        
        utils.write_csv_file("test.csv", data)
        
        mock_open.assert_called_once()
        mock_writer_instance.writeheader.assert_called_once()
        mock_writer_instance.writerows.assert_called_once_with(data)


class TestStringOperations:
    """Test string operation utilities."""
    
    def test_sanitize_string_for_filename(self):
        """Test string sanitization for filenames."""
        test_string = "Test Station/Name:With|Special*Chars?.txt"
        sanitized = utils.sanitize_string_for_filename(test_string)
        
        assert "/" not in sanitized
        assert ":" not in sanitized
        assert "|" not in sanitized
        assert "*" not in sanitized
        assert "?" not in sanitized
    
    def test_format_station_name(self):
        """Test station name formatting."""
        assert utils.format_station_name("FIVE POINTS") == "Five Points"
        assert utils.format_station_name("davis weather station") == "Davis Weather Station"
        assert utils.format_station_name("") == "Unknown"
        assert utils.format_station_name(None) == "Unknown"
    
    def test_parse_coordinate_string(self):
        """Test coordinate string parsing."""
        coord_str = "lat=39.36,lng=-121.74"
        lat, lng = utils.parse_coordinate_string(coord_str)
        
        assert lat == 39.36
        assert lng == -121.74
    
    def test_parse_coordinate_string_invalid(self):
        """Test coordinate string parsing with invalid input."""
        with pytest.raises(ValueError):
            utils.parse_coordinate_string("invalid")
        
        with pytest.raises(ValueError):
            utils.parse_coordinate_string("lat=invalid,lng=invalid")


class TestDateOperations:
    """Test date operation utilities."""
    
    def test_format_date_for_api(self):
        """Test date formatting for API calls."""
        from datetime import date, datetime
        
        # Test with date object
        test_date = date(2023, 1, 15)
        formatted = utils.format_date_for_api(test_date)
        assert formatted == "2023-01-15"
        
        # Test with datetime object
        test_datetime = datetime(2023, 1, 15, 10, 30, 45)
        formatted = utils.format_date_for_api(test_datetime)
        assert formatted == "2023-01-15"
        
        # Test with string
        formatted = utils.format_date_for_api("2023-01-15")
        assert formatted == "2023-01-15"
    
    def test_parse_api_date(self):
        """Test API date string parsing."""
        parsed = utils.parse_api_date("2023-01-15")
        
        assert parsed.year == 2023
        assert parsed.month == 1
        assert parsed.day == 15
    
    def test_get_date_range_days(self):
        """Test date range calculation in days."""
        days = utils.get_date_range_days("2023-01-01", "2023-01-10")
        assert days == 9  # 9 days between (exclusive end)
        
        days = utils.get_date_range_days("2023-01-01", "2023-01-01")
        assert days == 0


class TestErrorHandling:
    """Test error handling utilities."""
    
    def test_handle_api_error_response(self):
        """Test API error response handling."""
        error_response = {
            "Errors": [
                {
                    "ErrorCode": "ERR1006",
                    "ErrorMessage": "Invalid application key"
                }
            ]
        }
        
        error_info = utils.handle_api_error_response(error_response)
        
        assert error_info["code"] == "ERR1006"
        assert "Invalid application key" in error_info["message"]
    
    def test_handle_api_error_response_no_errors(self):
        """Test API error response handling with no errors."""
        response = {"Data": {"Records": []}}
        
        error_info = utils.handle_api_error_response(response)
        
        assert error_info is None
    
    def test_format_error_message(self):
        """Test error message formatting."""
        message = utils.format_error_message("ERR1006", "Invalid application key", 403)
        
        assert "ERR1006" in message
        assert "Invalid application key" in message
        assert "403" in message


class TestMiscellaneousUtilities:
    """Test miscellaneous utility functions."""
    
    def test_get_library_version(self):
        """Test library version retrieval."""
        version = utils.get_library_version()
        
        assert isinstance(version, str)
        assert len(version) > 0
    
    def test_get_default_user_agent(self):
        """Test default user agent string generation."""
        user_agent = utils.get_default_user_agent()
        
        assert isinstance(user_agent, str)
        assert "python-cimis" in user_agent.lower()
    
    def test_is_debug_mode(self):
        """Test debug mode detection."""
        # Should return a boolean
        debug_mode = utils.is_debug_mode()
        assert isinstance(debug_mode, bool)
    
    @patch.dict(os.environ, {'CIMIS_DEBUG': '1'})
    def test_is_debug_mode_enabled(self):
        """Test debug mode when enabled via environment variable."""
        debug_mode = utils.is_debug_mode()
        assert debug_mode == True
    
    def test_truncate_string(self):
        """Test string truncation utility."""
        long_string = "A" * 100
        truncated = utils.truncate_string(long_string, 50)
        
        assert len(truncated) <= 50
        assert truncated.endswith("...")
    
    def test_merge_dictionaries(self):
        """Test dictionary merging utility."""
        dict1 = {"a": 1, "b": 2}
        dict2 = {"b": 3, "c": 4}
        
        merged = utils.merge_dictionaries(dict1, dict2)
        
        assert merged["a"] == 1
        assert merged["b"] == 3  # dict2 should override dict1
        assert merged["c"] == 4


class TestEdgeCasesAndErrorConditions:
    """Test edge cases and error conditions."""
    
    def test_handle_none_inputs(self):
        """Test handling of None inputs."""
        assert utils.validate_date_format(None) == False
        assert utils.validate_station_numbers(None) == False
        assert utils.validate_zip_codes(None) == False
        assert utils.validate_coordinates(None) == False
        assert utils.validate_unit_of_measure(None) == False
        assert utils.format_station_name(None) == "Unknown"
    
    def test_handle_empty_inputs(self):
        """Test handling of empty inputs."""
        assert utils.validate_station_numbers([]) == False
        assert utils.validate_zip_codes([]) == False
        assert utils.validate_coordinates([]) == False
        assert utils.process_weather_data_records([]) == []
        assert utils.process_station_data_records([]) == []
    
    def test_handle_invalid_types(self):
        """Test handling of invalid input types."""
        # These should not raise exceptions but return sensible defaults
        try:
            utils.validate_date_format(123)  # Not a string
            utils.validate_station_numbers("not a list")  # Not a list
            utils.format_station_name(123)  # Not a string
        except Exception as e:
            pytest.fail(f"Function should handle invalid types gracefully: {e}")
    
    def test_large_data_handling(self):
        """Test handling of large datasets."""
        # Create a large dataset
        large_records = []
        for i in range(1000):
            large_records.append({
                "Station": i % 10 + 1,
                "Date": f"2023-01-{(i % 30) + 1:02d}",
                "DayAirTmpAvg": {"Value": 60.0 + i % 20, "Unit": "F", "QC": " "}
            })
        
        processed = utils.process_weather_data_records(large_records)
        
        # Should handle large datasets without issues
        assert isinstance(processed, list)
        assert len(processed) <= len(large_records)  # Could be filtered
    
    def test_memory_efficiency(self):
        """Test memory efficiency with repeated operations."""
        # This test ensures utilities don't have memory leaks
        for _ in range(100):
            utils.validate_date_format("2023-01-01")
            utils.sanitize_string_for_filename("test/file:name.csv")
            utils.format_station_name("TEST STATION")
        
        # If we get here without memory issues, test passes
        assert True
