"""
Comprehensive test suite for python_cimis.utils module.
Tests filename generation utilities and related functions.
"""

import pytest
import tempfile
import os
from pathlib import Path
from unittest.mock import Mock, patch, MagicMock
from python_cimis.utils import (
    FilenameGenerator, 
    generate_weather_filename,
    generate_stations_filename, 
    generate_zip_codes_filename,
    generate_custom_filename
)
from python_cimis.models import WeatherData, WeatherProvider, Station


class TestFilenameGenerator:
    """Test the FilenameGenerator class."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_init_default(self):
        """Test default initialization."""
        generator = FilenameGenerator()
        assert generator.base_directory == Path.cwd()
    
    def test_init_with_directory(self):
        """Test initialization with custom directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            generator = FilenameGenerator(temp_dir)
            assert generator.base_directory == Path(temp_dir)
    
    def test_set_base_directory(self):
        """Test setting base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            self.generator.set_base_directory(temp_dir)
            assert self.generator.base_directory == Path(temp_dir)


class TestWeatherFilenameGeneration:
    """Test weather data filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_generate_weather_filename_basic(self):
        """Test basic weather filename generation."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        # Create mock record with station attribute
        record = Mock()
        record.station = 2
        record.date = "2023-01-01"
        provider.records = [record]
        weather_data.providers = [provider]
        
        filename = self.generator.generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
        assert 'Station2' in filename
    
    def test_generate_weather_filename_empty(self):
        """Test weather filename generation with empty data."""
        weather_data = WeatherData()
        weather_data.providers = []
        
        filename = self.generator.generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename


class TestStationsFilenameGeneration:
    """Test station filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_generate_stations_filename_single(self):
        """Test filename generation for single station."""
        station = Station(station_nbr=2, name="Five Points", city="Five Points")
        filename = self.generator.generate_stations_filename([station])
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
        assert 'FivePoints' in filename
    
    def test_generate_stations_filename_multiple(self):
        """Test filename generation for multiple stations."""
        stations = [
            Station(station_nbr=2, name="Five Points", city="Five Points"),
            Station(station_nbr=6, name="Davis", city="Davis")
        ]
        filename = self.generator.generate_stations_filename(stations)
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_generate_stations_filename_empty(self):
        """Test filename generation for empty station list."""
        filename = self.generator.generate_stations_filename([])
        assert filename.endswith('.csv')
        assert 'all_stations' in filename


class TestZipCodesFilenameGeneration:
    """Test zip code filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_generate_zip_codes_filename_single(self):
        """Test filename generation for single zip code."""
        filename = self.generator.generate_zip_codes_filename(["95363"])
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
        assert '95363' in filename
    
    def test_generate_zip_codes_filename_multiple(self):
        """Test filename generation for multiple zip codes."""
        zip_codes = ["95363", "95618", "95616"]
        filename = self.generator.generate_zip_codes_filename(zip_codes)
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
    
    def test_generate_zip_codes_filename_empty(self):
        """Test filename generation for empty zip code list."""
        filename = self.generator.generate_zip_codes_filename([])
        assert filename.endswith('.csv')
        assert 'all_zipcodes' in filename


class TestCustomFilenameGeneration:
    """Test custom filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_generate_custom_filename_basic(self):
        """Test basic custom filename generation."""
        filename = self.generator.generate_custom_filename("weather")
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename
    
    def test_generate_custom_filename_with_identifiers(self):
        """Test custom filename with identifiers."""
        filename = self.generator.generate_custom_filename("weather", ["2", "6"])
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename
    
    def test_generate_custom_filename_with_date_range(self):
        """Test custom filename with date range."""
        filename = self.generator.generate_custom_filename("weather", date_range="20230101_20230131")
        assert filename.endswith('.csv')
        assert '20230101_20230131' in filename


class TestHelperMethods:
    """Test private helper methods."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_sanitize_name(self):
        """Test name sanitization."""
        result = self.generator._sanitize_name("Five Points")
        assert result == "FivePoints"
        
        result = self.generator._sanitize_name("Special!@#Characters")
        assert result == "SpecialCharacters"
    
    def test_sanitize_filename(self):
        """Test filename sanitization."""
        result = self.generator._sanitize_filename("file<>name.csv")
        assert '<' not in result
        assert '>' not in result
        assert result.endswith('.csv')
    
    def test_format_station_names(self):
        """Test station name formatting."""
        result = self.generator._format_station_names(["Station1", "Station2"])
        assert "Station1" in result
    
    def test_format_date_range(self):
        """Test date range formatting."""
        result = self.generator._format_date_range(["2023-01-01", "2023-01-31"])
        assert "20230101" in result


class TestModuleFunctions:
    """Test module-level convenience functions."""
    
    def test_generate_weather_filename_function(self):
        """Test the module-level generate_weather_filename function."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        record = Mock()
        record.station = 2
        record.date = "2023-01-01"
        provider.records = [record]
        weather_data.providers = [provider]
        
        filename = generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_stations_filename_function(self):
        """Test the module-level generate_stations_filename function."""
        station = Station(station_nbr=2, name="Five Points", city="Five Points")
        filename = generate_stations_filename([station])
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_generate_zip_codes_filename_function(self):
        """Test the module-level generate_zip_codes_filename function."""
        filename = generate_zip_codes_filename(["95363"])
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
    
    def test_generate_custom_filename_function(self):
        """Test the module-level generate_custom_filename function."""
        filename = generate_custom_filename("weather")
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename


class TestEdgeCasesAndErrorHandling:
    """Test edge cases and error conditions."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_weather_data_no_providers(self):
        """Test weather data with no providers."""
        weather_data = WeatherData()
        weather_data.providers = []
        filename = self.generator.generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
    
    def test_weather_data_empty_records(self):
        """Test weather data with empty records."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = []
        weather_data.providers = [provider]
        
        filename = self.generator.generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
    
    def test_long_filename_truncation(self):
        """Test that long filenames are truncated appropriately."""
        long_name = "very_long_station_name_" * 20
        result = self.generator._sanitize_filename(long_name + ".csv")
        assert len(result) <= 200
        assert result.endswith('.csv')
    
    def test_empty_inputs_handling(self):
        """Test handling of empty inputs."""
        # Empty station names
        result = self.generator._format_station_names([])
        assert result == "unknown"
        
        # Empty date ranges
        result = self.generator._format_date_range([])
        assert "unknown" in result
    
    def test_invalid_characters_removal(self):
        """Test removal of invalid filename characters."""
        invalid_name = 'file<>:"|?*name'
        result = self.generator._sanitize_filename(invalid_name + ".csv")
        for char in '<>:"|?*':
            assert char not in result


class TestPerformanceAndMemory:
    """Test performance and memory considerations."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = FilenameGenerator()
    
    def test_large_station_lists(self):
        """Test with large numbers of stations."""
        # Create many stations
        stations = []
        for i in range(100):
            stations.append(Station(station_nbr=i, name=f"Station{i}", city=f"City{i}"))
        
        filename = self.generator.generate_stations_filename(stations)
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_repeated_filename_generation(self):
        """Test repeated filename generation doesn't cause issues."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        record = Mock()
        record.station = 2
        record.date = "2023-01-01"
        provider.records = [record]
        weather_data.providers = [provider]
        
        # Generate many filenames
        filenames = []
        for _ in range(50):
            filename = self.generator.generate_weather_filename(weather_data)
            filenames.append(filename)
        
        # All should be valid
        for filename in filenames:
            assert filename.endswith('.csv')
            assert 'cimis_weather_data' in filename
