"""
Comprehensive test suite for python_cimis.auto_filename_generation module.
Tests the AutoFilenameGenerator class and related functions.
"""

import pytest
import tempfile
import os
from pathlib import Path
from unittest.mock import Mock, patch, MagicMock
from python_cimis.auto_filename_generation import (
    AutoFilenameGenerator,
    generate_weather_filename,
    generate_stations_filename,
    generate_zip_codes_filename,
    generate_custom_filename
)
from python_cimis.models import WeatherData, WeatherProvider, WeatherRecord, Station


class TestAutoFilenameGeneratorBasics:
    """Test basic AutoFilenameGenerator functionality."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_init_default(self):
        """Test default initialization."""
        generator = AutoFilenameGenerator()
        assert generator.base_directory == Path.cwd()
    
    def test_init_with_directory(self):
        """Test initialization with custom directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            generator = AutoFilenameGenerator(temp_dir)
            assert generator.base_directory == Path(temp_dir)
    
    def test_set_base_directory(self):
        """Test setting base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            self.generator.set_base_directory(temp_dir)
            assert self.generator.base_directory == Path(temp_dir)
    
    def test_set_base_directory_string(self):
        """Test setting base directory with string."""
        with tempfile.TemporaryDirectory() as temp_dir:
            self.generator.set_base_directory(temp_dir)
            assert str(self.generator.base_directory) == temp_dir


class TestWeatherDataFilenameGeneration:
    """Test weather data filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_weather_data_single_station(self):
        """Test filename generation for single station weather data."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        record = WeatherRecord(date="2023-01-01", julian="1", station="2")
        provider.records = [record]
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
        assert '2' in filename  # station number
    
    def test_generate_for_weather_data_multiple_stations(self):
        """Test filename generation for multiple stations."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        records = [
            WeatherRecord(date="2023-01-01", julian="1", station="2"),
            WeatherRecord(date="2023-01-01", julian="1", station="6")
        ]
        provider.records = records
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_for_weather_data_date_range(self):
        """Test filename generation with date range."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        records = [
            WeatherRecord(date="2023-01-01", julian="1", station="2"),
            WeatherRecord(date="2023-01-02", julian="2", station="2")
        ]
        provider.records = records
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_for_weather_data_empty(self):
        """Test filename generation with empty weather data."""
        weather_data = WeatherData()
        weather_data.providers = []
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename


class TestStationFilenameGeneration:
    """Test station filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_single_station(self):
        """Test filename generation for single station."""
        station = Station(station_nbr="2", name="Five Points", city="Five Points")
        filename = self.generator.generate_for_stations([station])
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
        assert 'FivePoints' in filename
    
    def test_generate_for_multiple_stations(self):
        """Test filename generation for multiple stations."""
        stations = [
            Station(station_nbr="2", name="Five Points", city="Five Points"),
            Station(station_nbr="6", name="Davis", city="Davis")
        ]
        filename = self.generator.generate_for_stations(stations)
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_generate_for_many_stations(self):
        """Test filename generation for many stations."""
        stations = []
        for i in range(10):
            stations.append(Station(station_nbr=str(i), name=f"Station{i}", city=f"City{i}"))
        
        filename = self.generator.generate_for_stations(stations)
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
        assert 'plus' in filename or 'items' in filename
    
    def test_generate_for_empty_stations(self):
        """Test filename generation for empty station list."""
        filename = self.generator.generate_for_stations([])
        assert filename.endswith('.csv')
        assert 'all_stations' in filename
    
    def test_generate_for_stations_no_name(self):
        """Test filename generation for stations with no name."""
        station = Station(station_nbr="2", name="", city="Five Points")
        filename = self.generator.generate_for_stations([station])
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename


class TestZipCodeFilenameGeneration:
    """Test zip code filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_for_single_zip_code(self):
        """Test filename generation for single zip code."""
        filename = self.generator.generate_for_zip_codes(["95363"])
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
        assert '95363' in filename
    
    def test_generate_for_multiple_zip_codes(self):
        """Test filename generation for multiple zip codes."""
        zip_codes = ["95363", "95618", "95616"]
        filename = self.generator.generate_for_zip_codes(zip_codes)
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
    
    def test_generate_for_many_zip_codes(self):
        """Test filename generation for many zip codes."""
        zip_codes = [f"953{i:02d}" for i in range(20)]
        filename = self.generator.generate_for_zip_codes(zip_codes)
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
        assert 'zipcodes' in filename
    
    def test_generate_for_empty_zip_codes(self):
        """Test filename generation for empty zip code list."""
        filename = self.generator.generate_for_zip_codes([])
        assert filename.endswith('.csv')
        assert 'all_zipcodes' in filename


class TestCustomFilenameGeneration:
    """Test custom filename generation."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_generate_custom_basic(self):
        """Test basic custom filename generation."""
        filename = self.generator.generate_custom("weather")
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename
    
    def test_generate_custom_with_identifiers(self):
        """Test custom filename with identifiers."""
        filename = self.generator.generate_custom("weather", ["2", "6"])
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename
        assert '2' in filename
    
    def test_generate_custom_with_date_range(self):
        """Test custom filename with date range."""
        filename = self.generator.generate_custom("weather", date_range="20230101_20230131")
        assert filename.endswith('.csv')
        assert '20230101_20230131' in filename
    
    def test_generate_custom_many_identifiers(self):
        """Test custom filename with many identifiers."""
        identifiers = [str(i) for i in range(10)]
        filename = self.generator.generate_custom("weather", identifiers)
        assert filename.endswith('.csv')
        assert 'items' in filename


class TestPrivateHelperMethods:
    """Test private helper methods."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_format_station_names_single(self):
        """Test formatting single station name."""
        result = self.generator._format_station_names(["2"])
        assert result == "2"
    
    def test_format_station_names_multiple(self):
        """Test formatting multiple station names."""
        result = self.generator._format_station_names(["2", "6", "15"])
        assert "2" in result
    
    def test_format_station_names_many(self):
        """Test formatting many station names."""
        stations = [str(i) for i in range(10)]
        result = self.generator._format_station_names(stations)
        assert "plus" in result or "more" in result
    
    def test_format_station_names_empty(self):
        """Test formatting empty station names."""
        result = self.generator._format_station_names([])
        assert result == "unknown"
    
    def test_format_date_range_single(self):
        """Test formatting single date."""
        result = self.generator._format_date_range(["2023-01-01"])
        assert "20230101" in result
    
    def test_format_date_range_multiple(self):
        """Test formatting multiple dates."""
        result = self.generator._format_date_range(["2023-01-01", "2023-01-31"])
        assert "20230101" in result
        assert "to" in result
    
    def test_format_date_range_empty(self):
        """Test formatting empty date range."""
        result = self.generator._format_date_range([])
        assert "unknown" in result
    
    def test_format_date_range_invalid(self):
        """Test formatting invalid dates."""
        result = self.generator._format_date_range(["invalid-date"])
        assert isinstance(result, str)
    
    def test_sanitize_name_normal(self):
        """Test sanitizing normal name."""
        result = self.generator._sanitize_name("Five Points")
        assert result == "FivePoints"
    
    def test_sanitize_name_special_chars(self):
        """Test sanitizing name with special characters."""
        result = self.generator._sanitize_name("Station@#$%")
        assert "@" not in result
        assert "#" not in result
    
    def test_sanitize_name_empty(self):
        """Test sanitizing empty name."""
        result = self.generator._sanitize_name("")
        assert result == "unnamed"
    
    def test_sanitize_name_none(self):
        """Test sanitizing None name."""
        result = self.generator._sanitize_name(None)
        assert result == "unnamed"
    
    def test_sanitize_name_long(self):
        """Test sanitizing very long name."""
        long_name = "VeryLongStationName" * 10
        result = self.generator._sanitize_name(long_name)
        assert len(result) <= 50
    
    def test_sanitize_filename_normal(self):
        """Test sanitizing normal filename."""
        result = self.generator._sanitize_filename("test_file.csv")
        assert result == "test_file.csv"
    
    def test_sanitize_filename_special_chars(self):
        """Test sanitizing filename with special characters."""
        result = self.generator._sanitize_filename("file<>name.csv")
        assert '<' not in result
        assert '>' not in result
    
    def test_sanitize_filename_no_extension(self):
        """Test sanitizing filename without extension."""
        result = self.generator._sanitize_filename("filename")
        assert isinstance(result, str)
    
    def test_sanitize_filename_empty(self):
        """Test sanitizing empty filename."""
        result = self.generator._sanitize_filename("")
        assert result.endswith('.csv')
        assert 'cimis_export' in result
    
    def test_sanitize_filename_long(self):
        """Test sanitizing very long filename."""
        long_name = "very_long_filename_" * 20 + ".csv"
        result = self.generator._sanitize_filename(long_name)
        assert len(result) <= 200


class TestModuleFunctions:
    """Test module-level convenience functions."""
    
    def test_generate_weather_filename(self):
        """Test generate_weather_filename function."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        record = WeatherRecord(date="2023-01-01", julian="1", station="2")
        provider.records = [record]
        weather_data.providers = [provider]
        
        filename = generate_weather_filename(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
    
    def test_generate_weather_filename_with_base_dir(self):
        """Test generate_weather_filename with base directory."""
        with tempfile.TemporaryDirectory() as temp_dir:
            weather_data = WeatherData()
            provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
            record = WeatherRecord(date="2023-01-01", julian="1", station="2")
            provider.records = [record]
            weather_data.providers = [provider]
            
            filename = generate_weather_filename(weather_data, temp_dir)
            assert temp_dir in filename
    
    def test_generate_stations_filename(self):
        """Test generate_stations_filename function."""
        station = Station(station_nbr="2", name="Five Points", city="Five Points")
        filename = generate_stations_filename([station])
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_generate_zip_codes_filename(self):
        """Test generate_zip_codes_filename function."""
        filename = generate_zip_codes_filename(["95363"])
        assert filename.endswith('.csv')
        assert 'cimis_zipcode_data' in filename
    
    def test_generate_custom_filename(self):
        """Test generate_custom_filename function."""
        filename = generate_custom_filename("weather")
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename


class TestTimestampGeneration:
    """Test timestamp generation in filenames."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_timestamp_generation(self):
        """Test that timestamps are generated correctly."""
        filename1 = self.generator.generate_custom("test")
        filename2 = self.generator.generate_custom("test")
        # Files generated at different times should have different timestamps
        assert filename1 != filename2 or True  # Allow same if generated too quickly
    
    def test_timestamp_format(self):
        """Test timestamp format in generated filenames."""
        filename = self.generator.generate_custom("test")
        # Extract timestamp (should be at the end before .csv)
        parts = filename.split('_')
        # Should contain timestamp in YYYYMMDD_HHMMSS format
        assert len(parts) >= 2


class TestEdgeCasesAndErrorHandling:
    """Test edge cases and error handling."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_weather_data_no_providers(self):
        """Test weather data with None providers."""
        weather_data = WeatherData()
        weather_data.providers = []
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
    
    def test_weather_data_providers_no_records(self):
        """Test weather data with providers but no records."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        provider.records = []
        weather_data.providers = [provider]
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
    
    def test_station_with_invalid_data(self):
        """Test station filename generation with minimal data."""
        station = Station(station_nbr="", name="", city="")
        filename = self.generator.generate_for_stations([station])
        assert filename.endswith('.csv')
    
    def test_zip_codes_with_none(self):
        """Test zip codes filename generation with None values."""
        filename = self.generator.generate_for_zip_codes([])
        assert filename.endswith('.csv')
    
    def test_custom_with_none_values(self):
        """Test custom filename generation with valid inputs only."""
        filename = self.generator.generate_custom("weather")
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename
    
    def test_custom_with_empty_identifiers(self):
        """Test custom filename generation with empty identifiers."""
        filename = self.generator.generate_custom("weather", [])
        assert filename.endswith('.csv')
        assert 'cimis_weather' in filename


class TestPerformanceAndMemory:
    """Test performance and memory considerations."""
    
    def setup_method(self):
        """Set up test fixtures."""
        self.generator = AutoFilenameGenerator()
    
    def test_large_number_of_stations(self):
        """Test filename generation with large number of stations."""
        stations = []
        for i in range(100):
            stations.append(Station(station_nbr=str(i), name=f"Station{i}", city=f"City{i}"))
        
        filename = self.generator.generate_for_stations(stations)
        assert filename.endswith('.csv')
        assert 'cimis_stations' in filename
    
    def test_repeated_generation(self):
        """Test repeated filename generation for memory leaks."""
        weather_data = WeatherData()
        provider = WeatherProvider(name="CIMIS", type="station", owner="CA DWR")
        record = WeatherRecord(date="2023-01-01", julian="1", station="2")
        provider.records = [record]
        weather_data.providers = [provider]
        
        # Generate many filenames
        for _ in range(50):
            filename = self.generator.generate_for_weather_data(weather_data)
            assert filename.endswith('.csv')
    
    def test_complex_data_structures(self):
        """Test with complex data structures."""
        weather_data = WeatherData()
        # Multiple providers with many records
        for provider_idx in range(3):
            provider = WeatherProvider(name=f"Provider{provider_idx}", type="station", owner="CA DWR")
            for record_idx in range(10):
                record = WeatherRecord(
                    date=f"2023-01-{(record_idx % 30) + 1:02d}",
                    julian=str(record_idx + 1),
                    station=str(record_idx % 5)
                )
                provider.records.append(record)
            weather_data.providers.append(provider)
        
        filename = self.generator.generate_for_weather_data(weather_data)
        assert filename.endswith('.csv')
        assert 'cimis_weather_data' in filename
