"""
Test suite for the Python CIMIS endpoints module.
"""

import pytest
from datetime import date, datetime
from python_cimis.endpoints import CimisEndpoints


class TestCimisEndpoints:
    """Test cases for CimisEndpoints class."""
    
    def test_base_url(self):
        """Test that base URL is correct."""
        assert CimisEndpoints.BASE_URL == "https://et.water.ca.gov/api"
    
    def test_endpoints_dict(self):
        """Test that endpoints dictionary contains expected keys."""
        expected_endpoints = [
            'data', 'station', 'stations', 'zip_code', 'zip_codes',
            'spatial_zip_code', 'spatial_zip_codes'
        ]
        
        for endpoint in expected_endpoints:
            assert endpoint in CimisEndpoints.ENDPOINTS
    
    def test_default_daily_data_items(self):
        """Test that default daily data items are defined."""
        items = CimisEndpoints.DEFAULT_DAILY_DATA_ITEMS
        assert isinstance(items, list)
        assert len(items) > 0
        assert "day-air-tmp-avg" in items
        assert "day-eto" in items
        assert "day-precip" in items
    
    def test_default_hourly_data_items(self):
        """Test that default hourly data items are defined."""
        items = CimisEndpoints.DEFAULT_HOURLY_DATA_ITEMS
        assert isinstance(items, list)
        assert len(items) > 0
        assert "hly-air-tmp" in items
        assert "hly-eto" in items
        assert "hly-precip" in items
    
    def test_error_codes(self):
        """Test that error codes mapping is defined."""
        error_codes = CimisEndpoints.ERROR_CODES
        assert isinstance(error_codes, dict)
        assert "ERR1006" in error_codes
        assert "ERR1019" in error_codes
        assert error_codes["ERR1006"] == "INVALID APP KEY"


class TestGetUrl:
    """Test cases for get_url method."""
    
    def test_get_url_simple_endpoint(self):
        """Test getting URL for simple endpoint."""
        url = CimisEndpoints.get_url('data')
        expected = "https://et.water.ca.gov/api/data"
        assert url == expected
    
    def test_get_url_with_parameters(self):
        """Test getting URL with parameters."""
        url = CimisEndpoints.get_url('station', station_id="2")
        expected = "https://et.water.ca.gov/api/station/2"
        assert url == expected
    
    def test_get_url_zip_code(self):
        """Test getting URL for zip code endpoint."""
        url = CimisEndpoints.get_url('zip_code', zip_code="93624")
        expected = "https://et.water.ca.gov/api/stationzipcode/93624"
        assert url == expected
    
    def test_get_url_spatial_zip_code(self):
        """Test getting URL for spatial zip code endpoint."""
        url = CimisEndpoints.get_url('spatial_zip_code', zip_code="93624")
        expected = "https://et.water.ca.gov/api/spatialzipcode/93624"
        assert url == expected
    
    def test_get_url_invalid_endpoint(self):
        """Test getting URL for invalid endpoint."""
        with pytest.raises(ValueError):
            CimisEndpoints.get_url('invalid_endpoint')


class TestPrepareDataRequestParams:
    """Test cases for prepare_data_request_params method."""
    
    def test_prepare_basic_params(self):
        """Test preparing basic data request parameters."""
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2", "8"],
            start_date="2023-01-01",
            end_date="2023-01-02"
        )
        
        assert params["appKey"] == "test-key"
        assert params["targets"] == "2,8"
        assert params["startDate"] == "2023-01-01"
        assert params["endDate"] == "2023-01-02"
        assert params["unitOfMeasure"] == "E"
        assert params["prioritizeSCS"] == "Y"
    
    def test_prepare_params_with_data_items(self):
        """Test preparing parameters with specific data items."""
        data_items = ["day-air-tmp-avg", "day-precip"]
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2"],
            start_date="2023-01-01",
            end_date="2023-01-02",
            data_items=data_items
        )
        
        assert params["dataItems"] == "day-air-tmp-avg,day-precip"
    
    def test_prepare_params_metric_units(self):
        """Test preparing parameters with metric units."""
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2"],
            start_date="2023-01-01",
            end_date="2023-01-02",
            unit_of_measure="M"
        )
        
        assert params["unitOfMeasure"] == "M"
    
    def test_prepare_params_no_prioritize_scs(self):
        """Test preparing parameters without prioritizing SCS."""
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2"],
            start_date="2023-01-01",
            end_date="2023-01-02",
            prioritize_scs=False
        )
        
        assert params["prioritizeSCS"] == "N"
    
    def test_prepare_params_with_date_objects(self):
        """Test preparing parameters with date objects."""
        start_date = date(2023, 1, 1)
        end_date = date(2023, 1, 2)
        
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2"],
            start_date=start_date,
            end_date=end_date
        )
        
        assert params["startDate"] == "2023-01-01"
        assert params["endDate"] == "2023-01-02"
    
    def test_prepare_params_with_datetime_objects(self):
        """Test preparing parameters with datetime objects."""
        start_date = datetime(2023, 1, 1, 10, 30)
        end_date = datetime(2023, 1, 2, 15, 45)
        
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=["2"],
            start_date=start_date,
            end_date=end_date
        )
        
        assert params["startDate"] == "2023-01-01"
        assert params["endDate"] == "2023-01-02"
    
    def test_prepare_params_coordinates(self):
        """Test preparing parameters with coordinate targets."""
        coord_targets = ["lat=39.36,lng=-121.74", "lat=38.22,lng=-122.82"]
        
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=coord_targets,
            start_date="2023-01-01",
            end_date="2023-01-02"
        )
        
        assert params["targets"] == "lat=39.36,lng=-121.74;lat=38.22,lng=-122.82"
    
    def test_prepare_params_addresses(self):
        """Test preparing parameters with address targets."""
        addr_targets = ["addr-name=Test,addr=123 Main St", "addr-name=Test2,addr=456 Oak Ave"]
        
        params = CimisEndpoints.prepare_data_request_params(
            app_key="test-key",
            targets=addr_targets,
            start_date="2023-01-01",
            end_date="2023-01-02"
        )
        
        assert params["targets"] == "addr-name=Test,addr=123 Main St;addr-name=Test2,addr=456 Oak Ave"


class TestValidationMethods:
    """Test cases for validation methods."""
    
    def test_validate_unit_of_measure_valid(self):
        """Test validation of valid units of measure."""
        assert CimisEndpoints.validate_unit_of_measure("E") is True
        assert CimisEndpoints.validate_unit_of_measure("M") is True
    
    def test_validate_unit_of_measure_invalid(self):
        """Test validation of invalid units of measure."""
        assert CimisEndpoints.validate_unit_of_measure("X") is False
        assert CimisEndpoints.validate_unit_of_measure("") is False
        assert CimisEndpoints.validate_unit_of_measure("metric") is False
    
    def test_validate_date_format_valid(self):
        """Test validation of valid date formats."""
        assert CimisEndpoints.validate_date_format("2023-01-01") is True
        assert CimisEndpoints.validate_date_format("2023-12-31") is True
    
    def test_validate_date_format_invalid(self):
        """Test validation of invalid date formats."""
        assert CimisEndpoints.validate_date_format("01-01-2023") is False
        assert CimisEndpoints.validate_date_format("2023/01/01") is False
        assert CimisEndpoints.validate_date_format("invalid-date") is False
        assert CimisEndpoints.validate_date_format("") is False


class TestUtilityMethods:
    """Test cases for utility methods."""
    
    def test_is_coordinate_list_valid(self):
        """Test coordinate list detection with valid coordinates."""
        coord_list = ["lat=39.36,lng=-121.74", "lat=38.22,lng=-122.82"]
        assert CimisEndpoints._is_coordinate_list(coord_list) is True
    
    def test_is_coordinate_list_invalid(self):
        """Test coordinate list detection with non-coordinates."""
        non_coord_list = ["95823", "94503", "2"]
        assert CimisEndpoints._is_coordinate_list(non_coord_list) is False
        
        # Empty list
        assert CimisEndpoints._is_coordinate_list([]) is False
        
        # Mixed list (returns True if ANY element is a coordinate)
        mixed_list = ["lat=39.36,lng=-121.74", "95823"]
        assert CimisEndpoints._is_coordinate_list(mixed_list) is True
    
    def test_is_address_list_valid(self):
        """Test address list detection with valid addresses."""
        addr_list = ["addr-name=Test,addr=123 Main St", "addr-name=Test2,addr=456 Oak Ave"]
        assert CimisEndpoints._is_address_list(addr_list) is True
    
    def test_is_address_list_invalid(self):
        """Test address list detection with non-addresses."""
        non_addr_list = ["95823", "94503", "2"]
        assert CimisEndpoints._is_address_list(non_addr_list) is False
        
        # Empty list
        assert CimisEndpoints._is_address_list([]) is False
        
        # Mixed list (returns True if ANY element is an address)
        mixed_list = ["addr-name=Test,addr=123 Main St", "95823"]
        assert CimisEndpoints._is_address_list(mixed_list) is True
    
    def test_get_daily_data_items(self):
        """Test getting daily data items."""
        items = CimisEndpoints.get_daily_data_items()
        assert isinstance(items, list)
        assert len(items) > 0
        assert "day-air-tmp-avg" in items
    
    def test_get_hourly_data_items(self):
        """Test getting hourly data items."""
        items = CimisEndpoints.get_hourly_data_items()
        assert isinstance(items, list)
        assert len(items) > 0
        assert "hly-air-tmp" in items
    
    def test_get_all_data_items(self):
        """Test getting all data items."""
        all_items = CimisEndpoints.get_all_data_items()
        daily_items = CimisEndpoints.get_daily_data_items()
        hourly_items = CimisEndpoints.get_hourly_data_items()
        
        assert isinstance(all_items, list)
        assert len(all_items) == len(daily_items) + len(hourly_items)
        
        for item in daily_items:
            assert item in all_items
        
        for item in hourly_items:
            assert item in all_items
    
    def test_get_error_description(self):
        """Test getting error descriptions."""
        # Known error code
        desc = CimisEndpoints.get_error_description("ERR1006")
        assert desc == "INVALID APP KEY"
        
        # Unknown error code
        desc = CimisEndpoints.get_error_description("ERR9999")
        assert desc == "Unknown error: ERR9999"


class TestPrepareBasicRequestParams:
    """Test cases for prepare_basic_request_params method."""
    
    def test_prepare_basic_params(self):
        """Test preparing basic request parameters."""
        params = CimisEndpoints.prepare_basic_request_params("test-api-key")
        
        assert params == {"appKey": "test-api-key"}
        assert isinstance(params, dict)
        assert len(params) == 1


if __name__ == "__main__":
    pytest.main([__file__])
