"""
Integration tests for the Python CIMIS Client library.

These tests verify that all components work together correctly,
but use mocked API responses to avoid making actual API calls.
"""

import pytest
import json
from datetime import date, datetime, timedelta
from unittest.mock import Mock, patch, mock_open
from pathlib import Path

from python_cimis import (
    CimisClient, CimisAPIError, CimisAuthenticationError,
    WeatherData, Station
)
from python_cimis.models import WeatherProvider, WeatherRecord, DataValue


@pytest.mark.integration
class TestCimisClientIntegration:
    """Integration tests for CimisClient with mocked API responses."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
        
        # Sample API response data
        self.sample_weather_response = {
            "Data": {
                "Providers": [
                    {
                        "Name": "cimis",
                        "Type": "station",
                        "Owner": "water.ca.gov",
                        "Records": [
                            {
                                "Date": "2023-01-01",
                                "Julian": "1",
                                "Station": "2",
                                "Standard": "english",
                                "ZipCodes": "93624",
                                "Scope": "daily",
                                "DayAirTmpAvg": {"Value": "39.5", "Qc": " ", "Unit": "(F)"},
                                "DayAirTmpMax": {"Value": "52.0", "Qc": " ", "Unit": "(F)"},
                                "DayAirTmpMin": {"Value": "27.0", "Qc": " ", "Unit": "(F)"},
                                "DayPrecip": {"Value": "0.00", "Qc": " ", "Unit": "(in)"},
                                "DayEto": {"Value": "0.05", "Qc": " ", "Unit": "(in)"}
                            },
                            {
                                "Date": "2023-01-02",
                                "Julian": "2",
                                "Station": "2",
                                "Standard": "english",
                                "ZipCodes": "93624",
                                "Scope": "daily",
                                "DayAirTmpAvg": {"Value": "41.2", "Qc": " ", "Unit": "(F)"},
                                "DayAirTmpMax": {"Value": "54.1", "Qc": " ", "Unit": "(F)"},
                                "DayAirTmpMin": {"Value": "28.3", "Qc": " ", "Unit": "(F)"},
                                "DayPrecip": {"Value": "0.12", "Qc": " ", "Unit": "(in)"},
                                "DayEto": {"Value": "0.04", "Qc": " ", "Unit": "(in)"}
                            }
                        ]
                    }
                ]
            }
        }
        
        self.sample_stations_response = {
            "Stations": [
                {
                    "StationNbr": "2",
                    "Name": "FivePoints",
                    "City": "Five Points",
                    "RegionalOffice": "Fresno",
                    "County": "Fresno",
                    "ConnectDate": "6/7/1982",
                    "DisconnectDate": "12/31/2030",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": "285",
                    "GroundCover": "Grass",
                    "HmsLatitude": "36°20'10N / 36.3360",
                    "HmsLongitude": "-120°6'47W / -120.1130",
                    "ZipCodes": ["93624"],
                    "SitingDesc": ""
                },
                {
                    "StationNbr": "8",
                    "Name": "Blackwells Corner",
                    "City": "Lost Hills",
                    "RegionalOffice": "Fresno",
                    "County": "Kern",
                    "ConnectDate": "7/1/1982",
                    "DisconnectDate": "12/31/2030",
                    "IsActive": "True",
                    "IsEtoStation": "True",
                    "Elevation": "292",
                    "GroundCover": "Grass",
                    "HmsLatitude": "35°38'44N / 35.6456",
                    "HmsLongitude": "-119°59'37W / -119.9936",
                    "ZipCodes": ["93249"],
                    "SitingDesc": ""
                }
            ]
        }
    
    @patch('requests.Session.get')
    def test_get_daily_data_complete_workflow(self, mock_get):
        """Test complete workflow of getting daily weather data."""
        # Mock successful API response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = self.sample_weather_response
        mock_get.return_value = mock_response
        
        # Call the method
        weather_data = self.client.get_daily_data(
            targets=[2, 8],
            start_date="2023-01-01",
            end_date="2023-01-02",
            unit_of_measure="E"
        )
        
        # Verify the request was made correctly
        mock_get.assert_called_once()
        call_args = mock_get.call_args
        
        # Check URL
        assert "data" in call_args[0][0]
        
        # Check parameters
        params = call_args[1]['params']
        assert params['appKey'] == "test-api-key"
        assert params['targets'] == "2,8"
        assert params['startDate'] == "2023-01-01"
        assert params['endDate'] == "2023-01-02"
        assert params['unitOfMeasure'] == "E"
        
        # Verify the response was parsed correctly
        assert isinstance(weather_data, WeatherData)
        assert len(weather_data.providers) == 1
        
        provider = weather_data.providers[0]
        assert provider.name == "cimis"
        assert provider.type == "station"
        assert len(provider.records) == 2
        
        # Check first record
        record1 = provider.records[0]
        assert record1.date == "2023-01-01"
        assert record1.station == "2"
        assert "DayAirTmpAvg" in record1.data_values
        assert record1.get_numeric_value("DayAirTmpAvg") == 39.5
        
        # Check second record
        record2 = provider.records[1]
        assert record2.date == "2023-01-02"
        assert record2.get_numeric_value("DayAirTmpAvg") == 41.2
    
    @patch('requests.Session.get')
    def test_get_stations_complete_workflow(self, mock_get):
        """Test complete workflow of getting station information."""
        # Mock successful API response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = self.sample_stations_response
        mock_get.return_value = mock_response
        
        # Call the method
        stations = self.client.get_stations()
        
        # Verify the request was made correctly
        mock_get.assert_called_once()
        call_args = mock_get.call_args
        
        # Check URL
        assert "station" in call_args[0][0]
        
        # Check parameters
        params = call_args[1]['params']
        assert params['appKey'] == "test-api-key"
        
        # Verify the response was parsed correctly
        assert len(stations) == 2
        assert all(isinstance(station, Station) for station in stations)
        
        # Check first station
        station1 = stations[0]
        assert station1.station_nbr == "2"
        assert station1.name == "FivePoints"
        assert station1.city == "Five Points"
        assert station1.latitude == 36.3360
        assert station1.longitude == -120.1130
        
        # Check second station
        station2 = stations[1]
        assert station2.station_nbr == "8"
        assert station2.name == "Blackwells Corner"
        assert station2.city == "Lost Hills"
    
    @patch('requests.Session.get')
    @patch('builtins.open', new_callable=mock_open)
    @patch('csv.writer')
    def test_export_to_csv_complete_workflow(self, mock_csv_writer, mock_file, mock_get):
        """Test complete workflow of getting data and exporting to CSV."""
        # Mock successful API response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = self.sample_weather_response
        mock_get.return_value = mock_response
        
        # Mock CSV writer
        mock_writer_instance = Mock()
        mock_csv_writer.return_value = mock_writer_instance
        
        # Get data and export
        weather_data = self.client.get_daily_data(
            targets=[2],
            start_date="2023-01-01",
            end_date="2023-01-02"
        )
        
        filename = self.client.export_to_csv(weather_data, "test_export.csv")
        
        # Verify file was opened for writing
        mock_file.assert_called_once()
        
        # Verify CSV writer was used
        mock_csv_writer.assert_called_once()
        
        # Verify data was written
        assert mock_writer_instance.writerow.call_count >= 3  # Header + 2 data rows
        
        # Verify filename was returned
        assert filename.endswith("test_export.csv")
    
    @patch('requests.Session.get')
    def test_get_data_and_export_csv_complete_workflow(self, mock_get):
        """Test complete workflow of get_data_and_export_csv method."""
        # Mock successful API response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = self.sample_weather_response
        mock_get.return_value = mock_response
        
        with patch.object(self.client, 'export_to_csv') as mock_export:
            mock_export.return_value = "test_file.csv"
            
            # Call the combined method
            weather_data, filename = self.client.get_data_and_export_csv(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02",
                filename="custom_export.csv"
            )
            
            # Verify both operations were performed
            assert isinstance(weather_data, WeatherData)
            mock_export.assert_called_once_with(weather_data, "custom_export.csv")
            assert filename == "test_file.csv"
    
    @patch('requests.Session.get')
    def test_error_handling_authentication_error(self, mock_get):
        """Test error handling for authentication errors."""
        # Mock 403 response
        mock_response = Mock()
        mock_response.status_code = 403
        mock_response.json.return_value = {"Message": "Invalid API key"}
        mock_get.return_value = mock_response
        
        # Should raise CimisAuthenticationError
        with pytest.raises(CimisAuthenticationError):
            self.client.get_daily_data(
                targets=[2],
                start_date="2023-01-01",
                end_date="2023-01-02"
            )
    
    @patch('requests.Session.get')
    def test_error_handling_api_error(self, mock_get):
        """Test error handling for general API errors."""
        # Mock 400 response
        mock_response = Mock()
        mock_response.status_code = 400
        mock_response.json.return_value = {
            "Message": "Station not found",
            "ErrorCode": "ERR1019"
        }
        mock_get.return_value = mock_response
        
        # Should raise CimisAPIError
        with pytest.raises(CimisAPIError) as exc_info:
            self.client.get_daily_data(
                targets=[999],
                start_date="2023-01-01",
                end_date="2023-01-02"
            )
        
        # Check error details
        error = exc_info.value
        assert error.http_code == 400
    
    def test_date_handling_with_date_objects(self):
        """Test that date objects are handled correctly."""
        with patch.object(self.client, '_make_request') as mock_request:
            mock_request.return_value = self.sample_weather_response
            
            # Use actual date objects
            start_date = date(2023, 1, 1)
            end_date = date(2023, 1, 2)
            
            self.client.get_daily_data(
                targets=[2],
                start_date=start_date,
                end_date=end_date
            )
            
            # Check that dates were converted to strings
            call_args = mock_request.call_args
            params = call_args[0][1]
            assert params['startDate'] == "2023-01-01"
            assert params['endDate'] == "2023-01-02"
    
    def test_date_handling_with_datetime_objects(self):
        """Test that datetime objects are handled correctly."""
        with patch.object(self.client, '_make_request') as mock_request:
            mock_request.return_value = self.sample_weather_response
            
            # Use datetime objects
            start_date = datetime(2023, 1, 1, 10, 30)
            end_date = datetime(2023, 1, 2, 15, 45)
            
            self.client.get_daily_data(
                targets=[2],
                start_date=start_date,
                end_date=end_date
            )
            
            # Check that dates were converted to strings (time ignored)
            call_args = mock_request.call_args
            params = call_args[0][1]
            assert params['startDate'] == "2023-01-01"
            assert params['endDate'] == "2023-01-02"
    
    @patch('requests.Session.get')
    def test_multiple_providers_handling(self, mock_get):
        """Test handling of responses with multiple data providers."""
        # Response with both station and spatial data
        multi_provider_response = {
            "Data": {
                "Providers": [
                    {
                        "Name": "cimis",
                        "Type": "station",
                        "Owner": "water.ca.gov",
                        "Records": [
                            {
                                "Date": "2023-01-01",
                                "Julian": "1",
                                "Station": "2",
                                "Standard": "english",
                                "ZipCodes": "93624",
                                "Scope": "daily",
                                "DayAirTmpAvg": {"Value": "39.5", "Qc": " ", "Unit": "(F)"}
                            }
                        ]
                    },
                    {
                        "Name": "scs",
                        "Type": "spatial",
                        "Owner": "scs.usda.gov",
                        "Records": [
                            {
                                "Date": "2023-01-01",
                                "Julian": "1",
                                "Station": "",
                                "Standard": "english",
                                "ZipCodes": "95823",
                                "Scope": "daily",
                                "DayAirTmpAvg": {"Value": "42.1", "Qc": " ", "Unit": "(F)"}
                            }
                        ]
                    }
                ]
            }
        }
        
        # Mock API response
        mock_response = Mock()
        mock_response.status_code = 200
        mock_response.json.return_value = multi_provider_response
        mock_get.return_value = mock_response
        
        # Get data
        weather_data = self.client.get_daily_data(
            targets=["2", "95823"],
            start_date="2023-01-01",
            end_date="2023-01-01",
            prioritize_scs=True
        )
        
        # Should have both providers
        assert len(weather_data.providers) == 2
        
        # Check provider types
        provider_types = [p.type for p in weather_data.providers]
        assert "station" in provider_types
        assert "spatial" in provider_types
        
        # Check total records
        all_records = weather_data.get_all_records()
        assert len(all_records) == 2
    
    def test_weather_data_filtering_methods(self):
        """Test WeatherData filtering methods with realistic data."""
        # Create weather data with multiple stations and dates
        weather_data = WeatherData()
        
        # Provider 1: Station 2
        provider1 = WeatherProvider(name="cimis", type="station", owner="test")
        record1 = WeatherRecord(date="2023-01-01", julian="1", station="2")
        record2 = WeatherRecord(date="2023-01-02", julian="2", station="2")
        provider1.records = [record1, record2]
        
        # Provider 2: Station 8
        provider2 = WeatherProvider(name="cimis", type="station", owner="test")
        record3 = WeatherRecord(date="2023-01-01", julian="1", station="8")
        record4 = WeatherRecord(date="2023-01-02", julian="2", station="8")
        provider2.records = [record3, record4]
        
        weather_data.providers = [provider1, provider2]
        
        # Test get_records_by_station
        station2_records = weather_data.get_records_by_station("2")
        assert len(station2_records) == 2
        assert all(r.station == "2" for r in station2_records)
        
        station8_records = weather_data.get_records_by_station("8")
        assert len(station8_records) == 2
        assert all(r.station == "8" for r in station8_records)
        
        # Test get_records_by_date
        jan1_records = weather_data.get_records_by_date("2023-01-01")
        assert len(jan1_records) == 2
        assert all(r.date == "2023-01-01" for r in jan1_records)
        
        jan2_records = weather_data.get_records_by_date("2023-01-02")
        assert len(jan2_records) == 2
        assert all(r.date == "2023-01-02" for r in jan2_records)


@pytest.mark.integration
class TestRealWorldScenarios:
    """Test scenarios that mimic real-world usage patterns."""
    
    def setup_method(self):
        """Set up test client."""
        self.client = CimisClient(app_key="test-api-key")
    
    def test_example_usage_scenario(self):
        """Test scenario matching the example_usage.py file."""
        with patch.object(self.client, '_make_request') as mock_request:
            # Mock weather data response
            weather_response = {
                "Data": {
                    "Providers": [
                        {
                            "Name": "cimis",
                            "Type": "station",
                            "Owner": "water.ca.gov",
                            "Records": [
                                {
                                    "Date": "2023-01-01",
                                    "Julian": "1",
                                    "Station": "2",
                                    "Standard": "english",
                                    "ZipCodes": "93624",
                                    "Scope": "daily",
                                    "DayAirTmpAvg": {"Value": "39.5", "Qc": " ", "Unit": "(F)"},
                                    "DayPrecip": {"Value": "0.00", "Qc": " ", "Unit": "(in)"},
                                    "DayEto": {"Value": "0.05", "Qc": " ", "Unit": "(in)"}
                                }
                            ]
                        }
                    ]
                }
            }
            
            # Mock stations response
            stations_response = {
                "Stations": [
                    {
                        "StationNbr": "2",
                        "Name": "FivePoints",
                        "City": "Five Points",
                        "ConnectDate": "6/7/1982",
                        "DisconnectDate": "12/31/2030",
                        "IsActive": "True",
                        "IsEtoStation": "True",
                        "Elevation": "285",
                        "GroundCover": "Grass",
                        "HmsLatitude": "36°20'10N / 36.3360",
                        "HmsLongitude": "-120°6'47W / -120.1130",
                        "ZipCodes": ["93624"],
                        "SitingDesc": ""
                    }
                ]
            }
            
            # Set up mock to return appropriate responses based on endpoint
            def mock_request_side_effect(endpoint, params):
                if endpoint == "data":
                    return weather_response
                elif endpoint == "stations":
                    return stations_response
                else:
                    return {}
            
            mock_request.side_effect = mock_request_side_effect
            
            # Define date range like in example
            end_date = date.today() - timedelta(days=1)
            start_date = end_date - timedelta(days=6)
            
            # Example 1: Get daily weather data by station numbers
            weather_data = self.client.get_daily_data(
                targets=[2, 8, 127],
                start_date=start_date,
                end_date=end_date,
                unit_of_measure="E"
            )
            
            assert isinstance(weather_data, WeatherData)
            assert len(weather_data.get_all_records()) > 0
            
            # Example 2: Get station information
            stations = self.client.get_stations()
            
            assert len(stations) == 1
            assert stations[0].station_nbr == "2"
            assert stations[0].name == "FivePoints"
            
            # Verify correct parameters were passed
            assert mock_request.call_count >= 2
    
    @patch('builtins.open', new_callable=mock_open)
    @patch('csv.DictWriter')
    def test_csv_export_with_all_columns(self, mock_dict_writer, mock_file):
        """Test CSV export includes all available data columns."""
        # Create weather data with multiple data items
        weather_data = WeatherData()
        provider = WeatherProvider(name="cimis", type="station", owner="test")
        
        record = WeatherRecord(date="2023-01-01", julian="1", station="2")
        record.data_values = {
            "DayAirTmpAvg": DataValue("39.5", " ", "(F)"),
            "DayAirTmpMax": DataValue("52.0", " ", "(F)"),
            "DayAirTmpMin": DataValue("27.0", " ", "(F)"),
            "DayPrecip": DataValue("0.00", " ", "(in)"),
            "DayEto": DataValue("0.05", " ", "(in)"),
            "DayRelHumAvg": DataValue("65.2", " ", "(%)")
        }
        
        provider.records = [record]
        weather_data.providers = [provider]
        
        # Mock DictWriter
        mock_writer_instance = Mock()
        mock_dict_writer.return_value = mock_writer_instance
        
        # Export to CSV
        filename = self.client.export_to_csv(weather_data, "test.csv", include_all_columns=True)
        
        # Verify file operations
        mock_file.assert_called_once()
        mock_dict_writer.assert_called_once()
        
        # Verify DictWriter was created with correct fieldnames
        call_args = mock_dict_writer.call_args
        fieldnames = call_args[1]['fieldnames']
        
        # Should include basic columns
        assert "Date" in fieldnames
        assert "Station" in fieldnames
        
        # Should include all data value columns
        assert "DayAirTmpAvg_Value" in fieldnames
        assert "DayAirTmpMax_Value" in fieldnames
        assert "DayPrecip_Value" in fieldnames
        assert "DayEto_Value" in fieldnames
        assert "DayRelHumAvg_Value" in fieldnames


if __name__ == "__main__":
    pytest.main([__file__])
