"""
Test suite for the Python CIMIS exceptions module.
"""

import pytest
from python_cimis.exceptions import (
    CimisError,
    CimisAPIError,
    CimisDataError,
    CimisConnectionError,
    CimisAuthenticationError
)


class TestCimisError:
    """Test cases for CimisError base class."""
    
    def test_init(self):
        """Test CimisError initialization."""
        error = CimisError("Test error message")
        assert str(error) == "Test error message"
    
    def test_inheritance(self):
        """Test that CimisError inherits from Exception."""
        error = CimisError("Test error")
        assert isinstance(error, Exception)
    
    def test_raise_and_catch(self):
        """Test raising and catching CimisError."""
        with pytest.raises(CimisError) as exc_info:
            raise CimisError("Test error message")
        
        assert str(exc_info.value) == "Test error message"


class TestCimisAPIError:
    """Test cases for CimisAPIError class."""
    
    def test_init_message_only(self):
        """Test CimisAPIError initialization with message only."""
        error = CimisAPIError("API error occurred")
        assert str(error) == "API error occurred"
        assert error.error_code is None
        assert error.http_code is None
    
    def test_init_with_error_code(self):
        """Test CimisAPIError initialization with error code."""
        error = CimisAPIError("Invalid API key", error_code="ERR1006")
        assert error.error_code == "ERR1006"
        assert error.http_code is None
    
    def test_init_with_http_code(self):
        """Test CimisAPIError initialization with HTTP code."""
        error = CimisAPIError("Forbidden", http_code=403)
        assert error.error_code is None
        assert error.http_code == 403
    
    def test_init_with_both_codes(self):
        """Test CimisAPIError initialization with both codes."""
        error = CimisAPIError("Invalid API key", error_code="ERR1006", http_code=403)
        assert error.error_code == "ERR1006"
        assert error.http_code == 403
    
    def test_str_with_codes(self):
        """Test string representation with error and HTTP codes."""
        error = CimisAPIError("Invalid API key", error_code="ERR1006", http_code=403)
        expected = "[ERR1006] HTTP 403: Invalid API key"
        assert str(error) == expected
    
    def test_str_without_codes(self):
        """Test string representation without codes."""
        error = CimisAPIError("General API error")
        assert str(error) == "General API error"
    
    def test_str_with_only_error_code(self):
        """Test string representation with only error code."""
        error = CimisAPIError("Invalid API key", error_code="ERR1006")
        assert str(error) == "Invalid API key"
    
    def test_str_with_only_http_code(self):
        """Test string representation with only HTTP code."""
        error = CimisAPIError("Forbidden", http_code=403)
        assert str(error) == "Forbidden"
    
    def test_inheritance(self):
        """Test that CimisAPIError inherits from CimisError."""
        error = CimisAPIError("API error")
        assert isinstance(error, CimisError)
        assert isinstance(error, Exception)
    
    def test_raise_and_catch(self):
        """Test raising and catching CimisAPIError."""
        with pytest.raises(CimisAPIError) as exc_info:
            raise CimisAPIError("API error", error_code="ERR1006", http_code=403)
        
        error = exc_info.value
        assert error.error_code == "ERR1006"
        assert error.http_code == 403
        assert str(error) == "[ERR1006] HTTP 403: API error"


class TestCimisDataError:
    """Test cases for CimisDataError class."""
    
    def test_init(self):
        """Test CimisDataError initialization."""
        error = CimisDataError("Data processing error")
        assert str(error) == "Data processing error"
    
    def test_inheritance(self):
        """Test that CimisDataError inherits from CimisError."""
        error = CimisDataError("Data error")
        assert isinstance(error, CimisError)
        assert isinstance(error, Exception)
    
    def test_raise_and_catch(self):
        """Test raising and catching CimisDataError."""
        with pytest.raises(CimisDataError) as exc_info:
            raise CimisDataError("Invalid data format")
        
        assert str(exc_info.value) == "Invalid data format"


class TestCimisConnectionError:
    """Test cases for CimisConnectionError class."""
    
    def test_init(self):
        """Test CimisConnectionError initialization."""
        error = CimisConnectionError("Connection failed")
        assert str(error) == "Connection failed"
    
    def test_inheritance(self):
        """Test that CimisConnectionError inherits from CimisError."""
        error = CimisConnectionError("Connection error")
        assert isinstance(error, CimisError)
        assert isinstance(error, Exception)
    
    def test_raise_and_catch(self):
        """Test raising and catching CimisConnectionError."""
        with pytest.raises(CimisConnectionError) as exc_info:
            raise CimisConnectionError("Network timeout")
        
        assert str(exc_info.value) == "Network timeout"


class TestCimisAuthenticationError:
    """Test cases for CimisAuthenticationError class."""
    
    def test_init(self):
        """Test CimisAuthenticationError initialization."""
        error = CimisAuthenticationError("Authentication failed")
        assert str(error) == "Authentication failed"
    
    def test_init_with_codes(self):
        """Test CimisAuthenticationError initialization with codes."""
        error = CimisAuthenticationError(
            "Invalid API key", 
            error_code="ERR1006", 
            http_code=403
        )
        assert error.error_code == "ERR1006"
        assert error.http_code == 403
        assert str(error) == "[ERR1006] HTTP 403: Invalid API key"
    
    def test_inheritance(self):
        """Test that CimisAuthenticationError inherits from CimisAPIError."""
        error = CimisAuthenticationError("Auth error")
        assert isinstance(error, CimisAPIError)
        assert isinstance(error, CimisError)
        assert isinstance(error, Exception)
    
    def test_raise_and_catch(self):
        """Test raising and catching CimisAuthenticationError."""
        with pytest.raises(CimisAuthenticationError) as exc_info:
            raise CimisAuthenticationError("Invalid credentials", error_code="ERR1006")
        
        error = exc_info.value
        assert error.error_code == "ERR1006"
        assert str(error) == "Invalid credentials"


class TestExceptionHierarchy:
    """Test cases for exception inheritance hierarchy."""
    
    def test_catch_base_exception(self):
        """Test that base CimisError can catch all CIMIS exceptions."""
        # CimisAPIError
        with pytest.raises(CimisError):
            raise CimisAPIError("API error")
        
        # CimisDataError
        with pytest.raises(CimisError):
            raise CimisDataError("Data error")
        
        # CimisConnectionError
        with pytest.raises(CimisError):
            raise CimisConnectionError("Connection error")
        
        # CimisAuthenticationError
        with pytest.raises(CimisError):
            raise CimisAuthenticationError("Auth error")
    
    def test_catch_specific_exceptions(self):
        """Test catching specific exception types."""
        # CimisAuthenticationError can be caught as CimisAPIError
        with pytest.raises(CimisAPIError):
            raise CimisAuthenticationError("Auth error")
        
        # But CimisDataError cannot be caught as CimisAPIError
        try:
            raise CimisDataError("Data error")
        except CimisAPIError:
            pytest.fail("CimisDataError should not be caught as CimisAPIError")
        except CimisDataError:
            pass  # This is expected
    
    def test_exception_with_cause(self):
        """Test exceptions with underlying causes."""
        original_error = ValueError("Original error")
        
        try:
            raise CimisDataError("Wrapper error") from original_error
        except CimisDataError as e:
            assert str(e) == "Wrapper error"
            assert e.__cause__ is original_error


if __name__ == "__main__":
    pytest.main([__file__])
