# Dev Notes: 2026-01-24

## python-clack Initial Release & PyPI Publishing

### What Was Built

Created `python-clack`, a Python port of the Node.js [clack](https://github.com/bombshell-dev/clack) package for beautiful CLI prompts.

**Features implemented:**
- Text input with placeholder, validation, default values
- Select (single selection with arrow keys)
- Multiselect (multiple selection with space toggle)
- Confirm (yes/no with y/n shortcuts)
- Password (masked input)
- Spinner (animated loading indicator)
- Log utilities (info, success, warn, error, step)
- Intro/outro/cancel message banners
- Group function for sequential prompts
- Automatic Unicode/ASCII fallback

**Tech stack:**
- `prompt_toolkit` - Cross-platform input handling
- `rich` - Terminal colors and styling
- `hatchling` - Build backend
- `uv` - Package management

---

## PyPI Publishing with Trusted Publishing (OIDC)

### Why Trusted Publishing?

- **No secrets to manage** - No API tokens stored in GitHub
- **More secure** - Uses OIDC (OpenID Connect) tokens
- **Automatic** - GitHub proves identity to PyPI cryptographically

### One-Time Setup on PyPI

1. Go to https://pypi.org/manage/account/publishing/
2. Click **"Add a new pending publisher"**
3. Fill in:
   - **PyPI Project Name:** `python-clack`
   - **Owner:** `oharu121`
   - **Repository:** `python-clack`
   - **Workflow name:** `publish.yml`
   - **Environment:** *(leave blank)*
4. Click **"Add"**

### GitHub Action Workflow

File: `.github/workflows/publish.yml`

```yaml
name: Publish to PyPI

on:
  release:
    types: [created]

jobs:
  publish:
    runs-on: ubuntu-latest
    permissions:
      contents: read
      id-token: write  # Required for trusted publishing

    steps:
      - name: Checkout
        uses: actions/checkout@v4

      - name: Setup Python
        uses: actions/setup-python@v5
        with:
          python-version: '3.12'

      - name: Install uv
        uses: astral-sh/setup-uv@v4

      - name: Install dependencies
        run: uv sync

      - name: Run linting
        run: uv run ruff check src/python_clack

      - name: Build package
        run: uv build

      - name: Publish to PyPI
        uses: pypa/gh-action-pypi-publish@release/v1
        # No token needed - uses OIDC trusted publishing
```

### Publishing Process

1. **Commit and push** all changes to GitHub
2. **Create a GitHub Release:**
   - Go to Releases → "Create a new release"
   - Create a new tag (e.g., `v0.1.0`)
   - Add release title and notes
   - Click "Publish release"
3. **GitHub Action triggers** automatically
4. **Package published** to PyPI

### Verifying Publication

- PyPI page: https://pypi.org/project/python-clack/
- Install test: `pip install python-clack`

---

## Alternative: Manual Publishing with API Token

If you need to publish manually or from a different CI:

### Get PyPI Token

1. Go to https://pypi.org/manage/account/token/
2. Create token with scope for `python-clack` project
3. Copy token (starts with `pypi-`)

### Publish Locally

```bash
uv build
uv publish --token pypi-your-token-here
```

### Publish via GitHub Actions (with secret)

```yaml
- name: Publish to PyPI
  uses: pypa/gh-action-pypi-publish@release/v1
  with:
    token: ${{ secrets.PYPI_TOKEN }}
```

Add `PYPI_TOKEN` at: `Settings → Secrets and variables → Actions → New repository secret`

---

## TestPyPI (for testing before real publish)

1. Create account at https://test.pypi.org
2. Set up trusted publishing (same steps, different site)
3. Publish:
   ```bash
   uv publish --publish-url https://test.pypi.org/legacy/
   ```
4. Test install:
   ```bash
   pip install -i https://test.pypi.org/simple/ python-clack
   ```

---

## Version Bumping Checklist

When releasing a new version:

1. Update version in `pyproject.toml`
2. Update `CHANGELOG.md`
3. Commit: `git commit -m "chore: bump version to vX.Y.Z"`
4. Push to GitHub
5. Create GitHub Release with tag `vX.Y.Z`

---

## Useful Links

- **PyPI Project:** https://pypi.org/project/python-clack/
- **GitHub Repo:** https://github.com/oharu121/python-clack
- **Trusted Publishing Docs:** https://docs.pypi.org/trusted-publishers/
- **uv Docs:** https://docs.astral.sh/uv/

---

## Windows Encoding Issues (Python vs Node.js)

### The Problem We Faced

When running on Windows legacy console (cmd.exe), Unicode symbols like `◆ │ ●` caused `UnicodeEncodeError` because Python tried to encode them to `cp1252` (Windows code page).

### Why Python Has This Issue (But Node.js Doesn't)

| Aspect | Node.js | Python |
|--------|---------|--------|
| String encoding | Always UTF-16 internal | Platform-dependent |
| Console output | Forces UTF-8 (bypasses code page) | Respects system code page |
| Windows legacy console | Works | Breaks without handling |

**Node.js** writes UTF-8 bytes directly to the console buffer, bypassing code page translation.

**Python** asks Windows "what encoding do you want?" and Windows says `cp1252`, which can't represent Unicode symbols.

```python
# On Windows legacy console
>>> import sys
>>> sys.stdout.encoding
'cp1252'  # Can't encode ◆, │, ●, etc.

# On macOS/Linux
>>> sys.stdout.encoding
'utf-8'   # Everything works
```

### Our Solution: Graceful Fallback

In `symbols.py`, we detect terminal capabilities:

```python
def _is_unicode_supported() -> bool:
    encoding = getattr(sys.stdout, "encoding", "") or ""

    # UTF-8 encoding = Unicode OK
    if encoding.lower() in ("utf-8", "utf8"):
        return True

    # Windows-specific checks
    if sys.platform == "win32":
        if os.environ.get("WT_SESSION"):  # Windows Terminal
            return True
        if os.environ.get("TERM_PROGRAM") == "vscode":
            return True
        return False  # Legacy console = ASCII fallback

    # macOS/Linux = Unicode OK
    return True
```

Then symbols fall back to ASCII:

```python
S_STEP_ACTIVE = _u("◆", "*")  # Unicode or ASCII
S_BAR = _u("│", "|")
```

### Platform Compatibility

| Platform | Terminal | Unicode | Notes |
|----------|----------|---------|-------|
| macOS | Terminal.app | ✓ | UTF-8 native |
| macOS | iTerm2 | ✓ | UTF-8 native |
| Linux | Any | ✓ | UTF-8 default |
| Windows | Windows Terminal | ✓ | Sets `WT_SESSION` |
| Windows | VS Code | ✓ | Sets `TERM_PROGRAM` |
| Windows | cmd.exe (legacy) | ✗ | Falls back to ASCII |
| Windows | PowerShell (legacy) | ✗ | Falls back to ASCII |

### Force UTF-8 on Windows (Alternative)

If users want Unicode on legacy Windows:

```bash
# Environment variable
set PYTHONUTF8=1
python app.py

# Or use our override
set PYTHON_CLACK_UNICODE=1
python app.py
```

### Key Takeaway

- **macOS/Linux**: Just works, no special handling needed
- **Windows Terminal/VS Code**: Just works
- **Windows legacy console**: Graceful ASCII fallback prevents crashes
- **Node.js doesn't have this problem** because it bypasses the OS encoding layer

---

## Python Package Example/Demo Patterns

### Common Patterns

| Pattern | When to Use | Examples |
|---------|-------------|----------|
| `examples/` directory | Libraries with multiple examples | rich, click, httpx |
| `__main__.py` in package | Runnable via `python -m pkg` | pip, pytest, black |
| `main.py` at root | Dev testing only (often gitignored) | - |

### Our Chosen Structure

```
examples/
    basic.py      # Full demo with all prompts
    group.py      # Group function example
src/python_clack/
    _demo.py      # CLI entry point for installed package
```

### Running Examples

```bash
# From source (development)
uv run python examples/basic.py
uv run python examples/group.py

# After pip install (users)
python-clack-demo
```

### Why This Structure?

1. **`examples/` directory** - Standard convention for libraries. Users browsing the repo immediately recognize these as example code.

2. **`_demo.py` inside package** - Enables the `python-clack-demo` CLI command for users who install via pip. The underscore prefix indicates it's internal.

3. **No `main.py` at root** - Avoids confusion. `main.py` is ambiguous (is it the package entry point? a script? tests?).

### CLI Entry Point

In `pyproject.toml`:

```toml
[project.scripts]
python-clack-demo = "python_clack._demo:main"
```

This creates an executable command when the package is installed:
- `pip install python-clack` → user can run `python-clack-demo`
- Useful for quick testing without writing code
