# Dev Notes - 2026-01-25

## Python Package Layout: `src/` vs Flat

### Two Common Layouts

| Layout | Structure | Import |
|--------|-----------|--------|
| Flat | `python_clack/__init__.py` | `import python_clack` |
| Src | `src/python_clack/__init__.py` | `import python_clack` |

### Why `src/python_clack/` not just `src/`?

The folder name inside `src/` must match the **import name**. If you put files directly under `src/`:

```
src/
    __init__.py
    text.py
```

You'd have to `import src.text` which is wrong. The package name would be `src`.

### Correct Structure

```
src/
    python_clack/        # This is the package name (matches import)
        __init__.py
        _core/
        _prompts/
        ...
```

This lets users do `from python_clack import text, select` after installing.

### Why Use `src/` Layout at All?

The `src/` layout is recommended by PyPA (Python Packaging Authority) because:

1. **Prevents accidental local imports** - Without `src/`, if you run `python` in the project root, Python might import the local `python_clack/` folder instead of the installed package. This can hide packaging bugs.

2. **Forces proper installation** - You must install the package (`pip install -e .` or `uv sync`) to test it. This catches missing files in `pyproject.toml`.

3. **Cleaner project root** - Separates source code from config files (`pyproject.toml`, `README.md`, etc.)

### Flat Layout Example

```
python_clack/
    __init__.py
    text.py
pyproject.toml
README.md
```

- Simpler structure
- Risk of importing local code instead of installed package
- Common in older projects

### Src Layout Example (Recommended)

```
src/
    python_clack/
        __init__.py
        text.py
pyproject.toml
README.md
```

- Modern best practice
- No risk of accidental local imports
- Used by major projects (pip, setuptools, etc.)

### pyproject.toml Configuration

For src layout with hatchling:

```toml
[build-system]
requires = ["hatchling"]
build-backend = "hatchling.build"

# hatchling automatically finds packages under src/
```

For explicit configuration:

```toml
[tool.hatch.build.targets.wheel]
packages = ["src/python_clack"]
```

---

## Publishing Gotcha: Clean dist/ Before Rebuild

When republishing a new version:

```powershell
# Wrong - old files still in dist/
uv build
uv publish  # Tries to upload old + new files!

# Correct - clean first
Remove-Item -Recurse -Force dist
uv build
uv publish
```

`uv publish` uploads **all files** in the `dist/` folder, including old versions that already exist on PyPI (which fails with 400 error).

---

## Tool Caches

### What Are They?

Dev tools create cache directories to speed up subsequent runs by storing intermediate results.

### Common Caches

| Cache | Tool | Created When | Contents |
|-------|------|--------------|----------|
| `__pycache__/` | Python | Running any `.py` file | Compiled bytecode (`.pyc`) |
| `.ruff_cache/` | Ruff | `uv run ruff check` | Linting results per file |
| `.pytest_cache/` | Pytest | `uv run pytest` | Test state, last failed tests |
| `.mypy_cache/` | Mypy | `uv run mypy` | Type checking results |

### When Do They Appear?

- **First run** of each tool creates its cache
- Caches are **per-project** (created in project root)
- They grow as you work, shrink when files are deleted

### Why Cache?

Without caching, tools would re-analyze every file on every run:

```bash
# First run - analyzes all files (slow)
uv run ruff check  # Creates .ruff_cache/

# Second run - only checks changed files (fast)
uv run ruff check  # Uses .ruff_cache/
```

### Safe to Delete?

Yes. Caches are always safe to delete - tools recreate them as needed:

```powershell
# Delete all caches
Remove-Item -Recurse -Force __pycache__, .ruff_cache, .pytest_cache, .mypy_cache
```

### Git Ignore

All caches should be in `.gitignore` (they're machine-specific):

```gitignore
__pycache__/
.ruff_cache/
.pytest_cache/
.mypy_cache/
```

### Note on Pyright

Pyright (which we use instead of mypy) doesn't create a persistent cache directory by default - it caches in memory during execution.
