# python-clack Roadmap

A Python port of [@clack/prompts](https://github.com/bombshell-dev/clack) for beautiful CLI interfaces.

## Feature Comparison

### ✅ Implemented (v0.1.x)

| Feature | Description | Status |
|---------|-------------|--------|
| `text` | Single-line text input with placeholder, validation, default values | ✅ Complete |
| `password` | Masked password input | ✅ Complete |
| `confirm` | Yes/No boolean confirmation | ✅ Complete |
| `select` | Single-choice selection with arrow navigation | ✅ Complete |
| `multiselect` | Multi-choice with space toggle, select all, invert | ✅ Complete |
| `spinner` | Animated loading indicator with start/stop/error | ✅ Complete |
| `log` | Styled logging (info/success/warn/error/step) | ✅ Complete |
| `intro` | Introduction banner | ✅ Complete |
| `outro` | Completion message | ✅ Complete |
| `cancel` | Cancellation message | ✅ Complete |
| `group` | Sequential prompt execution with result passing | ✅ Complete |
| `is_cancel` | Check if value is cancel symbol | ✅ Complete |

### 🔲 Not Yet Implemented

#### High Priority (v0.2.0)

| Feature | Description | Complexity |
|---------|-------------|------------|
| `note` | Formatted box with title and message | Low |
| `selectKey` | Single-character keyboard shortcut selection | Medium |
| `progress` | Progress bar with percentage | Medium |
| `tasks` | Execute sequential tasks with spinners | Medium |

#### Medium Priority (v0.3.0)

| Feature | Description | Complexity |
|---------|-------------|------------|
| `groupMultiselect` | Multi-select with grouped/nested options | Medium |
| `autocomplete` | Type-ahead search with single selection | High |
| `autocompleteMultiselect` | Type-ahead search with multiple selection | High |
| `stream` | Stream-based logging for async iterables | Medium |

#### Lower Priority (v0.4.0+)

| Feature | Description | Complexity |
|---------|-------------|------------|
| `path` | File/directory path autocomplete | High |
| `box` | Customizable bordered box for content | Low |
| `taskLog` | Advanced task logging with message buffering | Medium |
| `settings` | Global configuration management | Low |

---

## Detailed Feature Specifications

### v0.2.0 - Output & Progress

#### `note(message, title)`
Display a formatted box with title and content.

```python
from python_clack import note

note("Remember to commit your changes!", "Reminder")
# ┌─ Reminder ────────────────────────┐
# │                                   │
# │  Remember to commit your changes! │
# │                                   │
# └───────────────────────────────────┘
```

**Options:**
- `message`: Content to display
- `title`: Box title (optional)

---

#### `selectKey(message, options)`
Select an option by pressing a single key.

```python
from python_clack import selectKey

action = selectKey(
    "What would you like to do?",
    options=[
        {"value": "create", "label": "Create a new file", "key": "c"},
        {"value": "delete", "label": "Delete a file", "key": "d"},
        {"value": "rename", "label": "Rename a file", "key": "r"},
    ]
)
# Press 'c' to select "Create a new file" instantly
```

**Options:**
- `message`: Prompt message
- `options`: List of options with `value`, `label`, and optional `key`

---

#### `progress()`
Progress bar with percentage display.

```python
from python_clack import progress
import time

p = progress()
p.start("Downloading...")

for i in range(100):
    time.sleep(0.05)
    p.advance(1)  # Increment by 1

p.stop("Download complete!")
```

**Options:**
- `max`: Maximum value (default: 100)
- `size`: Bar width in characters (default: 40)
- `style`: Character style - 'light', 'heavy', 'block'

**Methods:**
- `start(message)`: Initialize progress bar
- `advance(step, message)`: Increment progress
- `stop(message)`: Complete successfully
- `error(message)`: Stop with error

---

#### `tasks(task_list)`
Execute sequential tasks with automatic spinner management.

```python
from python_clack import tasks

await tasks([
    {
        "title": "Installing dependencies",
        "task": lambda set_message: install_deps(),
    },
    {
        "title": "Building project",
        "task": lambda set_message: build_project(),
        "enabled": should_build,  # Skip if False
    },
    {
        "title": "Running tests",
        "task": lambda set_message: run_tests(),
    },
])
```

**Task structure:**
- `title`: Display name
- `task`: Async function receiving message callback
- `enabled`: Skip if False (default: True)

---

### v0.3.0 - Advanced Selection

#### `groupMultiselect(message, options)`
Multi-select with grouped options.

```python
from python_clack import groupMultiselect

selected = groupMultiselect(
    "Select packages to install",
    options={
        "Frontend": [
            {"value": "react", "label": "React"},
            {"value": "vue", "label": "Vue"},
        ],
        "Backend": [
            {"value": "fastapi", "label": "FastAPI"},
            {"value": "django", "label": "Django"},
        ],
    }
)
```

**Options:**
- `message`: Prompt message
- `options`: Dict of group name → option list
- `selectableGroups`: Allow selecting entire groups (default: True)

---

#### `autocomplete(message, options)`
Type-ahead search with filtering.

```python
from python_clack import autocomplete

country = autocomplete(
    "Select your country",
    options=[
        {"value": "us", "label": "United States"},
        {"value": "uk", "label": "United Kingdom"},
        {"value": "ca", "label": "Canada"},
        # ... many more options
    ],
    placeholder="Type to search...",
)
```

**Options:**
- `message`: Prompt message
- `options`: List of options or async function returning options
- `placeholder`: Search placeholder
- `maxItems`: Maximum visible items
- `filter`: Custom filter function

---

#### `autocompleteMultiselect(message, options)`
Type-ahead search with multiple selection.

```python
from python_clack import autocompleteMultiselect

tags = autocompleteMultiselect(
    "Select tags",
    options=fetch_all_tags,  # Can be async function
    placeholder="Search tags...",
)
```

---

#### `stream`
Stream-based logging for async content.

```python
from python_clack import stream

async def generate_content():
    for word in ["Hello", " ", "World", "!"]:
        yield word
        await asyncio.sleep(0.1)

await stream.info(generate_content())
```

---

### v0.4.0+ - Advanced Features

#### `path(message)`
File/directory path autocomplete.

```python
from python_clack import path

file = path(
    "Select a file",
    root="./src",  # Starting directory
    directory=False,  # Show files, not just directories
)
```

---

#### `box(message, title)`
Customizable bordered box.

```python
from python_clack import box

box(
    "This is the content",
    title="My Box",
    width=0.5,  # 50% of terminal width
    rounded=True,
    contentAlign="center",
)
```

---

#### `settings`
Global configuration.

```python
from python_clack import settings, updateSettings

updateSettings({
    "aliases": "vim",  # Use hjkl for navigation
    "messages": {
        "cancel": "Operation cancelled",
        "error": "An error occurred",
    },
    "withGuide": False,  # Disable decorative guides
})
```

---

## Implementation Notes

### Architecture

The package follows a layered architecture:

```
python_clack/
├── _core/           # Low-level primitives
│   ├── state.py     # State machine, CANCEL sentinel
│   ├── prompt.py    # Base Prompt class
│   └── render.py    # Terminal rendering
├── _prompts/        # Prompt implementations
│   ├── text.py
│   ├── select.py
│   └── ...
├── symbols.py       # Unicode/ASCII symbols
├── style.py         # Color utilities (Rich)
├── messages.py      # intro/outro/cancel
├── log.py           # Logging utilities
├── spinner.py       # Spinner implementation
└── __init__.py      # Public API
```

### Dependencies

- `prompt_toolkit`: Cross-platform terminal input (raw mode, key events)
- `rich`: Terminal colors and styling

### Compatibility

- Python 3.9+
- Windows, macOS, Linux
- Unicode with ASCII fallback

---

## Version History

| Version | Date | Features |
|---------|------|----------|
| v0.1.0 | 2025-01-24 | Initial release: text, password, confirm, select, multiselect, spinner, log, intro/outro/cancel, group |
| v0.1.1 | 2026-01-24 | Dependabot, examples structure, CLI entry point |

---

## Contributing

1. Check existing issues for the feature you want to implement
2. Create a plan in `.plans/` directory
3. Implement with tests
4. Submit PR referencing the issue

---

## References

- [clack repository](https://github.com/bombshell-dev/clack)
- [@clack/prompts documentation](https://github.com/bombshell-dev/clack/tree/main/packages/prompts)
- [@clack/core documentation](https://github.com/bombshell-dev/clack/tree/main/packages/core)
