import collections.abc

import cobe.ueid


class ModelError(Exception):
    """Raised when validating invalid :class:`Model`s."""


class Identifier:
    """Identifier for a foreign entity.

    A foreign entity is one whose state is not provided by *this* application.
    This class allows relations to be formed between entities provided by
    this application (:class:`Update`) and those which are not. For example,
    entities that are generated by a source on a different host.

    Instances of this class provide a mapping-like interface, where the
    identifying attributes of the entity can be accessed like a dictionary.
    Identifying attributes are entity attributes that are used to determine
    its Unique Entity Identifier. In order to accurately relate an entity to
    a forieng entity all identifying attributes for the given type must be
    set to appropriate values.

    Identifiers can be compared to each other. Identifiers are equal when
    they have the same type and attributes. Additionally, identifiers can
    be compared to satic :class:`cobe.UEID` objects, if the identifier,
    in its current state, would result in an equivalent UEID being generated.

    :param type: The type of the foreign entity. Note that this will be
        coerced to a string.
    :type type: str
    """

    def __init__(self, type):
        self.type = type
        self._attributes = {}

    def __iter__(self):
        """Iterate over the identifier's attributes."""
        yield from self._attributes

    def __len__(self):
        return len(self._attributes)

    def __eq__(self, other):
        if isinstance(other, Identifier):
            if set(self) != set(other):
                return False
            return self.type == other.type and all(
                self[attribute] == other[attribute] for attribute in self)
        elif isinstance(other, cobe.ueid.UEID):
            try:
                return self.ueid() == other
            except cobe.ueid.UEIDError:
                return False
        return False

    def __getitem__(self, attribute):
        return self._attributes[attribute]

    def __setitem__(self, attribute, value):
        self._attributes[attribute] = value

    def __delitem__(self, attribute):
        del self._attributes[attribute]

    def ueid(self):
        """Calculate the UEID from the identifying attributes."""
        attributes = Attributes()
        for name, value in self._attributes.items():
            attributes[name].set(value)
            attributes[name].traits.add('entity:id')
        return cobe.ueid.UEID.from_attributes(self.type, attributes)


class Attribute:
    """An update of an entity's attribute.

    Objects of this class represent an update to an entity's update.
    Instances should not be created directly. Instead,
    :attr:`Update.attributes` should be used.

    Each attribute can be in one of three states: *set*, *not set* or
    *deleted*. When an attribute is set it has a corresponding value which
    will be used as the new value of the attribute when sent from a
    :class:`cobe.Source`. In addition to the value, any number of traits
    may also be attached to the attribute. Traits are printable strings
    wich can be used to convey additional meaning to the attribute value.
    For example, to mark the attribute as being a metric either the
    ``metric:counter`` or ``metric:guage`` traits may be added.

    Alternatively, when an attribute is *deleted* both the value and the
    traits are ignored completely. When the attribute is sent as part of
    an entity :class:`Update` the attribute will be removed from the
    complete entity.

    Attributes which are *not set* do not get sent at all when their
    corresponding entity :class:`Update` does.

    When a Unique Entity Identifier (UEID) is calculated for an entity
    update, a subset of the update's attributes (:attr:`Update.attributes`)
    are used. To mark an attribute as contributing to the unique-ness of
    an entity the ``entity:id`` trait should be added:

    .. code-block:: python

        update = cobe.Update('Extension:Foo')
        update.attributes['bar'].set('snowflake')
        update.attributes['bar'].traits.add('entity:id')

    :cvar NOT_SET: A special opaque object that is used as the attribute's
        value to indicate that it's not set.
    :cvar DELETE: A special opaque object that is used as the attribute's
        value to indicate that it's not set.

    :ivar traits: A set of strings which signify the traits for the
        attribute. It may be empty.
    """

    NOT_SET = object()
    DELETE = object()

    def __init__(self):
        self._value = Attribute.NOT_SET
        self.traits = set()

    def is_set(self):
        """Check if the attribute has a value set."""
        return (self._value is not Attribute.NOT_SET
                and self._value is not Attribute.DELETE)

    def is_not_set(self):
        """Check if the attribute is not set."""
        return self._value is Attribute.NOT_SET

    def is_deleted(self):
        """Check if the attribute is to be deleted."""
        return self._value is Attribute.DELETE

    def is_identifier(self):
        """Check if the attribute is an identifying attribute."""
        return 'entity:id' in self.traits

    @property
    def value(self):
        """Get the value set for this attribute.

        To set the value, :meth:`set` should be used.

        :returns: The last value set via :meth:`set`. If the attribute is
            currently not set (e.g. :meth:`is_not_set` returns ``True``)
            then the return value is :attr:`Attribute.NOT_SET`. If the
            attribute is deleted then the return value is
            :attr:`Attribute.DELETE`.
        """
        return self._value

    def set(self, value):
        """Set a new value for the attribute."""
        self._value = value

    def unset(self):
        """Mark the attribute as having no value set.

        This is effectively the same as if the attribute had never been
        created. This also clears all traits that have been set.
        """
        self.traits.clear()
        self._value = Attribute.NOT_SET

    def delete(self):
        """Mark the attribute as being deleted.

        When an attribute is deleted the previous value and all previous
        traits are cleared.

        Setting a new value for the attribute (or unsetting it
        via :meth:`unset`) will result in the attribute no longer being
        marked for deletion.
        """
        self.traits.clear()
        self._value = Attribute.DELETE


class Attributes(collections.abc.Mapping):
    """A collection of entity update attributes.

    Attribute are keyed against their name as a string which maps to a
    :class:`Attribute` object.

    Instances of this class behave like a default dictionary in that
    accessing an attribute by name for the first time will actually
    create the corresponding :class:`Attribute` object. It is this
    :class:`Attribute` object that is used to perform the actual
    setting and deleting of attributes.

    Also note that instances of this class implement the full mapping API,
    as per :class:`collections.abc.Mapping`. However, the special methods
    that implement this API, such as :meth:`__iter__`, :meth:`__len__` and
    :meth:`__contains__`, only consider attributes that are either set or
    deleted. Those which are *not set* do not count, for example:

    .. code-block:: python

        update = cobe.Update()
        foo = update.attributes['foo']
        assert foo.is_not_set()
        assert len(update.attributes) == 0
        assert 'foo' not in update.attributes
        foo.set('bar')
        assert len(update.attributes) == 1
        assert 'foo' in update.attributes

    Attributes are comparable. Two collections of attributes are considered
    equivalent when they both have the exact same attributes with the same
    values set and deleted.
    """

    def __init__(self):
        self._attributes = {}  # str : Attribute

    def __eq__(self, other):
        if not isinstance(other, Attributes):
            return False
        if list(self) != list(other):
            return False
        return all(self[attribute].value
            == other[attribute].value for attribute in self)

    def __len__(self):
        return sum(1 for _ in self)

    def __iter__(self):
        for name, attribute in self._attributes.items():
            if not attribute.is_not_set():
                yield name

    def __getitem__(self, name):
        """Get an attribute by name.

        This never raises a :exc:`KeyError`.

        :param name: The name of the attribute to get.
        :type name: str
        """
        if name not in self._attributes:
            self._attributes[name] = Attribute()
        return self._attributes[name]


class Update:
    """Local entity update.

    Instances of this class represent an update to an entity. Each
    entity has a type, the time-stamp of the update, a user-facing label,
    a time-to-live and any number of attributes.

    Entities are uniquely identified by a combination of their type and
    *identifying attributes* combined. Identifying attributes are those
    which possess the ``entity:id`` trait.

    The entity update's type must be a *printable*, as in, it must be a
    Unicode string which doesn't contain characters which have the general
    category ``Z`` (separators such as whitespace) or ``C`` ("other",
    including control characters like line feeds). As well as this, the
    type may not contain a "/" (``SOLIDUS``).

    The timestamp of the update indicates *when* the update was created.
    If this is left as its default, ``None``, then no timestamp value is
    sent. If creating an entity for a past or future time, or a more
    accurate time can be provided then it the timestamp should be set
    explicitly.

    A label may optionally be a short string and help a human consumer of
    the entity to identify it. What makes an appropriate label depends on
    the entity type; for example: a ``Process`` entity may use the name of
    the process it represents. By default, the label defaults to ``None``
    which means that it won't be sent along with the rest of the update.
    Because of this, it means that any previously set label will continue
    to be used which may not be accurate.

    The time-to-live of the entity update indicates how long, in seconds,
    the state provided by this update should be considered valid. After
    the given number of seconds is exceeded the entity is considered as
    being stale. Therefore, the time-to-live should be an approximation
    of how long is expected between updates plus an additional margin of
    error. Once an entity is expired it may be garbage collected, but the
    time between expiration and garbage collection is not necessarily
    known by the source.

    In addition to the time-to-live, entity updates can be used to mark
    entities as no longer being in existence. Entities should be marked
    as such when the thing represented by the entity no longer exists.
    Doing this allows the consumer of the entity to indicate that the
    *thing* existed previously, but not anymore. Note that the existence
    of an entity is independent from the time-to-live; an entity may be
    marked as no longer existing but not yet expired. By default an update
    is assumed to represent an object that *does* exist.

    :ivar str type: The type of the entity being updated.
    :ivar datetime.datetime timestamp: The timestamp for the update.
        If set to ``None`` then the timestamp is not set.
    :ivar str label: A short, human readable label for the entity. If
        ``None`` then no label is sent.
    :ivar float ttl: Time-to-live for the update in seconds. If ``None``
        then the time-to-live is not sent.
    :ivar bool exists: Indicates whether the update represents an
        object that current exists.
    :ivar Attributes attributes: A collection of entities for the update.

    :param type: See :attr:`type`.
    :type type: str
    """

    def __init__(self, type):
        self.type = type
        self.timestamp = None
        self.label = None
        self.ttl = None
        self.exists = True
        self.attributes = Attributes()

    def __eq__(self, other):
        if isinstance(other, Update):
            if set(self.attributes) != set(other.attributes):
                return False
            return (
                self.type == other.type
                and self.timestamp == other.timestamp
                and self.label == other.label
                and self.ttl == other.ttl
                and self.exists == other.exists
                and self.attributes == other.attributes
            )
        return False

    def ueid(self):
        return cobe.ueid.UEID.from_attributes(self.type, self.attributes)


class Model:
    """Topological model of entities and relationships.

    Specifically instances of this class represent a sub-graph of the
    complete topological model at a point in time. The model can be
    serialised and sent to a receiver via a :class:`cobe.Source`.

    Note that methods such as :meth:`relate` do not raise exceptions.
    Instead the would-be error is deferred until :meth:`validate` is called.
    All models are validated before being sent by a :class:`cobe.Source`.
    """

    _FOREIGN_TYPES = (Identifier, cobe.ueid.UEID)

    def __init__(self):
        self._entities = []
        self._relationships = []
        self._errors = []

    def __len__(self):
        return sum(1 for _ in self)

    def __iter__(self):
        """Iterate over all entities in the model.

        This creates a generator of entity updates and their relationships
        as contained within the model. The updates are ordered chronologically
        by their timestamps. For updates with the same timestamp are
        further sorted by the order they were added to the model. When an
        update does not provide a timestamp -- a.k.a. :attr:`Update.timestamp`
        is ``None`` -- it is sorted by the order it was added to the model
        *after* those which are sorted chronologically.

        Duplicate equivalent relationships are ignored.

        :returns: An iterator of three-tuples containing, each containing
            an :class:`Update` followed by two tuples containing the parents
            and children of the entity respectively.
        """
        key = (lambda update:
            float('inf') if update.timestamp is None else update.timestamp)
        for entity in sorted(self._entities, key=key):
            parents = []
            children = []
            for relationship in self._relationships:
                if entity not in relationship:
                    continue
                if relationship[0] is entity:
                    if relationship[1] not in children:
                        children.append(relationship[1])
                if relationship[1] is entity:
                    if relationship[0] not in parents:
                        parents.append(relationship[0])
            yield entity, tuple(parents), tuple(children)

    def add(self, entity):
        """Add an entity model.

        The given entity will not be related to any other entities.
        Typically :meth:`relate` will be more useful, which also adds
        entities to the model.

        If the entity object already exists in the model or it is a
        :class:`Identifier` or a :class:`cobe.UEID` then this method
        does nothing.

        :param entity: The entity to add to the model.
        :type entity: cobe.Update
        """
        if (not isinstance(entity, self._FOREIGN_TYPES)
                and all(entity is not modeled for modeled in self._entities)):
            self._entities.append(entity)

    def relate(self, *, parent, child):
        """Relate entities together.

        This adds a relationship to the model which relates a parent
        entity to a child. This also implicitly adds the given entities
        to the model in the same manner as :meth:`add`.

        Relations to foreign entities, e.g. those not provided by this
        source, can by created by passing in an :class:`Identifier` or
        :class:`cobe.UEID` object as one of the related entities. Note
        that it is not possible to relate two foreign entities together.
        Any attempt to do so will result in the model being unmodified
        and any subsequent calls to :meth:`validate` to raise a
        :exc:`ModelError` exception.

        .. note::
            When the given entities are added to the model the parent is
            added before the child. This affects the order of entity
            updates when iterating over a model. See :meth:`__iter__`.
        """
        if (isinstance(parent, self._FOREIGN_TYPES)
                and isinstance(child, self._FOREIGN_TYPES)):
            self._errors.append('Model contains a relationship '
                                'between two foreign entity identifiers')
            return
        self.add(parent)
        self.add(child)
        self._relationships.append((parent, child))

    def validate(self):
        """Validate the model for errors.

        This checks the model for a number of flaws that would prevent it
        from being sent by a :class:`cobe.Source`.

        Specifically, updates which have the same UEID or if any attempts
        were made to relate two foreign entities together will result
        in an exception. Also, if the model contains objects whicha are
        :class:`Update` instances then a :exc:`ModelError` is also raised.

        :raises ModelError: If the model contains invalid updates
            or relationships
        """
        ueids = set()
        for error in self._errors:
            raise ModelError(error)
        for update in self._entities:
            if not isinstance(update, Update):
                raise ModelError(
                    'Model contains an update {!r} which is '
                    'of type {!r} but {!r} is expected'.format(
                        update,
                        type(update).__name__,
                        Update.__name__,
                    ))
            try:
                ueid = update.ueid()
            except cobe.ueid.UEIDError:
                pass  # We only care about conflicts here
            else:
                if ueid in ueids:
                    raise ModelError('Model contains updates with '
                                     'the same UEID: {}'.format(ueid))
                ueids.add(ueid)
