import re
from dataclasses import dataclass
from pathlib import Path

from src.entities.block import Block
from src.entities.block_pattern import BlockPattern
from src.entities.file import File
from src.enums.block_type import BlockType

BLOCK_START_PATTERN = re.compile(r"^[a-zA-Z_@#\(\)\"\'][a-zA-Z0-9_]*")

IMPORT_PATTERN = BlockPattern(name=BlockType.IMPORT, pattern=re.compile(r"^(import|from)\s+"))
FUNCTION_PATTERN = BlockPattern(name=BlockType.FUNCTION, pattern=re.compile(r"^(async\s+def|def)\s+([a-zA-Z0-9_]*)"))
CLASS_PATTERN = BlockPattern(name=BlockType.CLASS, pattern=re.compile(r"^class\s+([a-zA-Z0-9_]*)\s*[:\(]"))
VALUE_PATTERN = BlockPattern(name=BlockType.VALUE, pattern=re.compile(r"^([a-zA-Z_][a-zA-Z0-9_]*)\s*="))
COMMENT_PATTERN = BlockPattern(name=BlockType.COMMENT, pattern=re.compile(r"^\s*(#|\"\"\"|\'\'\')"))


@dataclass(frozen=True)
class LoadFileService:
    file_path: Path

    class ParseError(Exception):
        pass

    def execute(self) -> File:
        """Read the file and split it into classes, functions, and others"""
        assert self.file_path.is_file(), f"Error: File '{self.file_path}' does not exist."
        with self.file_path.open(mode="r") as file:
            lines = file.readlines()
        blocks: list[Block] = []
        codes: list[str] = []
        in_comment_or_decorator = False
        in_import = False
        for line in lines:
            # If the first line is not blank, consider it as a block delimiter
            if BLOCK_START_PATTERN.match(line):
                # Split immediately if not inside a comment or decorator
                if not in_comment_or_decorator and not in_import:
                    if codes:
                        blocks.append(self._generate_block(codes=codes))
                        codes = []
                if IMPORT_PATTERN.pattern.match(line):
                    in_import = True
                elif line.startswith("@") or COMMENT_PATTERN.pattern.match(line):
                    in_comment_or_decorator = True
                else:
                    # the comment or decorator ends
                    in_comment_or_decorator = False
                    in_import = False
            codes.append(line)
        else:
            if codes:
                blocks.append(self._generate_block(codes=codes))
        # Check if the total number of lines in the split blocks matches the original number of lines
        assert len(lines) == sum([len(block.codes) for block in blocks])
        return File(path=self.file_path, blocks=blocks)

    def _generate_block(self, codes: list[str]) -> Block:
        for code in codes:
            for pattern in [IMPORT_PATTERN, CLASS_PATTERN, FUNCTION_PATTERN, VALUE_PATTERN]:
                if match := pattern.pattern.match(code):
                    name = match and match.groups()[-1]
                    if not name:
                        raise self.ParseError(f"{code=} {match=}")
                    return Block(
                        codes=codes,
                        name=name,
                        type=pattern.name,
                    )
        else:
            raise self.ParseError(f"Block not found in {codes=}")
