from dataclasses import dataclass
from pathlib import Path

from src.entities.block import Block
from src.entities.file import File
from src.patterns import (
    BLOCK_START_PATTERN,
    CLASS_PATTERN,
    COMMENT_PATTERN,
    FUNCTION_PATTERN,
    IMPORT_PATTERN,
    VALUE_PATTERN,
)


@dataclass(frozen=True)
class LoadFileService:
    file_path: Path

    class ParseError(Exception):
        pass

    def execute(self) -> File:
        """Read the file and split it into classes, functions, and others"""
        assert self.file_path.is_file(), f"Error: File '{self.file_path}' does not exist."
        with self.file_path.open(mode="r") as file:
            lines = file.readlines()
        blocks: list[Block] = []
        codes: list[str] = []
        in_import, in_decorator, in_comment = False, False, False
        for line in lines:
            # If the first line is not blank, consider it as a block delimiter
            if BLOCK_START_PATTERN.match(line):
                # Split immediately if not inside a comment or decorator
                if not in_comment and not in_decorator and not in_import:
                    if codes:
                        blocks.append(self._generate_block(codes=codes))
                        codes = []
                if COMMENT_PATTERN.pattern.match(line):
                    in_import, in_decorator, in_comment = False, False, True
                elif IMPORT_PATTERN.pattern.match(line):
                    in_import, in_decorator, in_comment = True, False, False
                elif line.startswith("@"):
                    in_import, in_decorator, in_comment = False, True, False
                else:
                    in_import, in_decorator, in_comment = False, False, False
            codes.append(line)
        else:
            if codes:
                blocks.append(self._generate_block(codes=codes))
        # Check if the total number of lines in the split blocks matches the original number of lines
        assert len(lines) == sum([len(block.codes) for block in blocks])
        return File(path=self.file_path, blocks=blocks)

    def _generate_block(self, codes: list[str]) -> Block:
        print("@@@@@@@@@@")
        for code in codes:
            print(code)
        print("@@@@@@@@@@")
        for code in codes:
            for pattern in [IMPORT_PATTERN, CLASS_PATTERN, FUNCTION_PATTERN, VALUE_PATTERN]:
                if match := pattern.pattern.match(code):
                    name = match and match.groups()[-1]
                    if not name:
                        raise self.ParseError(f"{code=} {match=}")
                    return Block(
                        codes=codes,
                        name=name,
                        type=pattern.name,
                    )
        else:
            raise self.ParseError(f"Block not found in {codes=}")
