"""Basic tests to check that the main examples work."""

import os
import pytest
import python_codon_tables as pct


mock_table_path = os.path.join("tests", "data", "table_newline.csv")


def test_basics():

    # LOAD ONE TABLE BY NAME
    table = pct.get_codons_table("b_subtilis_1423")
    assert table["T"]["ACA"] == 0.4
    assert table["*"]["TAA"] == 0.61

    # LOAD ALL TABLES AT ONCE
    codon_tables = pct.get_all_available_codons_tables()
    assert codon_tables["c_elegans_6239"]["L"]["CTA"] == 0.09


def test_download_codon_table(tmpdir):
    table = pct.download_codons_table(taxid=316407)
    assert table["*"]["UAG"] == 0.07
    target = os.path.join(str(tmpdir), "test.csv")
    table = pct.download_codons_table(taxid=316407, target_file=target)


def test_readme_example():
    table = pct.get_codons_table("b_subtilis_1423")
    assert table["T"]["ACA"] == 0.4
    assert table["*"]["TAA"] == 0.61

    # LOAD ALL TABLES AT ONCE
    codons_tables = pct.get_all_available_codons_tables()
    assert codons_tables["c_elegans_6239"]["L"]["CTA"] == 0.09

    # GET A TABLE DIRECTLY FROM THE INTERNET
    table = pct.download_codons_table(taxid=316407)
    assert table["*"]["TGA"] == 0.29

    with pytest.raises(RuntimeError):
        table = pct.download_codons_table(taxid=000000000)  # does not exist


def test_readme_example():
    table = pct.get_codons_table("b_subtilis_1423")
    assert table["T"]["ACA"] == 0.4
    assert table["*"]["TAA"] == 0.61

    # LOAD ALL TABLES AT ONCE
    codons_tables = pct.get_all_available_codons_tables()
    assert codons_tables["c_elegans_6239"]["L"]["CTA"] == 0.09

    # GET A TABLE DIRECTLY FROM THE INTERNET
    table = pct.download_codons_table(taxid=316407)
    assert table["*"]["UGA"] == 0.29


def test_get_codons_table():
    for table_name in (1423, "1423", "b_subtilis", "b_subtilis_1423"):
        table = pct.get_codons_table(table_name)
        assert table["T"]["ACA"] == 0.4
        assert table["*"]["TAA"] == 0.61


def test_replace_U_by_T():
    table = pct.get_codons_table("b_subtilis_1423", replace_U_by_T=False)
    assert table["*"]["UAA"] == 0.61


def test_csv_string_to_codons_dict():
    with open(mock_table_path, "r") as f:
        codon_dict = pct.csv_string_to_codons_dict(f.read())
    assert type(codon_dict) == dict
