# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['config']

package_data = \
{'': ['*']}

extras_require = \
{'toml': ['toml>=0.10.0,<0.11.0'], 'yaml': ['pyyaml>=3.13,<4.0']}

setup_kwargs = {
    'name': 'python-configuration',
    'version': '0.1.0',
    'description': '',
    'long_description': '# python-configuration\n> A library to load configuration parameters\n\nThis library is intended as a helper mechanism to load configuration files\nhierarchically.  Current format types are:\n* Python files\n* Dictionaries\n* Environment variables\n* Filesystem paths\n* JSON files\n* INI files\n\nand optionally\n* YAML files\n* TOML files\n\n## Installing\n\nTo install the library:\n```shell\npip install python-configuration\n```\n\nTo include the optional TOML and/or YAML loaders, install the optional\ndependencies `toml` and ` yaml`. For example,\n```shell\npip install python-configuration[toml, yaml]\n```\n\n## Getting started\n\nThis library converts the config types above into dictionaries with \ndotted-based keys. That is, given a config `cfg` from the structure\n```python\n{\n    \'a\': {\n        \'b\': \'value\'\n    }\n}\n```\nwe are able to refer to the parameter above as any of \n```python\ncfg[\'a.b\']\ncfg[\'a\'][\'b\']\ncfg[\'a\'].b\ncfg.a.b\n```\nand extract specific data types such as dictionaries:\n```python\ncfg[\'a\'].as_dict == {\'b\': \'value\'}\n```\nThis is particularly useful in order to isolate group parameters.\nFor example, with the JSON configuration\n```json\n{\n  "database.host": "something",\n  "database.port": 12345,\n  "database.driver": "name",\n  "app.debug": true,\n  "app.environment": "development",\n  "app.secrets": "super secret",\n  "logging": {\n    "service": "service",\n    "token": "token",\n    "tags": "tags"\n  }\n}\n```\none can retrieve the dictionaries as \n```python\ncfg.database.as_dict()\ncfg.app.as_dict()\ncfg.logging.as_dict()\n```\nor simply as \n```python\ndict(cfg.database)\ndict(cfg.app)\ndict(cfg.logging)\n```\n## Configuration\nThere are two general types of objects in this library. The first one is the `Configuration`,\nwhich represents a single config source.  The second is a `ConfigurationSet` that allows for\nmultiple `Configuration` objects to be specified.\n\n### Single Config\n\n#### Python Files\nTo load a configuration from a Python module, the `config_from_python` can be used.\nThe first parameter must be a Python module and can be specified as an absolute path to the Python file or as an importable module.\n\nOptional parameters are the `prefix` and `separator`.  The following call \n```python\nconfig_from_python(\'foo.bar\', prefix=\'CONFIG\', separator=\'__\')\n```\nwill read every variable in the `foo.bar` module that starts with `CONFIG__` and replace\nevery occurrence of `__` with a `.`. For example,\n```python\n# foo.bar\nCONFIG__AA__BB_C = 1\nCONFIG__AA__BB__D = 2\nCONF__AA__BB__D = 3\n```\nwould result in the configuration\n```python\n{\n    \'aa.bb_c\': 1,\n    \'aa.bb.d\': 2,\n}\n```\nNote that the single underscore in `BB_C` is not replaced and the last line is not\nprefixed by `CONFIG`. \n\n#### Dictionaries\nDictionaries are loaded with `config_from_dict` and are converted internally to a \nflattened `dict`. \n```python\n{\n    \'a\': {\n        \'b\': \'value\'\n    }\n}\n```\nbecomes\n```python\n{\n    \'a.b\': \'value\'\n}\n```\n\n#### Environment Variables\nEnvironment variables starting with `prefix` can be read with `config_from_env`:\n```python\nconfig_from_env(prefix, separator=\'_\')\n```\n\n#### Filesystem Paths\nFolders with files named as `xxx.yyy.zzz` can be loaded with the `config_from_path` function.  This format is useful to load mounted\nKubernetes [ConfigMaps](https://kubernetes.io/docs/tasks/configure-pod-container/configure-pod-configmap/#populate-a-volume-with-data-stored-in-a-configmap)\nor [Secrets](https://kubernetes.io/docs/tasks/inject-data-application/distribute-credentials-secure/#create-a-pod-that-has-access-to-the-secret-data-through-a-volume).\n\n#### JSON, INI, YAML, TOML\nJSON, INI, YAML, TOML files are loaded respectively with\n`config_from_json`,\n`config_from_ini`,\n`config_from_yaml`, and\n`config_from_toml`.\nThe parameter `read_from_file` controls\nwhether a string should be interpreted as a filename.\n\n###### Caveats\nIn order for `Configuration` objects to act as `dict` and allow the syntax\n`dict(cfg)`, the `keys()` method is implemented as the typical `dict` keys.\nIf `keys` is an element in the configuration `cfg` then the `dict(cfg)` call will fail.\nIn that case, it\'s necessary to use the `cfg.as_dict()` method to retrieve the\n`dict` representation for the `Configuration` object.\n\nThe same applies to the methods `values()` and `items()`.\n \n\n### Configuration Sets\nConfiguration sets are used to hierarchically load configurations and merge\nsettings. Sets can be loaded by constructing a `ConfigurationSet` object directly or\nusing the simplified `config` function.\n\nTo construct a `ConfigurationSet`, pass in as many of the simple `Configuration` objects as needed:\n```python\ncfg = ConfigurationSet(\n    config_from_env(prefix=PREFIX),\n    config_from_json(path, read_from_file=True),\n    config_from_dict(DICT),\n)\n```\nThe example above will read first from Environment variables prefixed with `PREFIX`, \nand fallback first to the JSON file at `path`, and finally use the dictionary `DICT`.\n\nThe `config` function simplifies loading sets by assuming some defaults.\nThe example above can also be obtained by\n```python\ncfg = config(\n    (\'env\', PREFIX),\n    (\'json\', path, True),\n    (\'dict\', DICT),\n)\n```\nor, even simpler if `path` points to a file with a `.json` suffix:\n```python\ncfg = config(\'env\', path, DICT, prefix=PREFIX)\n```\nThe `config` function automatically detects the following:\n* extension `.py` for python modules\n* dot-separated python identifiers as a python module (e.g. `foo.bar`)\n* extension `.json` for JSON files\n* extension `.yaml` for YAML files\n* extension `.toml` for TOML files\n* extension `.ini` for INI files\n* filesystem folders as Filesystem Paths\n* the strings `env` or `environment` for Environment Variables\n\n###### Caveats\nAs long as the data types are consistent across all the configurations that are\npart of a `ConfigurationSet`, the behavior should be straightforward.  When different\nconfiguration objects are specified with competing data types, the first configuration to\ndefine the elements sets its datatype. For example, if in the example above \n`element` is interpreted as a `dict` from environment variables, but the \nJSON file specifies it as anything else besides a mapping, then the JSON value will be\ndropped automatically. \n\n## Developing\n\nTo develop this library, download the source code and install a local version.\n\n\n## Features\n\n* Load multiple configuration types\n* Hierarchical configuration\n* Ability to override with environment variables\n* Merge parameters from different configuration types\n\n## Contributing\n\nIf you\'d like to contribute, please fork the repository and use a feature\nbranch. Pull requests are welcome.\n\n## Links\n\n- Repository: https://github.com/tr11/python-configuration\n- Issue tracker: https://github.com/tr11/python-configuration\n\n## Licensing\n\nThe code in this project is licensed under MIT license.\n',
    'author': 'Tiago Requeijo',
    'author_email': 'tiago.requeijo.dev@gmail.com',
    'url': 'https://github.com/tr11/python-configuration',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.4,<4.0',
}


setup(**setup_kwargs)
