# Python Cowsay

[![PyPi](https://img.shields.io/pypi/v/python-cowsay?label=PyPi&labelColor=%23ffd343&color=%230073b7)](https://pypi.org/project/python-cowsay/)
[![License](https://img.shields.io/badge/GPL--3.0-green?label=license)](https://codeberg.org/james-ansley/cowsay/src/branch/main/LICENSE)

A rewrite of cowsay in python. Allows for parsing of existing `.cow` files.

## Install

    pip install python-cowsay

## Usage

### CLI

Installing `python-cowsay` adds `cowsay` and `cowthink` terminal programs.
These work similarly to the classic cowsay CLIs.

See `cowsay --help` or `cowthink --help` for more info.

```
> cowsay --help 
usage: cowsay [-h] [-e eye_string] [-f cowfile] [-l] [-n] [-T tongue_string]
↪ [-W column] [-b] [-d] [-g] [-p] [-s] [-t] [-w] [-y] [--random] [message]

Generates an ASCII image of a cow saying the given text

positional arguments:
  message           The message to include in the speech bubble. If not given, 
                    stdin is used instead.

options:
  -h, --help        show this help message and exit
  -e eye_string     An eye string. This is ignored if a preset mode is given
  -f cowfile        Either the name of a cow specified in the COWPATH, or a 
                    path to a cowfile (if provided as a path, the path must 
                    contain at least one path separator)
  -l                Lists all cows in the cow path and exits
  -n                If given, text in the speech bubble will not be wrapped
  -T tongue_string  A tongue string. This is ignored if a preset mode is given
  -W column         Width in characters to wrap the speech bubble (default 40)
  --random          If provided, picks a random cow from the COWPATH.
                    Is superseded by the -f option

Mode:
  There are several out of the box modes which change the appearance of the cow.
  If multiple modes are given, the one furthest down this list is selected

  -b                Borg
  -d                dead
  -g                greedy
  -p                paranoid
  -s                stoned
  -t                tired
  -w                wired
  -y                young
```

### Programmatic Execution

The classic cowsay text can be generated by the `cowsay` or `cowthink`
functions:

```python
from cowsay import cowsay

message = """
The most remarkable thing about my mother is that for thirty years she served
the family nothing but leftovers.  The original meal has never been found.
		-- Calvin Trillin
""".strip()
print(cowsay(message))
```

Will yield:

```text
 __________________________________________ 
/ The most remarkable thing about my       \
| mother is that for thirty years she      |
| served the family nothing but leftovers. |
| The original meal has never been found.  |
|                                          |
\ -- Calvin Trillin                        /
 ------------------------------------------ 
        \   ^__^
         \  (oo)\_______
            (__)\       )\/\
                ||----w |
                ||     ||
```

The parameters for these functions are:

- `message` – a string to wrap in the text bubble
- `cow='default'` – the name of the cow (valid names from `list_cows`)
- `preset=None` – the original cowsay presets: `-bgpstwy`
- `eyes=Option.eyes` – A custom eye string
- `tongue=Option.tongue` – A custom tongue string
- `width=40` – The width of the text bubble
- `wrap_text=True` – Whether text should be wrapped in the bubble
- `cowfile=None` – A custom string representing a cow

### Other Functions

The available builtin cows can be found with `list_cows`. A cow can be chosen
randomly from this list with `get_random_cow`.

### Using Your Own Cows

A custom `.cow` file can be parsed using the `read_dot_cow` function which takes
a `TextIO` stream. I.e., You can either create a `TextIO` from a string or read
a file.

The `read_dot_cow` will look for the first heredoc in the steam and extract the
heredoc contents. If no heredoc exists, the whole stream is used instead. Escape
characters are then escaped. The default escape characters can be changed by
passing in an optional `escape` dictionary parameter mapping escape codes to
their chars.

For example:

```python
from io import StringIO

from cowsay import read_dot_cow, cowthink

cow = read_dot_cow(StringIO("""
$the_cow = <<EOC;
         $thoughts
          $thoughts
           ___
          (o o)
         (  V  )
        /--m-m-
EOC
"""))
message = """
Nothing is illegal if one hundred businessmen decide to do it.
        -- Andrew Young
""".strip()
print(cowthink(message, cowfile=cow))
```

Will yield:

```text
 ___________________________________ 
( Nothing is illegal if one hundred )
( businessmen decide to do it.      )
(                                   )
( -- Andrew Young                   )
 ----------------------------------- 
         o
          o
           ___
          (o o)
         (  V  )
        /--m-m-
```

## Parsing `.cow` Files

`.cow` files are just files containing Perl code. This causes some issues for
more _advanced_ `.cow` files that do things like _define additional variables_
that get used in the cow heredoc. Most notably, this happens when using tools
like [Charc0al's cowsay file converter](https://charc0al.github.io/cowsay-files/converter/).

`python-cowsay` does not fully support Perl `.cow` files but has accounted for
this one case. When parsing `.cow` files, `python-cowsay` will look for any
string variable declarations at the start of each line and the cow heredoc.
If any string variables are found, these are inlined in the resulting cow.

## Changing the Cows

`python-cowsay` will attempt to retrieve a `COWPATH` environment variable, if
found, the path this variable references will be used instead of the default
that is installed with `python-cowsay`.

If you wish to change the default cows, set the `COWPATH` environment variable
in your shell profile:

```
export COWPATH=path/to/cows
```

## Full-Width Characters

A bit of a hack at the moment, but if any full-width characters are found in the
message string, ***all*** characters in the thought bubble are converted to
full-width. For example:

```text
　＿＿＿＿＿＿＿＿＿＿＿＿　
（　喵喵喵。我是一只猫。　）
　－－－－－－－－－－－－　
   o
    o

     |\_/|
     |o o|__
     --*--__\
     C_C_(___)
```

This works fine when all characters in the message are full-width, but does not
work so well when there is a mix of full- and neutral-width characters:

```text
　＿＿＿＿＿＿＿＿＿＿＿＿＿＿＿＿＿　
（　喵喵喵。Ｉ　ａｍ　ａ　ｃａｔ．　）
　－－－－－－－－－－－－－－－－－　
   o
    o

     |\_/|
     |o o|__
     --*--__\
     C_C_(___)
```

Each full-width character still only counts as one character when setting the
text width. For example:

```python
from cowsay import cowthink

message = "喵喵喵。我是一只猫。我想吃鱼和喝牛奶。"
print(cowthink(message, cow="kitten", width=10))
```

Will yield:

```text
　＿＿＿＿＿＿＿＿＿＿＿＿　
（　喵喵喵。我是一只猫。　）
（　我想吃鱼和喝牛奶。　　）
　－－－－－－－－－－－－　
   o
    o

     |\_/|
     |o o|__
     --*--__\
     C_C_(___)
```

## Related projects

- The original
  [cowsay repository](https://github.com/tnalpgge/rank-amateur-cowsay)
- [Cowexcept](https://github.com/James-Ansley/cowexcept) to jazz up Python
  exceptions
- [fortune-python](https://github.com/James-Ansley/fortune) — a rewrite of
  fortune in Python
- This [collection of additional cow files](https://github.com/paulkaefer/cowsay-files/tree/main)

## Notes

The cow files that are installed by default in this package are taken from
[the original cowsay repository](https://github.com/tnalpgge/rank-amateur-cowsay)

The cows provided in this package, save for minor edits to resolve issues
with parsing cow files, are otherwise provided as they were in the original
cowsay repository.
