"""
This script is called when a Bookshelf release is triggered on GitHub.

It will download all "bs.*.zip" files from the release and extract them to the "datapack" folder.
It will then update the "bookshelf.py" file with the new version information.
"""
# Imports
import os
import re
from copy import deepcopy

import requests
import stouputils as stp

# Constants
ROOT: str = stp.get_root_path(__file__, go_up=1)
GITHUB_REPO: str = "https://github.com/mcbookshelf/bookshelf/releases"
API_URL: str = "https://api.github.com/repos/mcbookshelf/bookshelf/releases/latest"
DESTINATION_FOLDER: str = f"{ROOT}/src/python_datapack/dependencies"
CONFIG_PATH: str = f"{DESTINATION_FOLDER}/bookshelf_config.json"
DEPS_TO_UPDATE: str = "src/python_datapack/dependencies/bookshelf.py"
MODULE_TEMPLATE: dict = {"version": [0, 0, 0], "name": "Title Name", "url": GITHUB_REPO, "is_used": False}


# Helper function to parse version string (e.g., "v1.2.3" or "1.2.3")
def parse_version(tag: str) -> list[int]:
	tag = tag.lstrip("v")
	return [int(part) for part in tag.split(".") if part.isdigit()]

# Convert PascalCase or dash-style module names to title format
def format_module_name(module: str) -> str:
    # Remove version suffix
    module = re.sub(r"[-_]?\d+(\.\d+)*(-v\d+(\.\d+)*)?$", "", module)
    # Replace dash/underscore with space, capitalize each word
    return "Bookshelf " + " ".join(word.capitalize() for word in re.split(r"[-_.]", module) if word != "bs")

# Convert PascalCase to snake_case
def pascal_to_snake(name: str) -> str:
	return re.sub(r'(?<!^)(?=[A-Z])', '_', name).lower()

@stp.handle_error()
@stp.measure_time(stp.progress)
def download_latest_release() -> None:
	"""
	Download the wanted files from the latest release on GitHub.

	Then, generate a new "bookshelf.py" file with the new modules looking like the following:

	BOOKSHELF_MODULES: dict[str, dict] = {
		"bs.block_barrel":    { ... },
		"bs.item_tool":       { ... },
		...
	}
	"""
	stp.info(f"Fetching latest release from {API_URL}")
	response: requests.Response = requests.get(API_URL)
	if response.status_code != 200:
		stp.info(f"Error fetching release info: {response.status_code}")
		return
	release_info: dict = response.json()

	tag_name: str = release_info.get("tag_name", "v0.0.0")
	stp.info(f"Latest release tag: {tag_name}")

	modules: dict = {}
	assets: list[dict] = release_info.get("assets", [])
	os.makedirs(f"{DESTINATION_FOLDER}/datapack", exist_ok=True)

	# Filter assets
	assets = [asset for asset in assets if re.match(r"^bs\..*\.zip$", asset.get("name", ""))]

	def download_module(asset: dict) -> tuple[str, dict] | None:
		file_name: str = asset.get("name", "")
		no_extension: str = os.path.splitext(file_name)[0]
		module_name: str = format_module_name(no_extension)
		download_url: str = asset.get("browser_download_url", "")
		if not download_url:
			stp.warning(f"No download URL for {file_name}, skipping.")
			return

		asset_response: requests.Response = requests.get(download_url)
		if asset_response.status_code != 200:
			stp.warning(f"Failed to download {file_name}: HTTP {asset_response.status_code}")
			return

		local_zip_path: str = f"{DESTINATION_FOLDER}/datapack/{module_name}.zip"
		with open(local_zip_path, "wb") as f:
			f.write(asset_response.content)
		stp.debug(f"Downloaded '{file_name}' to '...{local_zip_path[-40:]}'")

		# Extract module version from name (e.g. "bs.health-1.21.5-v3.0.0.zip" -> [3, 0, 0])
		version: list[int] = parse_version(no_extension.split('-')[-1])

		# Prepare module info
		module_key_snake: str = file_name.split('-')[0]
		module_info: dict = deepcopy(MODULE_TEMPLATE)
		module_info["version"] = version
		module_info["name"] = module_name
		return module_key_snake, module_info

	# Multithreading
	results: list[tuple[str, dict]] = stp.multithreading(download_module, assets, max_workers=len(assets))
	modules: dict = dict(sorted(results, key=lambda x: x[0]))

	dumped: str = stp.super_json_dump(modules, max_level=1).replace("false", "False").replace("true", "True")
	file_content: str = f"""
# This file is auto-generated by the release script ({DEPS_TO_UPDATE})
BOOKSHELF_MODULES = {dumped}
"""

	with open(f"{ROOT}/{DEPS_TO_UPDATE}", "w") as f:
		f.write(file_content)
	stp.info(f"Updated '{DEPS_TO_UPDATE}' with new module information.")

	# Update the config file
	with open(CONFIG_PATH, "w") as f:
		stp.super_json_dump({"version": tag_name}, f)
	stp.info(f"Updated '{CONFIG_PATH}' with new version information.")

# Main
if __name__ == "__main__":
	download_latest_release()

