# GitHub Actions 工作原理详解

## 🔍 test.yml 工作原理

### 1. 触发机制

```yaml
on:
  push:
    branches: [main]  # 推送到 main 分支时触发
  pull_request:
    branches: [main]  # PR 到 main 分支时触发
```

**工作流程**:
1. 你执行 `git push origin main`
2. GitHub 检测到 push 事件
3. GitHub 检查仓库中的 `.github/workflows/*.yml` 文件
4. 找到匹配的工作流（`on.push.branches: [main]`）
5. 自动启动工作流执行

### 2. 运行环境

```yaml
runs-on: ubuntu-latest
```

**GitHub 提供的免费虚拟机**:
- **操作系统**: Ubuntu 最新版（目前是 Ubuntu 22.04）
- **虚拟环境**: 全新的 VM，每次运行都是干净的
- **配置**: 2-core CPU, 7GB RAM, 14GB SSD
- **时长限制**: 单个 job 最多 6 小时
- **位置**: GitHub 的云端服务器（Azure）

### 3. 测试矩阵

```yaml
strategy:
  matrix:
    python-version: ['3.9', '3.10', '3.11', '3.12', '3.13']
  fail-fast: false
```

**并行执行**:
- GitHub 会创建 **5 个独立的虚拟机**
- 每个 VM 运行一个 Python 版本
- **并行运行**，不是串行
- `fail-fast: false` 表示一个失败不影响其他的继续

**示意图**:
```
GitHub 检测到 push
    ↓
启动 5 个并行任务
    ├─ VM1: Python 3.9  ─┐
    ├─ VM2: Python 3.10 ─┤
    ├─ VM3: Python 3.11 ─┼─ 同时运行
    ├─ VM4: Python 3.12 ─┤
    └─ VM5: Python 3.13 ─┘
         ↓
    所有任务完成
         ↓
    显示结果（✅ 或 ❌）
```

### 4. 执行步骤详解

#### Step 1: 检出代码
```yaml
- name: 检出代码
  uses: actions/checkout@v4
```
- **作用**: 将你的仓库代码克隆到虚拟机
- **原理**: 类似 `git clone`
- **版本**: `@v4` 是 action 的版本

#### Step 2: 设置 Python
```yaml
- name: 设置 Python ${{ matrix.python-version }} 环境
  uses: actions/setup-python@v5
  with:
    python-version: ${{ matrix.python-version }}
    cache: 'pip'
```
- **作用**: 安装指定版本的 Python
- **cache: 'pip'**: 自动缓存 pip 下载的包，下次运行更快
- **`${{ matrix.python-version }}`**: 变量替换，根据矩阵值动态设置

#### Step 3-4: 显示版本和升级 pip
```yaml
- name: 显示 Python 版本
  run: |
    python --version
    pip --version

- name: 升级 pip
  run: |
    python -m pip install --upgrade pip
```
- **`run:`**: 执行 shell 命令
- **`|`**: YAML 多行字符串

#### Step 5: 安装依赖
```yaml
- name: 安装项目和开发依赖
  run: |
    pip install -e '.[dev]'
```
- **原理**: 
  1. 读取 `pyproject.toml`
  2. 安装项目本身（`-e` 开发模式）
  3. 安装 `[project.optional-dependencies]` 下的 `dev` 依赖
  4. 包括 `pytest` 和 `pytest-cov`

#### Step 6: 运行测试
```yaml
- name: 运行单元测试
  run: |
    pytest tests/ -v --cov=src/debug_helpers --cov-report=term-missing --cov-report=xml
```
- **pytest tests/**: 运行测试
- **--cov=src/debug_helpers**: 生成覆盖率
- **--cov-report=xml**: 生成 XML 格式报告（给 Codecov 用）

#### Step 7: 上传覆盖率（可选）
```yaml
- name: 上传覆盖率报告到 Codecov
  if: success() && matrix.python-version == '3.11'
  uses: codecov/codecov-action@v4
```
- **`if:`**: 条件执行
- **`success()`**: 前面步骤都成功
- **只在 Python 3.11 上传**: 避免重复上传

---

## ⚙️ 需要在 GitHub 上设置什么？

### 必需设置（零配置）

✅ **无需任何设置！**

只需要：
1. 将 `.github/workflows/test.yml` 推送到仓库
2. GitHub 会自动识别并启用

### 可选设置

#### 1. 查看工作流状态

在仓库页面：
- 点击 **"Actions"** 标签
- 可以看到所有工作流的运行历史
- 点击某次运行可以查看详细日志

#### 2. 配置 Codecov（可选）

如果要使用覆盖率报告：

1. 访问 https://codecov.io/
2. 用 GitHub 账号登录
3. 添加你的仓库
4. 获取 `CODECOV_TOKEN`
5. 在 GitHub 仓库添加 Secret:
   - Settings → Secrets and variables → Actions
   - New repository secret
   - Name: `CODECOV_TOKEN`
   - Value: 粘贴 token

**如果不设置**: 覆盖率步骤会被跳过，测试仍然正常运行

#### 3. 保护分支（推荐）

Settings → Branches → Add branch protection rule:
- Branch name pattern: `main`
- ✅ Require status checks to pass before merging
- ✅ Require branches to be up to date before merging
- 选择: `test` (Python Unit Tests)

**效果**: PR 必须通过测试才能合并

---

## 💰 资源和费用

### 免费额度（Public 仓库）

| 项目 | 额度 |
|------|------|
| **运行时长** | ♾️ **无限制** |
| **并发任务** | 20 个 |
| **存储空间** | 500 MB |
| **成本** | 完全免费 |

### 免费额度（Private 仓库）

| 项目 | 免费额度 | 超出后价格 |
|------|----------|-----------|
| **运行时长** | 2,000 分钟/月 | $0.008/分钟 |
| **并发任务** | 20 个 | - |
| **存储空间** | 500 MB | $0.25/GB/月 |

### 本项目的资源消耗

以 **test.yml** 为例：

```
单次运行:
  - 5 个 Python 版本并行
  - 每个版本约 2-3 分钟
  - 并行运行，实际耗时: ~3 分钟
  - 计费时长: 5 × 3 = 15 分钟

每月推送 20 次:
  - 总耗时: 20 × 15 = 300 分钟
  - Public 仓库: 免费
  - Private 仓库: 300/2000，仍在免费额度内
```

### GitHub Actions 运行器

**GitHub 托管的运行器（Runners）**:

| 类型 | 配置 | 位置 |
|------|------|------|
| `ubuntu-latest` | 2-core, 7GB RAM | Azure 云端 |
| `ubuntu-22.04` | 2-core, 7GB RAM | Azure 云端 |
| `windows-latest` | 2-core, 7GB RAM | Azure 云端 |
| `macos-latest` | 3-core, 14GB RAM | Azure 云端 |

**自托管运行器**（可选）:
- 你也可以使用自己的服务器
- 但对于简单项目，GitHub 托管的完全够用

---

## 🔄 完整工作流程图

```
开发者操作
    |
    | git push origin main
    ↓
GitHub 服务器检测到 push 事件
    |
    | 读取 .github/workflows/test.yml
    ↓
GitHub Actions 调度器
    |
    | 分配 5 个虚拟机（并行）
    ↓
┌─────────────────────────────────────────┐
│  VM1 (Python 3.9)                       │
│  ├─ 检出代码                             │
│  ├─ 安装 Python 3.9                     │
│  ├─ 安装依赖                             │
│  ├─ 运行测试                             │
│  └─ ✅ 成功 / ❌ 失败                    │
└─────────────────────────────────────────┘
┌─────────────────────────────────────────┐
│  VM2 (Python 3.10)                      │
│  ... 同样的步骤 ...                      │
└─────────────────────────────────────────┘
┌─────────────────────────────────────────┐
│  VM3 (Python 3.11)                      │
│  ... 同样的步骤 ...                      │
│  └─ 额外：上传覆盖率到 Codecov           │
└─────────────────────────────────────────┘
┌─────────────────────────────────────────┐
│  VM4 (Python 3.12)                      │
│  ... 同样的步骤 ...                      │
└─────────────────────────────────────────┘
┌─────────────────────────────────────────┐
│  VM5 (Python 3.13)                      │
│  ... 同样的步骤 ...                      │
└─────────────────────────────────────────┘
    |
    | 所有任务完成
    ↓
GitHub 汇总结果
    |
    ├─ 发送邮件通知（如果失败）
    ├─ 在 PR 中显示状态
    └─ Actions 页面显示详细日志
```

---

## 📊 实际运行示例

### 成功的运行

```
✅ Python Unit Tests #42
   Triggered by: push to main
   Duration: 3m 24s
   
   Jobs:
   ✅ test (3.9)   - 2m 45s
   ✅ test (3.10)  - 2m 52s
   ✅ test (3.11)  - 3m 08s
   ✅ test (3.12)  - 2m 58s
   ✅ test (3.13)  - 3m 02s
```

### 失败的运行

```
❌ Python Unit Tests #43
   Triggered by: push to main
   Duration: 2m 15s
   
   Jobs:
   ✅ test (3.9)   - 2m 10s
   ❌ test (3.10)  - 1m 45s  ← 失败
   ✅ test (3.11)  - 2m 05s
   ✅ test (3.12)  - 1m 58s
   ✅ test (3.13)  - 2m 00s
   
   Error in test (3.10):
   > FAILED tests/test_example.py::test_add - AssertionError
```

---

## 🎯 总结

### 核心要点

1. **完全自动化**: 推送代码即可，无需手动触发
2. **零配置**: 只需添加 YAML 文件，GitHub 自动识别
3. **免费使用**: Public 仓库完全免费
4. **独立环境**: 每次运行都是全新的 VM
5. **并行执行**: 多个 Python 版本同时测试
6. **云端资源**: 使用 GitHub/Azure 的服务器

### 你需要做的

✅ **必需**:
1. 创建 `.github/workflows/test.yml` 文件
2. 推送到 GitHub

❌ **不需要**:
1. ❌ 不需要自己的服务器
2. ❌ 不需要额外的配置
3. ❌ 不需要支付费用（Public 仓库）
4. ❌ 不需要安装任何软件

就这么简单！🎉

---

## 📚 延伸阅读

- [GitHub Actions 官方文档](https://docs.github.com/en/actions)
- [工作流语法](https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions)
- [GitHub Actions 计费](https://docs.github.com/en/billing/managing-billing-for-github-actions/about-billing-for-github-actions)
