# 包名与模块名详解

## 快速对比

| 概念 | 名称 | 定义位置 | 使用场景 |
|------|------|---------|---------|
| **包名** | `yeannhua-example-package-demo` | `pyproject.toml` 的 `name` | pip 安装、PyPI 显示 |
| **模块名** | `example_package` | `src/example_package/` 目录名 | Python 代码中 import |

## 详细说明

### 1. 包名（Distribution Name）

**名称**：`yeannhua-example-package-demo`

**定义位置**：

```toml
# pyproject.toml
[project]
name = "yeannhua-example-package-demo"  # ← 这里定义包名
version = "0.2.0"
```

**使用场景**：

```bash
# 安装时使用包名
pip install yeannhua-example-package-demo

# pip list 显示包名
pip list | grep yeannhua-example-package-demo
# 输出: yeannhua-example-package-demo 0.2.0

# 卸载时使用包名
pip uninstall yeannhua-example-package-demo

# PyPI 上的 URL 使用包名
https://pypi.org/project/yeannhua-example-package-demo/
```

**命名规则**：
- ✅ 可以使用连字符 `-`
- ✅ 可以使用下划线 `_`
- ✅ 可以使用数字
- ❌ 不区分大小写（`My-Package` 和 `my-package` 视为相同）
- ⚠️ 必须在 PyPI 上全局唯一

---

### 2. 模块名（Package/Module Name）

**名称**：`example_package`

**定义位置**：

```
src/
└── example_package/        # ← 这个目录名就是模块名
    ├── __init__.py
    ├── main.py
    └── print.py
```

以及在 `pyproject.toml` 中指定：

```toml
[tool.hatch.build.targets.wheel]
packages = ["src/example_package"]  # ← 指向模块目录
```

**使用场景**：

```python
# Python 代码中导入时使用模块名
from example_package import hello, add, print_dict
import example_package

# 访问模块属性
print(example_package.__version__)
```

**命名规则**：
- ✅ 可以使用下划线 `_`
- ✅ 可以使用数字（但不能开头）
- ❌ 不能使用连字符 `-`（Python 标识符规则）
- ❌ 不能使用空格或其他特殊字符
- 📝 必须是合法的 Python 标识符

---

## 为什么要区分？

### 历史原因
- Python 包管理（pip）和 Python 语言（import）是两个不同的系统
- pip 的命名规则比 Python 标识符规则更宽松
- 早期 Python 生态就是这样设计的

### 实际好处

1. **灵活性**：包名可以更有描述性
   ```
   包名: my-awesome-ml-library
   模块名: mllib
   ```

2. **避免冲突**：可以用前缀区分不同作者的同名包
   ```
   包名: user1-utils, user2-utils
   模块名: utils (在各自的命名空间中)
   ```

3. **品牌友好**：包名可以包含公司/个人标识
   ```
   包名: yeannhua-example-package-demo
   模块名: example_package (简洁易用)
   ```

---

## 真实案例

### 案例 1: 名称一致
```python
# 包名和模块名相同
包名: requests
模块名: requests

# 使用
pip install requests
import requests
```

### 案例 2: 名称不同（很常见）
```python
# Pillow (PIL 的分支)
包名: Pillow
模块名: PIL

# 使用
pip install Pillow
from PIL import Image

# scikit-learn
包名: scikit-learn  # 包含连字符
模块名: sklearn      # 不能包含连字符

# 使用
pip install scikit-learn
import sklearn

# beautifulsoup4
包名: beautifulsoup4
模块名: bs4

# 使用
pip install beautifulsoup4
from bs4 import BeautifulSoup
```

### 案例 3: 本项目
```python
# 本项目的命名
包名: yeannhua-example-package-demo  # 带前缀，全局唯一
模块名: example_package              # 简洁，易于导入

# 使用
pip install yeannhua-example-package-demo
from example_package import hello
```

---

## 配置文件完整示例

### pyproject.toml

```toml
[project]
name = "yeannhua-example-package-demo"  # ← 包名（pip install 用）
version = "0.2.0"
description = "一个简单的 Python 包示例"

[build-system]
requires = ["hatchling"]
build-backend = "hatchling.build"

[tool.hatch.build.targets.wheel]
packages = ["src/example_package"]      # ← 模块位置（import 用）
```

### 目录结构

```
example_package/
├── pyproject.toml              # 定义包名
├── src/
│   └── example_package/        # 定义模块名（目录名）
│       ├── __init__.py
│       ├── main.py
│       └── print.py
└── README.md
```

---

## 如何修改

### 修改包名（安装时的名字）

编辑 `pyproject.toml`：

```toml
[project]
name = "your-new-package-name"  # 修改这里
```

然后重新构建：
```bash
python3 -m build
```

### 修改模块名（导入时的名字）

1. 重命名目录：
   ```bash
   mv src/example_package src/your_module_name
   ```

2. 更新 `pyproject.toml`：
   ```toml
   [tool.hatch.build.targets.wheel]
   packages = ["src/your_module_name"]
   ```

3. 重新构建：
   ```bash
   python3 -m build
   ```

---

## 最佳实践

### ✅ 推荐做法

1. **简单包**：包名和模块名一致
   ```
   pip install mypackage
   import mypackage
   ```

2. **复杂包**：使用连字符的包名，下划线的模块名
   ```
   pip install my-awesome-package
   import my_awesome_package
   ```

3. **命名空间包**：使用前缀避免冲突
   ```
   pip install company-utils
   import company_utils
   ```

### ❌ 避免的做法

1. **混淆的命名**：包名和模块名差异太大
   ```
   pip install abc-xyz-123
   import totally_different_name  # 用户很难记住
   ```

2. **违反规则**：模块名使用连字符
   ```python
   # 错误！Python 不允许
   import my-package  # SyntaxError
   ```

---

## 验证方法

### 查看已安装的包名
```bash
pip list | grep example
# yeannhua-example-package-demo 0.2.0

pip show yeannhua-example-package-demo
# Name: yeannhua-example-package-demo
# Location: /path/to/site-packages
```

### 查看可导入的模块名
```python
import sys
import pkgutil

# 查看所有可导入的模块
[name for _, name, _ in pkgutil.iter_modules() if 'example' in name]
# ['example_package']

# 或直接尝试导入
import example_package
print(example_package.__name__)  # example_package
```

---

## 总结

| 方面 | 包名 | 模块名 |
|------|------|--------|
| **定义** | `pyproject.toml` 的 `name` | `src/` 下的目录名 |
| **用途** | pip 安装管理 | Python 代码导入 |
| **命名规则** | 宽松（可用 `-`） | 严格（Python 标识符） |
| **全局性** | PyPI 全局唯一 | 可以重复（不同包） |
| **示例** | `yeannhua-example-package-demo` | `example_package` |
| **使用** | `pip install 包名` | `import 模块名` |

**记住**：
- 📦 **包名** = pip 看到的名字
- 📚 **模块名** = import 使用的名字
- 🔗 两者通过 `pyproject.toml` 关联

```
┌─────────────────────────────────────────────────────────────────────┐
│                    Python 包的两个名字                               │
└─────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────┐    ┌─────────────────────────────────┐
│         包名 (Package Name)      │    │      模块名 (Module Name)        │
│  用于 pip/PyPI 的分发名称         │    │   用于 Python 代码中的导入        │
├─────────────────────────────────┤    ├─────────────────────────────────┤
│ yeannhua-example-package-demo   │    │ example_package                 │
│                                 │    │                                 │
│ 定义位置:                        │    │ 定义位置:                        │
│ pyproject.toml                  │    │ src/ 下的目录名                  │
│ ┌─────────────────────────────┐ │    │ ┌─────────────────────────────┐ │
│ │ [project]                   │ │    │ │ src/                        │ │
│ │ name = "yeannhua-example-   │ │    │ │ └── example_package/        │ │
│ │         package-demo"       │ │    │ │     ├── __init__.py         │ │
│ │                             │ │    │ │     ├── main.py             │ │
│ └─────────────────────────────┘ │    │ │     └── print.py            │ │
│                                 │    │ └─────────────────────────────┘ │
│ 使用场景:                        │    │ 使用场景:                        │
│ • pip install                   │    │ • import / from ... import      │
│ • pip list                      │    │ • Python 代码中                  │
│ • pip uninstall                 │    │                                 │
│ • PyPI 网站 URL                 │    │ 命名规则:                        │
│                                 │    │ • 必须是 Python 标识符           │
│ 命名规则:                        │    │ • 可以用 _ (下划线)              │
│ • 可以用 - (连字符)             │    │ • 不能用 - (连字符)              │
│ • 可以用 _ (下划线)             │    │ • 不能以数字开头                 │
│ • PyPI 全局唯一                 │    │                                 │
└─────────────────────────────────┘    └─────────────────────────────────┘
         │                                          │
         │                                          │
         ▼                                          ▼
┌─────────────────────────────────┐    ┌─────────────────────────────────┐
│      命令行使用示例               │    │      代码中使用示例               │
├─────────────────────────────────┤    ├─────────────────────────────────┤
│ $ pip install \                 │    │ # Python 代码                    │
│   yeannhua-example-package-demo │    │ from example_package import \   │
│                                 │    │      hello, add, print_dict     │
│ $ pip list | grep example       │    │                                 │
│ yeannhua-example-package-demo   │    │ import example_package          │
│ 0.2.0                           │    │ print(example_package.          │
│                                 │    │       __version__)              │
│ $ pip show yeannhua-example-... │    │                                 │
│ Name: yeannhua-example-package- │    │ # 注意：不能这样导入！            │
│       demo                      │    │ # import yeannhua-example-      │
│ Version: 0.2.0                  │    │ #          package-demo         │
│                                 │    │ # SyntaxError! (因为有 -)        │
└─────────────────────────────────┘    └─────────────────────────────────┘


┌─────────────────────────────────────────────────────────────────────┐
│                         关联方式                                     │
└─────────────────────────────────────────────────────────────────────┘

pyproject.toml 中的关联:

[project]
name = "yeannhua-example-package-demo"    ← 包名 (pip 用)
version = "0.2.0"

[tool.hatch.build.targets.wheel]
packages = ["src/example_package"]        ← 指向模块位置 (import 用)


┌─────────────────────────────────────────────────────────────────────┐
│                      真实世界的例子                                   │
└─────────────────────────────────────────────────────────────────────┘

┌──────────────────────┬──────────────────────┬─────────────────────┐
│      包名 (pip)       │   模块名 (import)     │        原因          │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ requests             │ import requests      │ 一致（简单明了）      │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ Pillow               │ from PIL import      │ Pillow 是 PIL 的分支  │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ scikit-learn         │ import sklearn       │ - 不能在模块名中      │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ beautifulsoup4       │ from bs4 import      │ 版本号+简短别名       │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ opencv-python        │ import cv2           │ 历史遗留+简短易用     │
├──────────────────────┼──────────────────────┼─────────────────────┤
│ yeannhua-example-    │ import example_      │ 前缀标识+简洁导入     │
│ package-demo         │ package              │                     │
└──────────────────────┴──────────────────────┴─────────────────────┘


┌─────────────────────────────────────────────────────────────────────┐
│                        快速记忆                                      │
└─────────────────────────────────────────────────────────────────────┘

📦 包名 = pip 看到的名字
   • 可以有 - (yeannhua-example-package-demo)
   • pip install 用这个名字

📚 模块名 = import 使用的名字
   • 不能有 - (example_package)
   • Python 代码中用这个名字

💡 记住：Python 能 import 的，必须是合法的 Python 标识符！
```
