# Pull Request 中的 GitHub Actions 运行机制

## 🔄 PR 触发时机详解

### 触发配置

```yaml
on:
  pull_request:
    branches: [main]  # 向 main 分支提交 PR 时触发
```

### 完整时间线

```
时刻 0:00 - 开发者创建 PR
            ↓
时刻 0:01 - GitHub 检测到 PR 事件
            ↓
时刻 0:02 - 立即启动 GitHub Actions
            ↓ (显示状态：⏳ 排队中)
            ↓
时刻 0:05 - 分配虚拟机，开始运行
            ↓ (显示状态：🔄 运行中)
            ↓
时刻 0:10 - 检出代码、安装依赖
            ↓
时刻 1:00 - 运行测试
            ↓
时刻 3:00 - 完成测试
            ↓ (显示状态：✅ 成功 或 ❌ 失败)
            ↓
显示结果  - PR 页面显示测试结果
```

---

## 🎯 PR 中的运行场景

### 场景 1: 创建新 PR

**操作流程**：
```bash
# 开发者在自己的分支上工作
git checkout -b feature/new-feature
git add .
git commit -m "Add new feature"
git push origin feature/new-feature

# 在 GitHub 上创建 PR: feature/new-feature → main
```

**GitHub Actions 行为**：
1. ✅ **立即运行**（PR 创建后几秒内）
2. 🔄 PR 页面显示"Checks"部分
3. ⏳ 显示测试状态（排队、运行中、完成）
4. ✅/❌ 显示最终结果

**PR 页面显示**：
```
┌─────────────────────────────────────────────┐
│  Pull Request #42                           │
│  Add new feature                            │
├─────────────────────────────────────────────┤
│  Some checks haven't completed yet          │
│  🔄 Python Unit Tests  In progress...       │
│     ├─ test (3.9)   ✅ Passed               │
│     ├─ test (3.10)  ✅ Passed               │
│     ├─ test (3.11)  🔄 Running...           │
│     ├─ test (3.12)  ⏳ Queued               │
│     └─ test (3.13)  ⏳ Queued               │
└─────────────────────────────────────────────┘
```

### 场景 2: PR 中新增提交

**操作流程**：
```bash
# PR 已创建，但发现需要修改
git add .
git commit -m "Fix issues"
git push origin feature/new-feature
```

**GitHub Actions 行为**：
1. ✅ **再次立即运行**（每次新 commit 都会触发）
2. 🛑 如果之前的测试还在运行，会被取消
3. 🆕 使用最新的代码重新测试
4. 📊 历史记录可以查看所有运行

**PR 页面更新**：
```
┌─────────────────────────────────────────────┐
│  Pull Request #42                           │
│  2 commits                                  │
├─────────────────────────────────────────────┤
│  All checks have passed                     │
│  ✅ Python Unit Tests  Passed (3m 24s)     │
│     ├─ test (3.9)   ✅ Passed               │
│     ├─ test (3.10)  ✅ Passed               │
│     ├─ test (3.11)  ✅ Passed               │
│     ├─ test (3.12)  ✅ Passed               │
│     └─ test (3.13)  ✅ Passed               │
│                                             │
│  [✅ Merge pull request]  ← 可以合并        │
└─────────────────────────────────────────────┘
```

### 场景 3: 测试失败

**PR 页面显示**：
```
┌─────────────────────────────────────────────┐
│  Pull Request #42                           │
├─────────────────────────────────────────────┤
│  ❌ Some checks were not successful         │
│  ❌ Python Unit Tests  Failed               │
│     ├─ test (3.9)   ✅ Passed               │
│     ├─ test (3.10)  ❌ Failed               │
│     ├─ test (3.11)  ✅ Passed               │
│     ├─ test (3.12)  ✅ Passed               │
│     └─ test (3.13)  ✅ Passed               │
│                                             │
│  [🚫 Merge pull request]  ← 不建议合并      │
│                                             │
│  View details: test (3.10) failed           │
│  > FAILED tests/test_example.py::test_add  │
│  > AssertionError: assert 4 == 5            │
└─────────────────────────────────────────────┘
```

---

## ⚙️ 分支保护规则（推荐设置）

### 设置步骤

在 GitHub 仓库中：
1. Settings → Branches
2. Add branch protection rule
3. Branch name pattern: `main`
4. 勾选以下选项：

```
✅ Require status checks to pass before merging
   └─ ✅ Require branches to be up to date before merging
   └─ Status checks that are required:
      ├─ ✅ test (3.9)
      ├─ ✅ test (3.10)
      ├─ ✅ test (3.11)
      ├─ ✅ test (3.12)
      └─ ✅ test (3.13)

✅ Require review from Code Owners
✅ Do not allow bypassing the above settings
```

### 设置后的效果

**1. 测试未通过时**：
```
❌ Merging is blocked
   The following checks failed:
   - test (3.10): Failed

   [Details] [Re-run failed jobs]
```

**2. 测试通过后**：
```
✅ All checks have passed
   5 successful checks

   [✅ Merge pull request]
```

**3. 测试运行中**：
```
⏳ Merging is blocked
   Checks are still running...
   
   [View details]
```

---

## 🔄 多种 PR 事件触发

test.yml 可以响应多种 PR 事件：

### 默认配置（最简单）

```yaml
on:
  pull_request:
    branches: [main]
```

**触发时机**：
- ✅ PR 打开 (opened)
- ✅ PR 更新 (synchronize) - 新 commit 推送
- ✅ PR 重新打开 (reopened)

### 高级配置（精细控制）

```yaml
on:
  pull_request:
    types: [opened, synchronize, reopened, ready_for_review]
    branches: [main, develop]
```

**触发时机**：
- `opened` - PR 创建时
- `synchronize` - 新 commit 推送时
- `reopened` - PR 重新打开时
- `ready_for_review` - 草稿 PR 标记为"准备审查"时

---

## 📊 实际案例演示

### 案例 1: 创建 PR 并推送修复

```bash
# 1. 创建功能分支
$ git checkout -b feature/add-logging
$ git commit -m "Add logging"
$ git push origin feature/add-logging

# 在 GitHub 上创建 PR
→ GitHub Actions 立即启动 (0:02 秒内)
→ PR 页面显示: ⏳ Checks in progress...

# 2. 收到邮件: 测试失败
→ 查看详情: test (3.10) failed

# 3. 本地修复
$ git commit -m "Fix Python 3.10 compatibility"
$ git push origin feature/add-logging

# Push 后立即:
→ GitHub Actions 再次启动 (0:02 秒内)
→ 旧的运行被取消
→ 新的测试开始运行

# 4. 3 分钟后
→ 收到邮件: All checks passed ✅
→ PR 页面: 绿色勾号，可以合并
```

### 案例 2: Draft PR

```yaml
on:
  pull_request:
    types: [opened, synchronize, reopened, ready_for_review]
```

**流程**：
```bash
# 1. 创建草稿 PR (Draft PR)
→ ❌ 不会立即运行测试

# 2. 继续开发，推送多次提交
→ ❌ 仍然不运行

# 3. 标记为 "Ready for review"
→ ✅ 立即触发测试

# 好处: 节省资源，避免在开发中期频繁测试
```

---

## 🎯 最佳实践

### 1. 快速反馈循环

```
开发 → Push → PR创建 (0:00)
                ↓ (立即触发)
             测试运行 (0:02)
                ↓ (3分钟)
             显示结果 (3:00)
                ↓
        修复或继续开发
```

### 2. 并行与效率

```
5 个 Python 版本并行测试
├─ 串行: 5 × 3分钟 = 15分钟
└─ 并行: 3分钟 ✅
   
结果: 提速 5 倍！
```

### 3. 合并前检查

```
PR 创建 → 测试运行 → 代码审查 → 测试通过 → 合并
         ↑                              ↑
      立即执行                      分支保护确保
```

---

## 🔔 通知机制

### 自动通知

GitHub 会在以下情况通知你：

| 事件 | 通知方式 |
|------|----------|
| 测试开始 | 无通知（太频繁） |
| 测试失败 | 📧 邮件 + 🔔 网站通知 |
| 测试通过 | 🔔 网站通知（可关闭邮件） |
| PR 可合并 | 🔔 网站通知 |

### 设置通知偏好

Settings → Notifications → Actions:
- ✅ Send notifications for failed workflows only
- ❌ Send notifications for all workflow runs

---

## 💡 总结

### 核心要点

1. **立即运行**: PR 创建后 **几秒内**就会触发
2. **实时反馈**: PR 页面实时显示测试进度
3. **每次更新都会重新测试**: 推送新 commit 会自动重新运行
4. **可以设置必须通过**: 分支保护规则强制测试通过才能合并
5. **完全自动化**: 无需手动触发，一切都是自动的

### 时间对比

| 操作 | 触发延迟 |
|------|----------|
| 创建 PR | **< 5 秒** |
| 推送新 commit | **< 5 秒** |
| 开始运行测试 | **< 30 秒** |
| 完成所有测试 | **~ 3 分钟** |

所以，是的，**PR 一提交就立即运行，不用等待！** ⚡

---

## 📖 查看运行状态

### 在 PR 页面查看

1. 打开你的 PR
2. 滚动到底部的 "Checks" 区域
3. 实时查看所有测试状态
4. 点击 "Details" 查看详细日志

### 在 Actions 页面查看

1. 进入仓库的 "Actions" 标签
2. 查看所有工作流运行历史
3. 可以过滤：PR、分支、状态等
4. 点击任意运行查看详细日志

非常直观和实时！🎉
