# `make publish-test` 执行过程详解

## 📋 命令概述

`make publish-test` 是一个复合命令，它会依次执行：
1. `make clean` - 清理旧的构建文件
2. `make build` - 构建新的分发包
3. 交互式确认
4. 上传到 TestPyPI

---

## 🔍 详细步骤解析

### 步骤 1: 清理构建文件 (`make clean`)

```bash
==> 清理构建文件...
删除 dist/ build/ *.egg-info
rm -rf dist/ build/ *.egg-info src/*.egg-info
```

**作用**: 删除之前构建产生的文件，确保全新构建

**清理的内容**:
- `dist/` - 之前构建的分发包（.whl 和 .tar.gz）
- `build/` - 构建过程中的临时文件
- `*.egg-info` - Python 包的元数据目录
- `src/*.egg-info` - 源代码目录下的元数据

**为什么需要清理**:
- ✅ 避免使用旧的构建文件
- ✅ 确保版本号正确
- ✅ 避免文件冲突

---

```bash
删除 __pycache__ 和 .pyc 文件
find . -type d -name "__pycache__" -exec rm -rf {} + 2>/dev/null || true
find . -type f -name "*.pyc" -delete
find . -type f -name "*.pyo" -delete
```

**作用**: 清理 Python 编译缓存

**清理的内容**:
- `__pycache__/` - Python 字节码缓存目录
- `*.pyc` - 编译后的 Python 字节码文件
- `*.pyo` - 优化后的字节码文件

**为什么需要清理**:
- ✅ 避免包含缓存文件到分发包
- ✅ 确保使用最新的源代码
- ✅ 减少分发包大小

---

```bash
删除 pytest 缓存
rm -rf .pytest_cache
删除覆盖率文件
rm -rf .coverage htmlcov/
删除 mypy 缓存
rm -rf .mypy_cache
```

**作用**: 清理测试和代码分析产生的文件

**清理的内容**:
- `.pytest_cache/` - pytest 测试框架的缓存
- `.coverage` - 代码覆盖率数据文件
- `htmlcov/` - HTML 格式的覆盖率报告
- `.mypy_cache/` - 类型检查工具 mypy 的缓存

**为什么需要清理**:
- ✅ 这些文件不应该包含在分发包中
- ✅ 保持仓库干净
- ✅ 减少分发包大小

---

### 步骤 2: 构建分发包 (`make build`)

```bash
==> 构建分发包...
python3 -m build
```

**作用**: 使用 Python 的 `build` 模块构建分发包

#### 2.1 创建隔离环境

```
* Creating isolated environment: venv+pip...
```

**说明**:
- `build` 工具会创建一个**临时的虚拟环境**
- 这个环境与你的系统环境隔离
- 确保构建过程干净、可重复

**为什么需要隔离环境**:
- ✅ 避免使用系统已安装的包
- ✅ 确保构建依赖版本正确
- ✅ 避免污染系统环境

---

#### 2.2 安装构建依赖

```
* Installing packages in isolated environment:
  - hatch-fancy-pypi-readme
  - hatchling==1.26.3
```

**说明**:
- `hatchling` - 构建后端（在 `pyproject.toml` 中指定）
- `hatch-fancy-pypi-readme` - 用于处理 README 的插件

**这些依赖来自**:
```toml
[build-system]
requires = ["hatchling", "hatch-fancy-pypi-readme"]
build-backend = "hatchling.build"
```

---

## 📚 构建后端（Build Backend）详解

### 什么是构建后端？

**构建后端（Build Backend）** 是一个**工具或库**，负责将你的 Python 源代码**转换成可分发的包**（`.whl` 和 `.tar.gz` 文件）。

**简单理解**:
```
源代码（你的项目）
    ↓
构建后端（"打包工人"）
    ↓
分发包（可以上传到 PyPI 的文件）
```

**构建后端的作用**:
1. ✅ 读取项目配置（`pyproject.toml`）
2. ✅ 发现所有源代码文件
3. ✅ 打包成 `.whl` 和 `.tar.gz`
4. ✅ 包含元数据（版本、作者、依赖等）

---

### 构建后端的工作流程

```
开发者执行: python -m build
    ↓
build 工具查找构建后端
    ↓
读取 pyproject.toml:
  [build-system]
  build-backend = "hatchling.build"
    ↓
调用构建后端（hatchling）
    ↓
构建后端执行:
  1. 读取 [project] 配置
  2. 读取 [tool.hatch.build] 配置
  3. 发现包结构
  4. 生成分发包
    ↓
生成文件:
  - dist/xxx.whl
  - dist/xxx.tar.gz
```

---

### 常见的构建后端

| 构建后端 | 配置复杂度 | 功能 | 速度 | 推荐度 |
|---------|-----------|------|------|--------|
| **hatchling** | ⭐ 简单 | ⭐⭐⭐ 中等 | ⭐⭐⭐ 快 | ⭐⭐⭐⭐⭐ 推荐 |
| **setuptools** | ⭐⭐⭐ 复杂 | ⭐⭐⭐⭐⭐ 强大 | ⭐⭐ 中等 | ⭐⭐⭐ 传统 |
| **flit** | ⭐ 极简 | ⭐⭐ 有限 | ⭐⭐⭐ 快 | ⭐⭐⭐ 简单项目 |
| **poetry** | ⭐⭐ 中等 | ⭐⭐⭐⭐⭐ 强大 | ⭐⭐ 中等 | ⭐⭐⭐⭐ 大型项目 |
| **pdm** | ⭐⭐ 中等 | ⭐⭐⭐⭐ 强大 | ⭐⭐⭐ 快 | ⭐⭐⭐⭐ 现代项目 |

**推荐**: 对于大多数项目，使用 `hatchling`（现代、简单、快速）

---

### 如何配置构建后端

在 `pyproject.toml` 中配置：

```toml
[build-system]
requires = ["hatchling"]           # 构建依赖
build-backend = "hatchling.build"  # 构建后端名称
```

**两个关键字段**:

1. **`requires`** - 构建时需要的依赖
   - 这些包会在构建时安装
   - 不会包含在最终的分发包中
   - 只在构建时使用

2. **`build-backend`** - 构建后端的入口点
   - 告诉 `build` 工具使用哪个后端
   - 格式：`包名.模块名`

---

### 构建后端的工作原理

#### PEP 517/518 标准

Python 打包遵循 **PEP 517** 和 **PEP 518** 标准：

- **PEP 517**: 定义了构建后端的接口
- **PEP 518**: 定义了如何指定构建依赖

**好处**:
- ✅ 标准化接口
- ✅ 可以切换构建后端
- ✅ 工具之间可以互操作

#### 实际调用过程

```python
# build 工具的内部逻辑（简化版）
def build():
    # 1. 读取 pyproject.toml
    config = read_pyproject_toml()
    
    # 2. 创建隔离环境
    env = create_isolated_environment()
    
    # 3. 安装构建依赖
    install_build_dependencies(
        config['build-system']['requires'],
        env
    )
    
    # 4. 导入构建后端
    backend = import_backend(
        config['build-system']['build-backend']
    )
    
    # 5. 调用构建后端
    wheel_file = backend.build_wheel('dist/')
    sdist_file = backend.build_sdist('dist/')
    
    print(f"Built: {wheel_file}, {sdist_file}")
```

---

### 关键概念

#### 构建工具 vs 构建后端

| 概念 | 作用 | 示例 |
|------|------|------|
| **构建工具** | 调用构建后端 | `build`, `pip` |
| **构建后端** | 实际执行构建 | `hatchling`, `setuptools` |

**关系**:
```
构建工具 (build)
    ↓ 调用
构建后端 (hatchling)
    ↓ 执行
生成分发包
```

#### 为什么需要 `requires`？

```toml
[build-system]
requires = ["hatchling"]  # ← 为什么需要这个？
build-backend = "hatchling.build"
```

**原因**:
- 构建后端本身是一个 Python 包
- 需要先安装才能使用
- `build` 工具会在隔离环境中安装这些依赖

**过程**:
```
python -m build
  ↓
创建隔离环境
  ↓
安装 requires 中的包（hatchling）
  ↓
导入 build-backend（hatchling.build）
  ↓
调用构建方法
```

---

### 详细说明：hatchling 和 hatch-fancy-pypi-readme

#### 1. hatchling - 构建后端

**什么是 hatchling**:
- 一个**现代的 Python 包构建工具**
- 由 Hatch 项目提供
- 是 `setuptools` 的现代替代品

**主要功能**:

1. **读取 `pyproject.toml` 配置**
   ```toml
   [project]
   name = "eos"
   version = "0.0.2"
   description = "The official Python library for the eos API"
   ```

2. **自动发现包结构**
   ```toml
   [tool.hatch.build.targets.wheel]
   packages = ["src/debug_helpers"]
   # 或自动发现所有包
   ```

3. **生成分发包**
   - 创建 `.whl` (wheel) 文件
   - 创建 `.tar.gz` (sdist) 文件
   - 包含所有必要的元数据

**配置示例**:

**简单配置**（debug-helpers 使用）:
```toml
[build-system]
requires = ["hatchling"]
build-backend = "hatchling.build"

[tool.hatch.build.targets.wheel]
packages = ["src/debug_helpers"]
```

**高级配置**（eos-python 使用）:
```toml
[build-system]
requires = ["hatchling", "hatch-fancy-pypi-readme"]
build-backend = "hatchling.build"

[tool.hatch.build.targets.wheel]
packages = ["src/eos"]

[tool.hatch.metadata]
allow-direct-references = true
```

**hatchling 的优势**:
- ✅ 使用 `pyproject.toml`（PEP 517/518 标准）
- ✅ 配置简单，无需 `setup.py`
- ✅ 自动处理包发现
- ✅ 支持现代 Python 打包标准
- ✅ 构建速度快

**对比传统方式**:

```python
# 旧方式：setup.py（已弃用）
from setuptools import setup, find_packages

setup(
    name="eos",
    version="0.0.2",
    packages=find_packages(),
    # ... 很多配置
)
```

```toml
# 新方式：pyproject.toml（推荐）
[project]
name = "eos"
version = "0.0.2"
# 配置更简洁
```

---

#### 2. hatch-fancy-pypi-readme - README 处理插件

**什么是 hatch-fancy-pypi-readme**:
- 一个 **hatchling 插件**
- 专门用于处理 PyPI 上的 README 显示
- 支持多种 README 格式和功能

**主要功能**:

1. **自动检测 README 格式**
   - Markdown (`.md`)
   - reStructuredText (`.rst`)
   - Plain text (`.txt`)

2. **处理 README 内容类型**
   ```toml
   [project]
   dynamic = ["readme"]  # 动态读取 README
   ```

3. **支持高级功能**
   - 从文件读取
   - 从 URL 读取
   - 内容类型自动检测
   - 支持多个 README 文件

**配置示例**:

**方式 1: 简单配置**（自动检测）
```toml
[project]
readme = "README.md"  # 直接指定文件
```

**方式 2: 动态配置**（使用插件）
```toml
[project]
dynamic = ["readme"]  # 动态读取

[tool.hatch.metadata.hooks.fancy-pypi-readme]
content-type = "text/markdown"
files = ["README.md"]
```

**方式 3: 高级配置**（eos-python 使用）
```toml
[project]
dynamic = ["readme"]

[tool.hatch.metadata.hooks.fancy-pypi-readme]
content-type = "text/markdown"
files = [
    { path = "README.md", content-type = "text/markdown" },
    { path = "CHANGELOG.md", content-type = "text/markdown" },
]
```

**为什么需要这个插件**:

**不使用插件**:
```toml
[project]
readme = "README.md"  # 简单，但功能有限
```
- ✅ 简单直接
- ❌ 只能处理单个文件
- ❌ 不能合并多个文件
- ❌ 不能从 URL 读取

**使用插件**:
```toml
[tool.hatch.metadata.hooks.fancy-pypi-readme]
files = ["README.md", "CHANGELOG.md"]
```
- ✅ 可以合并多个文件
- ✅ 支持从 URL 读取
- ✅ 更灵活的配置
- ✅ 支持内容类型检测

**实际效果**:

**在 PyPI 上显示**:
```
https://pypi.org/project/eos/

Project description:
┌─────────────────────────────────────┐
│ # eos                               │
│                                     │
│ The official Python library...      │
│                                     │
│ ## Installation                    │
│ ...                                 │
│                                     │
│ ## Changelog                        │
│ ...                                 │
└─────────────────────────────────────┘
```

**插件处理过程**:
```
README.md + CHANGELOG.md
    ↓
hatch-fancy-pypi-readme 处理
    ↓
合并内容
    ↓
转换为 PyPI 格式
    ↓
嵌入到分发包元数据
    ↓
PyPI 显示
```

---

### 完整配置示例对比

#### 示例 1: 简单项目（debug-helpers）

```toml
[project]
name = "debug-helpers"
version = "0.4.0"
readme = "README.md"  # 直接指定，不需要插件

[build-system]
requires = ["hatchling"]  # 只需要 hatchling
build-backend = "hatchling.build"

[tool.hatch.build.targets.wheel]
packages = ["src/debug_helpers"]
```

**特点**:
- ✅ 配置简单
- ✅ 适合小型项目
- ✅ 单个 README 文件

---

#### 示例 2: 复杂项目（eos-python）

```toml
[project]
name = "eos"
version = "0.0.2"
dynamic = ["readme"]  # 动态读取，需要插件

[build-system]
requires = ["hatchling", "hatch-fancy-pypi-readme"]  # 需要插件
build-backend = "hatchling.build"

[tool.hatch.build.targets.wheel]
packages = ["src/eos"]

[tool.hatch.metadata.hooks.fancy-pypi-readme]
content-type = "text/markdown"
files = ["README.md"]  # 可以配置多个文件
```

**特点**:
- ✅ 功能更强大
- ✅ 支持多个 README 文件
- ✅ 更灵活的配置
- ✅ 适合大型项目

---

### 何时使用哪个？

| 场景 | 使用 | 原因 |
|------|------|------|
| **单个 README.md** | `readme = "README.md"` | 简单直接 |
| **多个 README 文件** | `hatch-fancy-pypi-readme` | 需要合并 |
| **从 URL 读取** | `hatch-fancy-pypi-readme` | 插件支持 |
| **复杂内容类型** | `hatch-fancy-pypi-readme` | 更灵活 |
| **简单项目** | 只用 `hatchling` | 够用 |
| **复杂项目** | `hatchling + 插件` | 功能更强 |

---

### 构建过程详解

**使用 hatchling（无插件）**:
```
python -m build
  ↓
读取 pyproject.toml
  ↓
hatchling 处理:
  ├─ 读取 [project] 配置
  ├─ 读取 [tool.hatch.build] 配置
  ├─ 发现包结构
  ├─ 读取 README.md（直接）
  └─ 生成分发包
```

**使用 hatchling + 插件**:
```
python -m build
  ↓
读取 pyproject.toml
  ↓
hatchling 处理:
  ├─ 读取 [project] 配置
  ├─ 读取 [tool.hatch.build] 配置
  ├─ 发现包结构
  └─ 调用插件
      ↓
hatch-fancy-pypi-readme 处理:
  ├─ 读取 README.md
  ├─ 读取 CHANGELOG.md（如果配置）
  ├─ 合并内容
  ├─ 检测内容类型
  └─ 返回处理后的 README
      ↓
生成分发包（包含处理后的 README）
```

---

### 实际文件示例

**项目结构**:
```
eos-python/
├── README.md          # 主 README
├── CHANGELOG.md      # 更新日志
├── pyproject.toml    # 配置文件
└── src/
    └── eos/
        └── __init__.py
```

**pyproject.toml 配置**:
```toml
[project]
dynamic = ["readme"]

[tool.hatch.metadata.hooks.fancy-pypi-readme]
files = [
    { path = "README.md", content-type = "text/markdown" },
    { path = "CHANGELOG.md", content-type = "text/markdown" },
]
```

**构建结果**:
- PyPI 上会显示：README.md + CHANGELOG.md 的合并内容
- 分发包元数据中包含完整的 README 内容

---

### 总结

| 工具 | 作用 | 必需性 |
|------|------|--------|
| **hatchling** | 构建后端，生成分发包 | ✅ **必需** |
| **hatch-fancy-pypi-readme** | 处理 README，支持高级功能 | ⚠️ **可选** |

**选择建议**:
- 简单项目：只用 `hatchling`
- 复杂项目：`hatchling + hatch-fancy-pypi-readme`

---

#### 2.3 构建源代码分发包 (sdist)

```
* Getting build dependencies for sdist...
* Building sdist...
```

**说明**:
- `sdist` = Source Distribution（源代码分发包）
- 生成 `.tar.gz` 文件
- 包含完整的源代码

**生成的文件**: `eos-0.0.2.tar.gz`

**包含内容**:
- 所有源代码文件
- `pyproject.toml`
- `README.md`、`LICENSE` 等
- 构建脚本（如果有）

---

#### 2.4 构建 Wheel 分发包

```
* Building wheel from sdist
* Creating isolated environment: venv+pip...
* Installing packages in isolated environment:
  - hatch-fancy-pypi-readme
  - hatchling==1.26.3
* Getting build dependencies for wheel...
* Building wheel...
```

**说明**:
- `wheel` = 预编译的分发包格式
- 生成 `.whl` 文件
- 安装速度更快

**生成的文件**: `eos-0.0.2-py3-none-any.whl`

**包含内容**:
- 预编译的字节码
- 打包好的模块
- 元数据

**为什么需要两个格式**:
- `.tar.gz` - 兼容性好，所有平台通用
- `.whl` - 安装快，现代 pip 优先使用

---

#### 2.5 构建完成

```
Successfully built eos-0.0.2.tar.gz and eos-0.0.2-py3-none-any.whl
```

**文件命名规则**:
```
{包名}-{版本号}-{Python版本标签}-{平台标签}.{扩展名}

eos-0.0.2-py3-none-any.whl
│   │     │  │   │
│   │     │  │   └─ 平台: none-any (纯 Python，跨平台)
│   │     │  └─ ABI: none
│   │     └─ Python 版本: py3 (Python 3)
│   └─ 版本号: 0.0.2
└─ 包名: eos
```

---

### 步骤 3: 检查分发包

```bash
==> 检查分发包...
python3 -m twine check dist/*
Checking dist/eos-0.0.2-py3-none-any.whl: PASSED
Checking dist/eos-0.0.2.tar.gz: PASSED
```

**作用**: 使用 `twine check` 验证分发包

**检查内容**:
- ✅ 文件格式是否正确
- ✅ 元数据是否完整
- ✅ 是否符合 PyPI 规范
- ✅ README 格式是否正确

**为什么需要检查**:
- ✅ 提前发现问题
- ✅ 避免上传失败
- ✅ 确保分发包质量

---

### 步骤 4: 显示构建结果

```bash
✅ 构建完成！
生成的文件：
total 616
-rw-r--r--@ 1 admin  staff    85K 25 Jan 08:55 eos-0.0.2-py3-none-any.whl
-rw-r--r--@ 1 admin  staff   219K 25 Jan 08:55 eos-0.0.2.tar.gz
```

**说明**:
- `.whl` 文件: 85KB（更小，因为已编译）
- `.tar.gz` 文件: 219KB（更大，包含源代码）

---

### 步骤 5: 准备发布到 TestPyPI

```bash
==========================================
  准备发布到 TestPyPI
==========================================

包名: eos
版本: 0.0.2

确认上传到 TestPyPI? (y/n) y
```

**作用**: 
- 显示包信息
- 要求用户确认（防止误操作）

**交互式确认的好处**:
- ✅ 防止意外上传
- ✅ 给用户最后一次检查机会
- ✅ 可以取消操作

---

### 步骤 6: 上传到 TestPyPI

```bash
==> 上传到 TestPyPI...
Uploading distributions to https://test.pypi.org/legacy/
```

**说明**: 开始上传到 TestPyPI 服务器

---

#### 6.1 上传 Wheel 文件

```
Uploading eos-0.0.2-py3-none-any.whl
100% ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━ 105.4/105.4 kB • 00:00 • 45.7 MB/s
```

**说明**:
- 文件大小: 105.4 KB
- 上传速度: 45.7 MB/s
- 进度条显示: 100% 完成

---

#### 6.2 上传源代码包

```
Uploading eos-0.0.2.tar.gz
100% ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━ 242.5/242.5 kB • 00:00 • 2.5 MB/s
```

**说明**:
- 文件大小: 242.5 KB
- 上传速度: 2.5 MB/s（比 wheel 慢，因为文件更大）
- 进度条显示: 100% 完成

---

### 步骤 7: 上传成功

```bash
View at:
https://test.pypi.org/project/eos/0.0.2/

✅ 上传成功！

查看: https://test.pypi.org/project/eos/
安装: pip install -i https://test.pypi.org/simple/ eos
```

**说明**:
- ✅ 上传完成
- 📦 包已发布到 TestPyPI
- 🔗 提供了查看和安装链接

---

## 🔄 完整流程图

```
make publish-test
    ↓
┌─────────────────────────────────────┐
│ 1. make clean                       │
│    ├─ 删除 dist/, build/            │
│    ├─ 删除缓存文件                   │
│    └─ 删除测试文件                   │
└─────────────────────────────────────┘
    ↓
┌─────────────────────────────────────┐
│ 2. make build                       │
│    ├─ 创建隔离环境                   │
│    ├─ 安装构建依赖                   │
│    ├─ 构建 .tar.gz                  │
│    └─ 构建 .whl                     │
└─────────────────────────────────────┘
    ↓
┌─────────────────────────────────────┐
│ 3. twine check                      │
│    └─ 验证分发包格式                 │
└─────────────────────────────────────┘
    ↓
┌─────────────────────────────────────┐
│ 4. 显示包信息                       │
│    └─ 要求用户确认                  │
└─────────────────────────────────────┘
    ↓
┌─────────────────────────────────────┐
│ 5. twine upload                     │
│    ├─ 上传 .whl                     │
│    └─ 上传 .tar.gz                  │
└─────────────────────────────────────┘
    ↓
✅ 上传成功！
```

---

## 📊 关键概念解释

### 1. 隔离环境 (Isolated Environment)

**什么是隔离环境**:
- 一个临时的虚拟环境
- 只包含构建所需的依赖
- 与系统环境完全隔离

**为什么需要**:
```
系统环境可能有:
- Python 3.9
- 各种已安装的包
- 可能版本不匹配

隔离环境:
- 干净的 Python 环境
- 只安装构建依赖
- 版本完全可控
```

### 2. 两种分发包格式

| 格式 | 文件 | 大小 | 安装速度 | 兼容性 |
|------|------|------|----------|--------|
| **sdist** | `.tar.gz` | 219KB | 慢（需要编译） | 最好 |
| **wheel** | `.whl` | 85KB | 快（已编译） | 好 |

**现代 pip 的行为**:
```
pip install eos
  ↓
1. 先尝试下载 .whl（如果可用）
   ✅ 成功 → 直接安装（快）
   ❌ 失败 → 下载 .tar.gz
2. 下载 .tar.gz
   → 解压 → 编译 → 安装（慢）
```

### 3. TestPyPI vs PyPI

| 平台 | 用途 | URL | 特点 |
|------|------|-----|------|
| **TestPyPI** | 测试发布 | test.pypi.org | 可以重复上传相同版本 |
| **PyPI** | 正式发布 | pypi.org | 不能重复上传相同版本 |

**为什么先发 TestPyPI**:
- ✅ 测试上传流程
- ✅ 验证包是否正确
- ✅ 可以重复测试
- ✅ 不影响正式 PyPI

---

## ⚙️ 底层命令详解

### `python -m build`

**等价于**:
```bash
# 旧方式（已弃用）
python setup.py sdist bdist_wheel

# 新方式（推荐）
python -m build
```

**优势**:
- ✅ 使用 `pyproject.toml`（现代标准）
- ✅ 自动创建隔离环境
- ✅ 更可靠、更安全

### `twine check`

**检查内容**:
```python
# 伪代码
def check(package_file):
    # 1. 验证文件格式
    validate_format(package_file)
    
    # 2. 验证元数据
    metadata = extract_metadata(package_file)
    validate_metadata(metadata)
    
    # 3. 验证 README
    if has_readme(metadata):
        validate_readme_format(metadata.readme)
    
    # 4. 检查必需字段
    required_fields = ['name', 'version', 'author']
    for field in required_fields:
        assert field in metadata
```

### `twine upload`

**上传过程**:
```python
# 伪代码
def upload(files, repository='pypi'):
    # 1. 读取认证信息
    username = os.getenv('TWINE_USERNAME')
    password = os.getenv('TWINE_PASSWORD')
    
    # 2. 连接到服务器
    connection = connect(repository_url, username, password)
    
    # 3. 上传每个文件
    for file in files:
        upload_file(connection, file)
        show_progress(file)
    
    # 4. 验证上传
    verify_upload(connection, files)
```

---

## 🎯 最佳实践

### 1. 总是先清理

```bash
make clean  # 确保干净构建
make build  # 构建
```

### 2. 先测试后正式

```bash
make publish-test  # 先发 TestPyPI
# 测试安装
pip install -i https://test.pypi.org/simple/ eos
# 确认无误后
make publish-pypi  # 再发正式 PyPI
```

### 3. 检查构建结果

```bash
# 查看生成的文件
ls -lh dist/

# 检查文件
twine check dist/*
```

### 4. 验证上传

```bash
# 从 TestPyPI 安装测试
pip install -i https://test.pypi.org/simple/ eos

# 验证版本
python -c "from eos import __version__; print(__version__)"
```

---

## 📝 总结

### 核心步骤

1. **清理** - 删除旧文件
2. **构建** - 创建分发包（.whl 和 .tar.gz）
3. **检查** - 验证分发包格式
4. **确认** - 用户确认上传
5. **上传** - 上传到 TestPyPI
6. **完成** - 显示成功信息

### 关键文件

- `pyproject.toml` - 项目配置和元数据
- `dist/*.whl` - Wheel 分发包
- `dist/*.tar.gz` - 源代码分发包

### 重要工具

- `build` - 构建分发包
- `twine` - 检查和上传分发包

整个过程自动化、安全、可靠！🚀
