from typing import TYPE_CHECKING, Literal

from dev_utils.alembic.utils import get_config_variable_as_list

if TYPE_CHECKING:
    from alembic.config import Config
    from sqlalchemy.sql.schema import SchemaItem

    Name = str | None
    Type = Literal[
        "schema",
        "table",
        "column",
        "index",
        "unique_constraint",
        "foreign_key_constraint",
    ]


def build_include_object_function(
    config: "Config",
    exclude_tables_config_key: str,
    exclude_tables_config_section: str | None = None,
):
    """Function-wrapper to build ``include_object`` function.

    Function get list of exclude_tables from ``alembic.ini`` config and then exclude tables from
    autogenerated migration.

    Useful for tables, which created by database extensions, which you have no access for.
    """
    exclude_tables = set(
        get_config_variable_as_list(
            config,
            exclude_tables_config_key,
            exclude_tables_config_section,
        )
    )

    def include_object(
        object: "SchemaItem",  # noqa
        name: "Name",
        type_: "Type",
        reflected: bool,  # noqa
        compare_to: "SchemaItem | None",  # noqa
    ) -> bool:  # noqa
        """Custom filter function for inner alembic use.

        Check if table in `exclude tables` variable: if table name in `exclude_tables`, don't pass it in
        autogenerated migrations, else pass.

        https://alembic.sqlalchemy.org/en/latest/api/runtime.html#alembic.runtime.environment.EnvironmentContext.configure.params.include_object
        """
        return not (type_ == "table" and name in exclude_tables)

    return include_object
