
import inspect

from django.db.models import QuerySet
from django.urls import path
from rest_framework.decorators import renderer_classes, api_view
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response

from Injection.utils import Singleton, UrlDefinitionIncomplete, parametrized, pass_handler, validate_allowed_method, \
    extract_res_serialiser, extract_req_data, verify_permissions


class InstanceMaintainer(metaclass=Singleton):
    urls = []
    managers = {}

    def add_url(self, method=None, url=None, handler=None):
        if not (url) or not (handler) or not (method):
            raise UrlDefinitionIncomplete("Please Give Url, function and method to create a route")
        self.urls.append({'method': method, 'url': url, 'handler': handler})

    def add_manager(self, obj=None):
        if not (obj):
            raise UrlDefinitionIncomplete("Please make this class exposeable")
        self.managers[obj.__class__.__name__] = obj

    def export_urls(self):
        return [path(f'{u["url"]}', u['handler']) for u in self.urls]


root_instance = InstanceMaintainer()

def exposeable(f):
    def inner(self, *args, **kwargs):
        root_instance.add_manager(self)
        return f(self, *args, **kwargs)

    return inner


@parametrized
def expose(function, method=None, url=None, response_serializer=None, schema=None, permissions=None):
    if not url:
        url = f'api/v1/{function.__module__.split(".")[0]}/{function.__name__}/'
    if '.' not in function.__qualname__:
        root_instance.add_url(method, url, handler_for_util(query_method=function,permissions=permissions, method=method, response_serializer=response_serializer,schema=schema))
    else:
        root_instance.add_url(method, url, handler(query_method=function, method=method,permissions=permissions, response_serializer=response_serializer, schema=schema))
    return function


def handler_for_util(query_method=None, method=None, response_serializer=None, schema=None, permissions=None):
    @api_view(('GET', 'POST'))
    @renderer_classes((JSONRenderer,))
    @pass_handler(query_method=query_method, response_serializer=response_serializer, schema=schema, permissions=permissions)
    def inner(request, *args, **kwargs):
        validate_allowed_method(method, request.method)
        if type(args[1]) == tuple:
            for perm in args[1]:
                verify_permissions(request,args[1])
        else:
            verify_permissions(request,args[1])
        serializer = extract_res_serialiser(response_serializer)
        query_method = args[0]
        req_data, method_sp_args, method_sp_kwargs, __filter_criteria = extract_req_data(method, request, schema)

        params_req = inspect.getargspec(query_method).args or []

        given_method_args = tuple([req_data.get(i) for i in params_req])

        try:
            query = query_method(*given_method_args,*method_sp_args,**method_sp_kwargs)
            if __filter_criteria  and type(query)==QuerySet:
                query = query.filter(**__filter_criteria)
            if serializer:
                return Response(serializer(query, many=query.count() > 1).data if query else None)
            return Response(query.values() if query else None)
        except Exception as e:
            return Response({'error': e.__str__()})
    return inner

def handler(query_method=None,  method=None, response_serializer=None, schema=None, permissions=None):
    @api_view(('GET', 'POST'))
    @renderer_classes((JSONRenderer,))
    @pass_handler(query_method=query_method, permissions=permissions, response_serializer=response_serializer, schema=schema)
    def inner(request, *args, **kwargs):
        validate_allowed_method(method, request.method)
        if type(args[1]) == tuple:
            for perm in args[1]:
                verify_permissions(request,args[1])
        else:
            verify_permissions(request,args[1])
        serializer = extract_res_serialiser(response_serializer)
        query_method = args[0]
        manager = root_instance.managers[query_method.__qualname__.split('.')[0]]
        req_data, method_sp_args, method_sp_kwargs, __filter_criterea = extract_req_data(method, request, schema)
        params_req = inspect.getargspec(query_method).args[1:] or []
        given_method_args = tuple([req_data.get(i) for i in params_req])
        try:
            query = query_method(manager, *given_method_args, *method_sp_args, **method_sp_kwargs)
            if __filter_criterea and type(query)==QuerySet:
                query = query.filter(**__filter_criterea)
            if serializer:
                return Response(serializer(query, many=query.count() > 1).data if query else None)
            return Response(query.values())
        except Exception as e:
            return Response({'error': e.__str__()})

    return inner
