"""Server module for the Python Docker MCP package.

This module provides the MCP server implementation that handles API requests
and dispatches them to the Docker execution environment.
"""

import asyncio
import logging
import os
import sys
import uuid
from typing import Any, Dict

import mcp.server.stdio
import mcp.types as types
from mcp.server import NotificationOptions, Server
from mcp.server.models import InitializationOptions
from pydantic import AnyUrl

from .config import load_config
from .docker_manager import DockerManager

# Set up logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("python-docker-mcp")

# Initialize the configuration
config = load_config()

# Initialize the Docker manager
docker_manager = DockerManager(config)

# Store sessions for persistent code execution environments
sessions = {}

# Create the MCP server
server = Server("python-docker-mcp")


@server.list_resources()
async def handle_list_resources() -> list[types.Resource]:
    """List available resources.

    Currently there are no resources to list.
    """
    return []


@server.read_resource()
async def handle_read_resource(uri: AnyUrl) -> str:
    """Read a specific resource by its URI.

    Currently there are no resources to read.
    """
    raise ValueError(f"Unsupported resource URI: {uri}")


@server.list_prompts()
async def handle_list_prompts() -> list[types.Prompt]:
    """List available prompts.

    Currently there are no prompts defined.
    """
    return []


@server.get_prompt()
async def handle_get_prompt(name: str, arguments: dict[str, str] | None) -> types.GetPromptResult:
    """Generate a prompt.

    Currently there are no prompts defined.
    """
    raise ValueError(f"Unknown prompt: {name}")


@server.list_tools()
async def handle_list_tools() -> list[types.Tool]:
    """List available tools that can be called by clients."""
    logger.info("Listing tools")
    return [
        types.Tool(
            name="execute-transient",
            description="Execute Python code in a transient Docker container",
            inputSchema={
                "type": "object",
                "properties": {
                    "code": {"type": "string", "description": "Python code to execute"},
                    "state": {"type": "object", "description": "Optional state dictionary"},
                },
                "required": ["code"],
            },
        ),
        types.Tool(
            name="execute-persistent",
            description="Execute Python code in a persistent Docker container",
            inputSchema={
                "type": "object",
                "properties": {
                    "code": {"type": "string", "description": "Python code to execute"},
                    "session_id": {"type": "string", "description": "Session identifier"},
                },
                "required": ["code"],
            },
        ),
        types.Tool(
            name="install-package",
            description="Install a Python package in a Docker container",
            inputSchema={
                "type": "object",
                "properties": {
                    "package_name": {"type": "string", "description": "Package name"},
                    "session_id": {"type": "string", "description": "Optional session ID"},
                },
                "required": ["package_name"],
            },
        ),
        types.Tool(
            name="cleanup-session",
            description="Clean up a persistent session and its resources",
            inputSchema={
                "type": "object",
                "properties": {
                    "session_id": {"type": "string", "description": "Session identifier"},
                },
                "required": ["session_id"],
            },
        ),
    ]


def _format_execution_result(result: Dict[str, Any]) -> str:
    """Format the execution result for the MCP response."""
    if result.get("status") == "error":
        error = result.get("error", "Unknown error occurred")
        error_info = result.get("error_info", {})
        stdout = result.get("stdout", "")
        stderr = result.get("stderr", "")

        return f"Error: {error}\nError Info: {error_info}\nOutput: {stdout}\nError Output: {stderr}"

    # For successful execution, return the output
    return result.get("stdout", "")


@server.call_tool()
async def handle_call_tool(name: str, arguments: dict | None) -> list[types.TextContent | types.ImageContent | types.EmbeddedResource]:
    """Handle tool execution requests for Python code execution and package management."""
    logger.info(f"Calling tool: {name}")

    if not arguments:
        raise ValueError("Missing arguments")

    try:
        if name == "execute-transient":
            code = arguments.get("code")
            state = arguments.get("state", {})

            if not code:
                raise ValueError("Missing code")

            result = await docker_manager.execute_transient(code, state)
            output = _format_execution_result(result)
            return [types.TextContent(type="text", text=output)]

        elif name == "execute-persistent":
            code = arguments.get("code")
            session_id = arguments.get("session_id")

            if not code:
                raise ValueError("Missing code")

            # Create a new session if not provided
            if not session_id:
                session_id = str(uuid.uuid4())
                sessions[session_id] = {"created_at": asyncio.get_event_loop().time()}

            result = await docker_manager.execute_persistent(session_id, code)
            output = _format_execution_result(result)
            return [types.TextContent(type="text", text=output)]

        elif name == "install-package":
            package_name = arguments.get("package_name")
            session_id = arguments.get("session_id")

            if not package_name:
                raise ValueError("Missing package name")

            output = await docker_manager.install_package(session_id, package_name)
            return [types.TextContent(type="text", text=f"Package installation result:\n\n{output}")]

        elif name == "cleanup-session":
            session_id = arguments.get("session_id")

            if not session_id:
                raise ValueError("Missing session ID")

            result = await docker_manager.cleanup_session(session_id)

            if session_id in sessions:
                del sessions[session_id]

            return [types.TextContent(type="text", text=f"Session {session_id} cleaned up successfully")]

        else:
            raise ValueError(f"Unknown tool: {name}")
    except Exception as e:
        logger.error(f"Error executing tool {name}: {str(e)}")
        import traceback

        logger.error(traceback.format_exc())

        # Return a properly formatted error response
        error_message = f"Error executing {name}: {str(e)}"
        return [types.TextContent(type="text", text=error_message)]


async def main() -> None:
    """Start the MCP server."""
    # Configure logging based on debug flag from command line or environment
    debug_mode = "--debug" in sys.argv or os.environ.get("PYTHON_DOCKER_MCP_DEBUG", "").lower() in ["true", "1", "yes"]

    if debug_mode:
        logging.basicConfig(level=logging.DEBUG)
        logger.setLevel(logging.DEBUG)
        logger.debug("Debug mode enabled")
    else:
        # Set info level logging by default for better diagnostics
        logging.basicConfig(level=logging.INFO)

    # Disable pooling for now until we can verify basic functionality
    try:
        logger.info("Temporarily disabling container pooling for stability")
        if hasattr(config.docker, "pool_enabled"):
            config.docker.pool_enabled = False
    except Exception as e:
        logger.error(f"Error configuring container pooling: {e}")

    # Run the server using stdin/stdout streams
    logger.info("Starting MCP server using stdio transport")
    try:
        async with mcp.server.stdio.stdio_server() as (read_stream, write_stream):
            logger.info("stdio server initialized, running MCP server")
            await server.run(
                read_stream,
                write_stream,
                InitializationOptions(
                    server_name="python-docker-mcp",
                    server_version="0.1.10",
                    capabilities=server.get_capabilities(
                        notification_options=NotificationOptions(),
                        experimental_capabilities={},
                    ),
                ),
            )
    except Exception as e:
        logger.error(f"Error running MCP server: {e}")
        import traceback

        logger.error(traceback.format_exc())
        raise
    finally:
        # Clean up any remaining sessions when the server shuts down
        logger.info("Cleaning up sessions")
        for session_id in list(sessions.keys()):
            try:
                await docker_manager.cleanup_session(session_id)
            except Exception as e:
                logger.error(f"Error cleaning up session {session_id}: {e}")

        logger.info("Server shutdown complete")


# If this module is run directly, start the server
if __name__ == "__main__":
    asyncio.run(main())
