"""
Test to verify that deprecation warnings are properly shown when using the old historian module.
This ensures users are properly notified about the deprecation.
"""
import sys
import os
import unittest
import warnings

# Import the common test configuration
sys.path.append(os.path.join(os.path.dirname(__file__), '..'))
from common import BaseHistorianTest

class TestDeprecationWarning(BaseHistorianTest):
    """Test class to verify deprecation warnings are working correctly"""
    
    def test_historian_deprecation_warning_is_raised(self):
        """Test that using get_historian raises a deprecation warning"""
        from eigeningenuity import EigenServer, get_historian
        
        # Capture warnings to verify the deprecation warning is raised
        with warnings.catch_warnings(record=True) as warning_list:
            # Ensure all warnings are captured
            warnings.simplefilter("always")
            
            # This should trigger the deprecation warning
            server = EigenServer("demo-dev.eigen.co")
            _ = get_historian("Demo-influxdb", server)  # Create historian to trigger warning
            
            # Verify that at least one deprecation warning was raised
            deprecation_warnings = [w for w in warning_list if issubclass(w.category, DeprecationWarning)]
            self.assertGreater(len(deprecation_warnings), 0, 
                               "Expected deprecation warning was not raised when using get_historian")
            
            # Verify the specific deprecation message
            expected_message = "eigeningenuity.historian.get_historian is deprecated"
            found_expected_warning = any(expected_message in str(w.message) for w in deprecation_warnings)
            self.assertTrue(found_expected_warning, 
                          f"Expected deprecation warning message not found. Got warnings: {[str(w.message) for w in deprecation_warnings]}")
    
    def test_historian_multi_does_not_raise_deprecation_warning(self):
        """Test that using get_historian_multi does NOT raise a deprecation warning"""
        from eigeningenuity.historianmulti import get_historian_multi
        from eigeningenuity import EigenServer
        
        # Capture warnings to verify no deprecation warning is raised
        with warnings.catch_warnings(record=True) as warning_list:
            # Ensure all warnings are captured
            warnings.simplefilter("always")
            
            # This should NOT trigger a deprecation warning
            server = EigenServer("demo-dev.eigen.co")
            _ = get_historian_multi(server, "Demo-influxdb")  # Create historian_multi (should not warn)
            
            # Verify that no deprecation warnings were raised for historian
            historian_deprecation_warnings = [w for w in warning_list 
                                            if issubclass(w.category, DeprecationWarning) 
                                            and "historian" in str(w.message).lower()]
            self.assertEqual(len(historian_deprecation_warnings), 0, 
                           f"Unexpected deprecation warning raised when using get_historian_multi: {[str(w.message) for w in historian_deprecation_warnings]}")

if __name__ == '__main__':
    unittest.main()