#!/usr/bin/env python
# -*- coding: utf-8 -*-

__author__ = 'Bruno Stuyts'

from pyeng.general.validation import ValidationDecorator
import numpy as np

PRESSUREDROP_RELATIVEROUGHNESS_MOODY = {
    'reynolds_number': {'type': 'float', 'min_value': 500.0, 'max_value': 100000000.0},
    'pipe_diameter': {'type': 'float', 'min_value': 0.0, 'max_value': None},
    'pipe_material': {'type': 'string', 'options': (
    "Concrete,coarse", "Concrete,new smooth", "Drawn tubing", "Glass,plastic,perspex", "Iron,cast", "Sewers,old",
    "Steel,mortar lined", "Steel,rusted", "Steel,structural or forged", "Water mains,old"), 'regex': None},
    'pipe_length': {'type': 'float', 'min_value': 0.0, 'max_value': None},
    'average_velocity': {'type': 'float', 'min_value': 0.0, 'max_value': None},
    'fluid_density': {'type': 'float', 'min_value': 500.0, 'max_value': 2500.0},
    'gravity_coefficient': {'type': 'float', 'min_value': 9.7, 'max_value': 10.0},
    'relative_roughness': {'type': 'float', 'min_value': None, 'max_value': 0.05},
}


@ValidationDecorator(PRESSUREDROP_RELATIVEROUGHNESS_MOODY)
def pressuredrop_relativeroughness_moody(reynolds_number, pipe_diameter, pipe_material, pipe_length, average_velocity,
                                         fluid_density, gravity_coefficient=9.81, relative_roughness=np.NaN,
                                         fail_silently=True, **kwargs):
    """
    The Moody diagram relates Reynolds number to Darcy-Weissbach friction factor. This friction factor is used to
    calculate the head loss in a pipe. The underlying dataset consists of more than 10,000 experiments from various
    sources. For the laminar flow regime (Re < 3000), roughness has no discernible effect,
    and the Darcy-Weisbach friction factor was determined analytically by Poisseuille. For the turbulent flow regime,
    the relationship between the friction factor, the Reynolds number Re, and the relative roughness is more complex
    and is derived from the Moody chart. The figure shows values for typical pipe walls. These values are coded into
    the function but an override can also be specified.

    :param reynolds_number: Reynolds number (:math:`Re`) [:math:`-`]  - Suggested range: 500.0<=reynolds_number<=100000000.0
    :param pipe_diameter: Pipe inside diameter (:math:`d`) [:math:`m`]  - Suggested range: 0.0<=pipe_diameter
    :param pipe_material: Pipe material [:math:`-`] Options: ("Concrete,coarse","Concrete,new smooth","Drawn tubing","Glass,plastic,perspex","Iron,cast","Sewers,old","Steel,mortar lined","Steel,rusted","Steel,structural or forged","Water mains,old"), regex: None
    :param pipe_length: Pipe length (:math:`L`) [:math:`m`]  - Suggested range: 0.0<=pipe_length
    :param average_velocity: Average velocity of the fluid in the pipe (:math:`V`) [:math:`m/s`]  - Suggested range: 0.0<=average_velocity
    :param fluid_density: Density of the fluid (:math:`\\rho`) [:math:`kg/m3`]  - Suggested range: 500.0<=fluid_density<=2500.0
    :param gravity_coefficient: Acceleration due to gravity (:math:`g`) [:math:`m/s2`] (optional, default=9.81) - Suggested range: 9.7<=gravity_coefficient<=10.0
    :param relative_roughness: Relative roughness override [:math:`-`] (optional, default=np.NaN) - Suggested range: relative_roughness<=0.05

    .. math::
        \\text{Head loss:} \\quad f_D \\frac{L}{d} \\frac{V^2}{2 g}

        \\text{Pressure drop:} \\quad \\rho g h_f

        \\text{Relative pipe roughness} = \\epsilon / d

        \\text{Laminar flow:} \\quad f_D = \\frac{64}{Re}

    :returns:   Darcy Weissbach friction factor (:math:`f_D`) [-], Pipe roughness (:math:`\\epsilon`) [:math:`mm`], Head loss (:math:`h_f`) [:math:`m`], Pressure drop (:math:`\\Delta P`) [:math:`Pa`], Flow regime [-], Friction factor for laminar flow (:math:`f_{d,laminar}`) [-], Friction factor for fully turbulent flow (:math:`f_{d,turbulent}`) [-]

    :rtype: Python dictionary with keys ['friction_factor [-]','roughness [mm]','head_loss [m]','pressure_drop [Pa]','flow_regime [-]','friction_factor_laminar [-]','friction_factor_turbulent [-]']


    .. figure:: images/Moody_diagram_matplotlib.png
        :figwidth: 500
        :width: 400
        :align: center

        Moody diagram - Matplotlib output

    Reference - Moody, L. F. (1944), "Friction factors for pipe flow", Transactions of the ASME, 66 (8): 671-684

    """

    try:

        if not kwargs['validated']:
            raise ValueError("Error during function validation, %s" % kwargs['errorstring'])

        # Calculation statements
        if not np.isnan(relative_roughness):
            roughness = relative_roughness * pipe_diameter * 1000.0
        else:
            if pipe_material == "Concrete,coarse":
                roughness = 0.25
            elif pipe_material == "Concrete,new smooth":
                roughness = 0.025
            elif pipe_material == "Drawn tubing":
                roughness = 0.0025
            elif pipe_material == "Glass,plastic,perspex":
                roughness = 0.0025
            elif pipe_material == "Iron,cast":
                roughness = 0.15
            elif pipe_material == "Sewers,old":
                roughness = 3.0
            elif pipe_material == "Steel,mortar lined":
                roughness = 0.1
            elif pipe_material == "Steel,rusted":
                roughness = 0.5
            elif pipe_material == "Steel,structural or forged":
                roughness = 0.025
            elif pipe_material == "Water mains,old":
                roughness = 1.0
            else:
                raise ValueError(
                    "Pipe material not recognized, select from the options or specify relative roughness directly")

            relative_roughness = 0.001 * roughness / pipe_diameter

        if relative_roughness > 0.05:
            friction_factor = 10.0 ** (np.interp(np.log10(reynolds_number),
                                                 [3.30443548387097, 3.48588709677419, 3.6491935483871, 3.84879032258064,
                                                  4.06048387096774, 4.28427419354839, 4.5625, 4.74395161290323,
                                                  5.16129032258065,
                                                  6.21975806451613, 7.99798387096774],
                                                 [-1.06711409395973, -1.08724832214765, -1.10290827740492,
                                                  -1.11633109619687,
                                                  -1.12751677852349, -1.13199105145414, -1.13870246085011,
                                                  -1.14093959731544,
                                                  -1.14317673378076, -1.14317673378076, -1.14541387024608]))
        elif 0.05 >= relative_roughness > 0.04:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.30443548387097, 3.48588709677419, 3.6491935483871, 3.84879032258064,
                                      4.06048387096774, 4.28427419354839, 4.5625, 4.74395161290323, 5.16129032258065,
                                      6.21975806451613, 7.99798387096774],
                                     [-1.06711409395973, -1.08724832214765, -1.10290827740492, -1.11633109619687,
                                      -1.12751677852349, -1.13199105145414, -1.13870246085011, -1.14093959731544,
                                      -1.14317673378076, -1.14317673378076, -1.14541387024608])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.50403225806451, 3.89112903225806, 4.24798387096774,
                                      4.5866935483871, 4.88911290322581, 5.19758064516129, 5.83266129032258,
                                      6.36491935483871, 6.91532258064516, 7.99798387096774],
                                     [-1.09619686800895, -1.12527964205817, -1.15883668903803, -1.1744966442953,
                                      -1.18120805369127, -1.18568232662192, -1.18791946308725, -1.18791946308725,
                                      -1.18791946308725, -1.18791946308725, -1.18791946308725])
                           )
            friction_factor = np.interp(relative_roughness, [0.04, 0.05], [ff2, ff1])
        elif 0.04 >= relative_roughness > 0.03:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.50403225806451, 3.89112903225806, 4.24798387096774,
                                      4.5866935483871, 4.88911290322581, 5.19758064516129, 5.83266129032258,
                                      6.36491935483871, 6.91532258064516, 7.99798387096774],
                                     [-1.09619686800895, -1.12527964205817, -1.15883668903803, -1.1744966442953,
                                      -1.18120805369127, -1.18568232662192, -1.18791946308725, -1.18791946308725,
                                      -1.18791946308725, -1.18791946308725, -1.18791946308725])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.42540322580645, 3.57056451612903, 3.73991935483871,
                                      3.90927419354839, 4.13306451612903, 4.51411290322581, 4.82258064516129,
                                      5.04032258064516, 5.35483870967742, 5.53024193548387, 5.82661290322581,
                                      6.0866935483871, 6.47379032258064, 6.90322580645161, 7.3991935483871,
                                      7.99798387096774],
                                     [-1.13422818791946, -1.15212527964206, -1.1744966442953, -1.1923937360179,
                                      -1.20581655480984, -1.21923937360179, -1.23042505592841, -1.23713646532438,
                                      -1.23937360178971, -1.24161073825503, -1.24161073825503, -1.24161073825503,
                                      -1.24161073825503, -1.24384787472036, -1.24161073825503, -1.24161073825503,
                                      -1.24161073825503])
                           )
            friction_factor = np.interp(relative_roughness, [0.03, 0.04], [ff2, ff1])
        elif 0.03 >= relative_roughness > 0.02:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.42540322580645, 3.57056451612903, 3.73991935483871,
                                      3.90927419354839, 4.13306451612903, 4.51411290322581, 4.82258064516129,
                                      5.04032258064516, 5.35483870967742, 5.53024193548387, 5.82661290322581,
                                      6.0866935483871, 6.47379032258064, 6.90322580645161, 7.3991935483871,
                                      7.99798387096774],
                                     [-1.13422818791946, -1.15212527964206, -1.1744966442953, -1.1923937360179,
                                      -1.20581655480984, -1.21923937360179, -1.23042505592841, -1.23713646532438,
                                      -1.23937360178971, -1.24161073825503, -1.24161073825503, -1.24161073825503,
                                      -1.24161073825503, -1.24384787472036, -1.24161073825503, -1.24161073825503,
                                      -1.24161073825503])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.43145161290322, 3.61290322580645, 3.82459677419355,
                                      4.07258064516129, 4.29637096774193, 4.61693548387097, 4.87701612903226,
                                      5.13709677419355, 5.37298387096774, 5.76612903225806, 6.12903225806451,
                                      6.52822580645161, 7.14516129032258, 7.99798387096774],
                                     [-1.17673378076063, -1.19910514541387, -1.23042505592841, -1.2572706935123,
                                      -1.27740492170022, -1.29082774049217, -1.29977628635347, -1.30425055928412,
                                      -1.30872483221477, -1.31096196868009, -1.31096196868009, -1.31319910514541,
                                      -1.31319910514541, -1.31319910514541, -1.31319910514541])
                           )
            friction_factor = np.interp(relative_roughness, [0.02, 0.03], [ff2, ff1])
        elif 0.02 >= relative_roughness > 0.015:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.43145161290322, 3.61290322580645, 3.82459677419355,
                                      4.07258064516129, 4.29637096774193, 4.61693548387097, 4.87701612903226,
                                      5.13709677419355, 5.37298387096774, 5.76612903225806, 6.12903225806451,
                                      6.52822580645161, 7.14516129032258, 7.99798387096774],
                                     [-1.17673378076063, -1.19910514541387, -1.23042505592841, -1.2572706935123,
                                      -1.27740492170022, -1.29082774049217, -1.29977628635347, -1.30425055928412,
                                      -1.30872483221477, -1.31096196868009, -1.31096196868009, -1.31319910514541,
                                      -1.31319910514541, -1.31319910514541, -1.31319910514541])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.4616935483871, 3.66733870967742, 3.84879032258064,
                                      4.09677419354838, 4.44153225806452, 4.7741935483871, 5.11895161290322,
                                      5.59072580645161, 6.27419354838709, 6.84879032258064, 7.32661290322581,
                                      7.78629032258064, 7.99798387096774],
                                     [-1.19910514541387, -1.23266219239374, -1.26845637583893, -1.29306487695749,
                                      -1.31767337807606, -1.33557046979866, -1.34675615212528, -1.35346756152125,
                                      -1.35570469798658, -1.36017897091723, -1.36017897091723, -1.36017897091723,
                                      -1.36017897091723, -1.36017897091723])
                           )
            friction_factor = np.interp(relative_roughness, [0.015, 0.02], [ff2, ff1])
        elif 0.015 >= relative_roughness > 0.01:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.4616935483871, 3.66733870967742, 3.84879032258064,
                                      4.09677419354838, 4.44153225806452, 4.7741935483871, 5.11895161290322,
                                      5.59072580645161, 6.27419354838709, 6.84879032258064, 7.32661290322581,
                                      7.78629032258064, 7.99798387096774],
                                     [-1.19910514541387, -1.23266219239374, -1.26845637583893, -1.29306487695749,
                                      -1.31767337807606, -1.33557046979866, -1.34675615212528, -1.35346756152125,
                                      -1.35570469798658, -1.36017897091723, -1.36017897091723, -1.36017897091723,
                                      -1.36017897091723, -1.36017897091723])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.38911290322581, 3.52217741935484, 3.74596774193548,
                                      3.98790322580645, 4.23588709677419, 4.48387096774193, 4.83467741935484,
                                      5.28225806451613, 5.92943548387097, 6.40725806451613, 6.9758064516129,
                                      7.99798387096774],
                                     [-1.22645739910314, -1.24887892376682, -1.28026905829596, -1.32062780269058,
                                      -1.3542600896861, -1.37892376681614, -1.39461883408072, -1.40807174887892,
                                      -1.41479820627803, -1.41928251121076, -1.42152466367713, -1.42152466367713,
                                      -1.42152466367713])
                           )
            friction_factor = np.interp(relative_roughness, [0.01, 0.015], [ff2, ff1])
        elif 0.01 >= relative_roughness > 0.005:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.38911290322581, 3.52217741935484, 3.74596774193548,
                                      3.98790322580645, 4.23588709677419, 4.48387096774193, 4.83467741935484,
                                      5.28225806451613, 5.92943548387097, 6.40725806451613, 6.9758064516129,
                                      7.99798387096774],
                                     [-1.22645739910314, -1.24887892376682, -1.28026905829596, -1.32062780269058,
                                      -1.3542600896861, -1.37892376681614, -1.39461883408072, -1.40807174887892,
                                      -1.41479820627803, -1.41928251121076, -1.42152466367713, -1.42152466367713,
                                      -1.42152466367713])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.4375, 3.55846774193548, 3.74596774193548, 3.90927419354839,
                                      4.05443548387097, 4.30241935483871, 4.57459677419355, 4.85887096774194, 5.125,
                                      5.43951612903226, 5.76008064516129, 6.16532258064516, 6.47983870967742,
                                      6.96370967741936, 7.99798387096774],
                                     [-1.25784753363229, -1.29372197309417, -1.32735426008969, -1.36995515695067,
                                      -1.39910313901345, -1.42600896860987, -1.45739910313901, -1.47982062780269,
                                      -1.49551569506727, -1.50448430493274, -1.51121076233184, -1.51569506726457,
                                      -1.51569506726457, -1.51793721973094, -1.51793721973094, -1.51793721973094])
                           )
            friction_factor = np.interp(relative_roughness, [0.005, 0.01], [ff2, ff1])
        elif 0.005 >= relative_roughness > 0.002:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.4375, 3.55846774193548, 3.74596774193548, 3.90927419354839,
                                      4.05443548387097, 4.30241935483871, 4.57459677419355, 4.85887096774194, 5.125,
                                      5.43951612903226, 5.76008064516129, 6.16532258064516, 6.47983870967742,
                                      6.96370967741936, 7.99798387096774],
                                     [-1.25784753363229, -1.29372197309417, -1.32735426008969, -1.36995515695067,
                                      -1.39910313901345, -1.42600896860987, -1.45739910313901, -1.47982062780269,
                                      -1.49551569506727, -1.50448430493274, -1.51121076233184, -1.51569506726457,
                                      -1.51569506726457, -1.51793721973094, -1.51793721973094, -1.51793721973094])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.59475806451613, 3.77620967741935,
                                      3.9758064516129, 4.1633064516129, 4.45967741935484, 4.73790322580645,
                                      5.05846774193548, 5.42137096774193, 5.95362903225806, 6.55241935483871,
                                      7.99798387096774],
                                     [-1.2780269058296, -1.32286995515695, -1.36547085201794, -1.41479820627803,
                                      -1.45964125560538, -1.49775784753363, -1.54484304932735, -1.5762331838565,
                                      -1.59865470852018, -1.61659192825112, -1.62556053811659, -1.63004484304933,
                                      -1.6322869955157])
                           )
            friction_factor = np.interp(relative_roughness, [0.002, 0.005], [ff2, ff1])
        elif 0.002 >= relative_roughness > 0.001:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.59475806451613, 3.77620967741935,
                                      3.9758064516129, 4.1633064516129, 4.45967741935484, 4.73790322580645,
                                      5.05846774193548, 5.42137096774193, 5.95362903225806, 6.55241935483871,
                                      7.99798387096774],
                                     [-1.2780269058296, -1.32286995515695, -1.36547085201794, -1.41479820627803,
                                      -1.45964125560538, -1.49775784753363, -1.54484304932735, -1.5762331838565,
                                      -1.59865470852018, -1.61659192825112, -1.62556053811659, -1.63004484304933,
                                      -1.6322869955157])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.61290322580645, 3.76411290322581,
                                      3.91532258064516, 4.13306451612903, 4.41129032258064, 4.76814516129032,
                                      5.01008064516129, 5.30040322580645, 5.65120967741935, 6.00201612903226,
                                      6.32862903225806, 6.6491935483871, 6.91532258064516, 7.3991935483871,
                                      7.99798387096774],
                                     [-1.28699551569507, -1.33183856502242, -1.38340807174888, -1.42600896860987,
                                      -1.46412556053812, -1.51569506726457, -1.57174887892377, -1.62556053811659,
                                      -1.65246636771301, -1.67488789237668, -1.69058295964126, -1.69955156950673,
                                      -1.70627802690583, -1.70627802690583, -1.70627802690583, -1.7085201793722,
                                      -1.7085201793722])
                           )
            friction_factor = np.interp(relative_roughness, [0.001, 0.002], [ff2, ff1])
        elif 0.001 >= relative_roughness > 5.0e-4:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.61290322580645, 3.76411290322581,
                                      3.91532258064516, 4.13306451612903, 4.41129032258064, 4.76814516129032,
                                      5.01008064516129, 5.30040322580645, 5.65120967741935, 6.00201612903226,
                                      6.32862903225806, 6.6491935483871, 6.91532258064516, 7.3991935483871,
                                      7.99798387096774],
                                     [-1.28699551569507, -1.33183856502242, -1.38340807174888, -1.42600896860987,
                                      -1.46412556053812, -1.51569506726457, -1.57174887892377, -1.62556053811659,
                                      -1.65246636771301, -1.67488789237668, -1.69058295964126, -1.69955156950673,
                                      -1.70627802690583, -1.70627802690583, -1.70627802690583, -1.7085201793722,
                                      -1.7085201793722])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.43145161290322, 3.56451612903226, 3.70967741935484,
                                      3.84274193548387, 4.0241935483871, 4.33266129032258, 4.60483870967742,
                                      4.97983870967742, 5.44556451612903, 5.85685483870968, 6.30443548387097,
                                      6.96975806451613, 8.00403225806451],
                                     [-1.28923766816144, -1.32959641255605, -1.37443946188341, -1.41704035874439,
                                      -1.45515695067265, -1.50448430493274, -1.5762331838565, -1.63004484304933,
                                      -1.68834080717489, -1.73542600896861, -1.75784753363229, -1.77130044843049,
                                      -1.77578475336323, -1.78026905829597])
                           )
            friction_factor = np.interp(relative_roughness, [5.0e-4, 0.001], [ff2, ff1])
        elif 5.0e-4 >= relative_roughness > 2.0e-4:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.43145161290322, 3.56451612903226, 3.70967741935484,
                                      3.84274193548387, 4.0241935483871, 4.33266129032258, 4.60483870967742,
                                      4.97983870967742, 5.44556451612903, 5.85685483870968, 6.30443548387097,
                                      6.96975806451613, 8.00403225806451],
                                     [-1.28923766816144, -1.32959641255605, -1.37443946188341, -1.41704035874439,
                                      -1.45515695067265, -1.50448430493274, -1.5762331838565, -1.63004484304933,
                                      -1.68834080717489, -1.73542600896861, -1.75784753363229, -1.77130044843049,
                                      -1.77578475336323, -1.78026905829597])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.49798387096774, 3.69758064516129, 3.92741935483871,
                                      4.13306451612903, 4.32661290322581, 4.58064516129032, 4.76209677419355,
                                      5.02822580645161, 5.28225806451613, 5.56653225806452, 5.86895161290323,
                                      6.30443548387097, 6.77620967741935, 7.22983870967742, 7.61693548387097,
                                      7.99798387096774],
                                     [-1.2914798206278, -1.35650224215247, -1.42152466367713, -1.48878923766816,
                                      -1.54260089686099, -1.58744394618834, -1.64349775784753, -1.67937219730942,
                                      -1.72645739910314, -1.76457399103139, -1.79820627802691, -1.82286995515695,
                                      -1.84529147982063, -1.85650224215247, -1.8609865470852, -1.86322869955157,
                                      -1.86322869955157])
                           )
            friction_factor = np.interp(relative_roughness, [2.0e-4, 5.0e-4], [ff2, ff1])
        elif 2.0e-4 >= relative_roughness > 1.0e-4:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.49798387096774, 3.69758064516129, 3.92741935483871,
                                      4.13306451612903, 4.32661290322581, 4.58064516129032, 4.76209677419355,
                                      5.02822580645161, 5.28225806451613, 5.56653225806452, 5.86895161290323,
                                      6.30443548387097, 6.77620967741935, 7.22983870967742, 7.61693548387097,
                                      7.99798387096774],
                                     [-1.2914798206278, -1.35650224215247, -1.42152466367713, -1.48878923766816,
                                      -1.54260089686099, -1.58744394618834, -1.64349775784753, -1.67937219730942,
                                      -1.72645739910314, -1.76457399103139, -1.79820627802691, -1.82286995515695,
                                      -1.84529147982063, -1.85650224215247, -1.8609865470852, -1.86322869955157,
                                      -1.86322869955157])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.54637096774194, 3.75806451612903, 3.94556451612903,
                                      4.10887096774193, 4.30846774193548, 4.46572580645161, 4.61693548387097,
                                      4.81048387096774, 5.07661290322581, 5.34274193548387, 5.53629032258064,
                                      5.73588709677419, 6.0141129032258, 6.26814516129032, 6.54032258064516,
                                      6.82459677419355, 7.18145161290322, 7.53225806451613, 7.99798387096774],
                                     [-1.29372197309417, -1.37443946188341, -1.44170403587444, -1.49551569506727,
                                      -1.54035874439462, -1.58968609865471, -1.62556053811659, -1.65919282511211,
                                      -1.69955156950673, -1.74887892376682, -1.79372197309417, -1.82062780269058,
                                      -1.84529147982063, -1.87219730941704, -1.89013452914798, -1.90358744394619,
                                      -1.91255605381166, -1.9170403587444, -1.91928251121076, -1.92152466367713])
                           )
            friction_factor = np.interp(relative_roughness, [1.0e-4, 2.0e-4], [ff2, ff1])
        elif 1.0e-4 >= relative_roughness > 5.0e-5:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.54637096774194, 3.75806451612903, 3.94556451612903,
                                      4.10887096774193, 4.30846774193548, 4.46572580645161, 4.61693548387097,
                                      4.81048387096774, 5.07661290322581, 5.34274193548387, 5.53629032258064,
                                      5.73588709677419, 6.0141129032258, 6.26814516129032, 6.54032258064516,
                                      6.82459677419355, 7.18145161290322, 7.53225806451613, 7.99798387096774],
                                     [-1.29372197309417, -1.37443946188341, -1.44170403587444, -1.49551569506727,
                                      -1.54035874439462, -1.58968609865471, -1.62556053811659, -1.65919282511211,
                                      -1.69955156950673, -1.74887892376682, -1.79372197309417, -1.82062780269058,
                                      -1.84529147982063, -1.87219730941704, -1.89013452914798, -1.90358744394619,
                                      -1.91255605381166, -1.9170403587444, -1.91928251121076, -1.92152466367713])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.70362903225806, 4.04233870967742,
                                      4.30241935483871, 4.48991935483871, 4.76814516129032, 4.95564516129032, 5.125,
                                      5.33064516129032, 5.56048387096774, 5.95362903225806, 6.26209677419355,
                                      6.52822580645161, 6.91532258064516, 7.24193548387096, 7.65322580645161,
                                      7.99798387096774],
                                     [-1.29372197309417, -1.34304932735426, -1.4237668161435, -1.52466367713005,
                                      -1.59192825112108, -1.63677130044843, -1.69955156950673, -1.73318385650224,
                                      -1.76681614349776, -1.80269058295964, -1.84080717488789, -1.89237668161435,
                                      -1.9237668161435, -1.94394618834081, -1.96188340807175, -1.96860986547085,
                                      -1.97533632286996, -1.97757847533632])
                           )
            friction_factor = np.interp(relative_roughness, [5.0e-5, 1.0e-4], [ff2, ff1])
        elif 5.0e-5 >= relative_roughness > 1.0e-5:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.44959677419355, 3.70362903225806, 4.04233870967742,
                                      4.30241935483871, 4.48991935483871, 4.76814516129032, 4.95564516129032, 5.125,
                                      5.33064516129032, 5.56048387096774, 5.95362903225806, 6.26209677419355,
                                      6.52822580645161, 6.91532258064516, 7.24193548387096, 7.65322580645161,
                                      7.99798387096774],
                                     [-1.29372197309417, -1.34304932735426, -1.4237668161435, -1.52466367713005,
                                      -1.59192825112108, -1.63677130044843, -1.69955156950673, -1.73318385650224,
                                      -1.76681614349776, -1.80269058295964, -1.84080717488789, -1.89237668161435,
                                      -1.9237668161435, -1.94394618834081, -1.96188340807175, -1.96860986547085,
                                      -1.97533632286996, -1.97757847533632])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.52822580645161, 3.71572580645161, 3.98185483870968,
                                      4.22983870967742, 4.50806451612903, 4.70766129032258, 4.94354838709677,
                                      5.20362903225806, 5.57258064516129, 5.875, 6.11693548387097, 6.35282258064516,
                                      6.61895161290323, 6.98185483870968, 7.32056451612903, 7.71975806451613,
                                      7.99798387096774],
                                     [-1.29372197309417, -1.3677130044843, -1.42825112107623, -1.5067264573991,
                                      -1.57174887892377, -1.64125560538117, -1.68834080717489, -1.73766816143498,
                                      -1.78923766816144, -1.85874439461883, -1.91031390134529, -1.94394618834081,
                                      -1.97757847533632, -2.00896860986547, -2.04260089686099, -2.06502242152466,
                                      -2.08071748878924, -2.08744394618834])
                           )
            friction_factor = np.interp(relative_roughness, [1.0e-5, 5.0e-5], [ff2, ff1])
        elif 1.0e-5 >= relative_roughness > 5.0e-6:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.52822580645161, 3.71572580645161, 3.98185483870968,
                                      4.22983870967742, 4.50806451612903, 4.70766129032258, 4.94354838709677,
                                      5.20362903225806, 5.57258064516129, 5.875, 6.11693548387097, 6.35282258064516,
                                      6.61895161290323, 6.98185483870968, 7.32056451612903, 7.71975806451613,
                                      7.99798387096774],
                                     [-1.29372197309417, -1.3677130044843, -1.42825112107623, -1.5067264573991,
                                      -1.57174887892377, -1.64125560538117, -1.68834080717489, -1.73766816143498,
                                      -1.78923766816144, -1.85874439461883, -1.91031390134529, -1.94394618834081,
                                      -1.97757847533632, -2.00896860986547, -2.04260089686099, -2.06502242152466,
                                      -2.08071748878924, -2.08744394618834])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.55846774193548, 3.8366935483871, 4.12701612903226,
                                      4.3991935483871, 4.65927419354839, 4.9133064516129, 5.14314516129032, 5.6875,
                                      6.02620967741935, 6.4133064516129, 6.81854838709677, 7.17540322580645,
                                      7.59879032258065, 7.99798387096774],
                                     [-1.29372197309417, -1.38116591928251, -1.46412556053812, -1.54484304932735,
                                      -1.61434977578475, -1.67713004484305, -1.73318385650224, -1.78026905829597,
                                      -1.88340807174888, -1.94170403587444, -1.9932735426009, -2.04484304932735,
                                      -2.08071748878924, -2.10762331838565, -2.12556053811659])
                           )
            friction_factor = np.interp(relative_roughness, [5.0e-6, 1.0e-5], [ff2, ff1])
        elif 5.0e-6 >= relative_roughness > 1.0e-6:
            ff1 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.55846774193548, 3.8366935483871, 4.12701612903226,
                                      4.3991935483871, 4.65927419354839, 4.9133064516129, 5.14314516129032, 5.6875,
                                      6.02620967741935, 6.4133064516129, 6.81854838709677, 7.17540322580645,
                                      7.59879032258065, 7.99798387096774],
                                     [-1.29372197309417, -1.38116591928251, -1.46412556053812, -1.54484304932735,
                                      -1.61434977578475, -1.67713004484305, -1.73318385650224, -1.78026905829597,
                                      -1.88340807174888, -1.94170403587444, -1.9932735426009, -2.04484304932735,
                                      -2.08071748878924, -2.10762331838565, -2.12556053811659])
                           )
            ff2 = 10.0 ** (np.interp(np.log10(reynolds_number),
                                     [3.29838709677419, 3.52217741935484, 3.78225806451613, 4.13306451612903,
                                      4.44153225806452, 4.71370967741935, 4.94959677419355, 5.25201612903226,
                                      5.60282258064516, 5.8508064516129, 6.14112903225806, 6.45564516129032,
                                      6.72177419354839, 6.98185483870968, 7.25403225806451, 7.59879032258065,
                                      7.99798387096774],
                                     [-1.29372197309417, -1.3677130044843, -1.44618834080718, -1.54708520179372,
                                      -1.62331838565023, -1.68834080717489, -1.73766816143498, -1.80044843049328,
                                      -1.86771300448431, -1.91255605381166, -1.96188340807175, -2.01121076233184,
                                      -2.04932735426009, -2.08295964125561, -2.11659192825112, -2.15470852017937,
                                      -2.18834080717489])
                           )
            friction_factor = np.interp(relative_roughness, [1.0e-6, 5.0e-6], [ff2, ff1])
        else:
            friction_factor = 10.0 ** (np.interp(np.log10(reynolds_number),
                                                 [3.29838709677419, 3.42540322580645, 3.67338709677419,
                                                  3.92137096774193, 4.25403225806452, 4.66532258064516,
                                                  4.88911290322581, 5.40927419354839, 5.72983870967742,
                                                  6.03225806451613, 6.29838709677419, 6.59475806451613,
                                                  6.96370967741936, 7.27822580645161, 7.58064516129032,
                                                  7.99798387096774],
                                                 [-1.29372197309417, -1.33632286995516, -1.41479820627803,
                                                  -1.48654708520179, -1.57399103139013, -1.67713004484305,
                                                  -1.72645739910314, -1.83183856502242, -1.89013452914798,
                                                  -1.94394618834081, -1.9865470852018, -2.03363228699552,
                                                  -2.08520179372197, -2.13004484304933, -2.17040358744395,
                                                  -2.2219730941704]))

        head_loss = friction_factor * (pipe_length / pipe_diameter) * (
        (average_velocity ** 2.0) / (2.0 * gravity_coefficient))
        pressure_drop = fluid_density * gravity_coefficient * head_loss

        friction_factor_laminar = 64.0 / reynolds_number
        friction_factor_turbulent = 10.0 ** (np.interp(np.log10(reynolds_number),
                                                       [3.58266129032258, 3.89112903225806, 4.07258064516129,
                                                        4.28427419354839, 4.48991935483871, 4.75, 5.14314516129032,
                                                        5.43951612903226, 6.0625, 6.38306451612903, 6.70362903225806,
                                                        7.13911290322581, 7.48991935483871, 7.99798387096774],
                                                       [-1, -1.10313901345291, -1.16591928251121, -1.23542600896861,
                                                        -1.29820627802691, -1.38116591928251, -1.48878923766816,
                                                        -1.56726457399103, -1.7085201793722, -1.76457399103139,
                                                        -1.81838565022422, -1.86995515695067, -1.89910313901345,
                                                        -1.93946188340808]))

        if friction_factor < friction_factor_laminar:
            flow_regime = "Laminar Flow"
        elif friction_factor_laminar <= friction_factor < friction_factor_turbulent:
            flow_regime = "Transition Region"
        else:
            flow_regime = "Complete turbulence"

        return {
            'friction_factor [-]': friction_factor,
            'roughness [mm]': roughness,
            'head_loss [m]': head_loss,
            'pressure_drop [Pa]': pressure_drop,
            'flow_regime [-]': flow_regime,
            'friction_factor_laminar [-]': friction_factor_laminar,
            'friction_factor_turbulent [-]': friction_factor_turbulent,
        }

    except:
        if fail_silently or fail_silently is None:
            return {
                'friction_factor [-]': np.NaN,
                'roughness [mm]': np.NaN,
                'head_loss [m]': np.NaN,
                'pressure_drop [Pa]': np.NaN,
                'flow_regime [-]': None,
                'friction_factor_laminar [-]': np.NaN,
                'friction_factor_turbulent [-]': np.NaN,
            }
        else:
            raise