import asyncio
import contextlib
import logging
import threading
import time
from typing import List, Optional

import click
import kopf
from kopf._cogs.helpers import loaders

from fastflow.setup import get_appsettings

logger = logging.getLogger(__name__)


def kopf_thread(
    ready_flag: threading.Event,
    stop_flag: threading.Event,
):
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    with contextlib.closing(loop):
        loop.run_until_complete(
            kopf.operator(ready_flag=ready_flag, stop_flag=stop_flag, **get_appsettings().get_kopf_kwargs())
        )


def run_kopf_in_separate_thread():
    """see https://kopf.readthedocs.io/en/stable/embedding/#manual-execution"""
    ready_flag = threading.Event()
    stop_flag = threading.Event()
    thread = threading.Thread(
        target=kopf_thread,
        kwargs=dict(
            stop_flag=stop_flag,
            ready_flag=ready_flag,
        ),
    )
    thread.start()
    ready_flag.wait()
    while True:
        try:
            time.sleep(60)
        except KeyboardInterrupt:
            stop_flag.set()
            break

    thread.join()


@click.group(name="fastflow")
@click.version_option(prog_name="fastflow")
def main() -> None:
    pass


@main.command()
@click.option("-n", "--namespace", "namespace", multiple=False)
@click.option("--dev", "priority", type=int, is_flag=True, flag_value=666)
@click.option("-p", "--priority", type=int)
@click.option("-m", "--module", "modules", multiple=True)
@click.argument("paths", nargs=-1)
def run(
    priority: Optional[int],
    namespace: str,
    paths: List[str],
    modules: List[str],
) -> None:
    get_appsettings().namespace = namespace
    if priority is not None:
        get_appsettings().kopf_priority = priority

    # kopf.configure(verbose=True)  # log formatting

    loaders.preload(paths=[], modules=["fastflow.engine"])
    loaders.preload(
        paths=paths,
        modules=modules,
    )

    run_kopf_in_separate_thread()
