#
#
#
from __future__ import annotations

from typing import (
    TYPE_CHECKING,
    Protocol,
    TypeVar,
    Iterator,
    Iterable,
    Any,
    Self,
    Mapping,
    Sequence,
    overload,
)

from .flint_base.flint_base import (
    flint_elem,
    flint_scalar,
    flint_poly,
    flint_mpoly,
    flint_mpoly_context,
    flint_series,
    Ordering,
)

from flint.types.fmpz import fmpz

__all__ = [
    # Abstract base classes
    "flint_elem",
    "flint_scalar",
    "flint_poly",
    "flint_mpoly",
    "flint_mpoly_context",
    "flint_series",

    # Protocols
    "elem_p",
    "scalar_p",
    "poly_p",
    # "series_p",
    "mpoly_p",
    "mpoly_context_p",

    # Types
    "Ordering",
    "fmpz",
    "ifmpz",
]


_str = str


ifmpz = fmpz | int


class elem_p(Protocol):
    """FLINT element Protocol."""
    def str(self) -> _str: ...
    def repr(self) -> _str: ...


class scalar_p(elem_p, Protocol):
    """FLINT scalar Protocol."""
    def is_zero(self) -> bool: ...
    def __pos__(self) -> Self: ...
    def __neg__(self) -> Self: ...
    def __add__(self, other: Self | int, /) -> Self: ...
    def __radd__(self, other: int, /) -> Self: ...
    def __sub__(self, other: Self | int, /) -> Self: ...
    def __rsub__(self, other: int, /) -> Self: ...
    def __mul__(self, other: Self | int, /) -> Self: ...
    def __rmul__(self, other: int, /) -> Self: ...
    def __truediv__(self, other: Self | int, /) -> Self: ...
    def __rtruediv__(self, other: int, /) -> Self: ...
    def __pow__(self, other: int, /) -> Self: ...
    def __rpow__(self, other: int, /) -> Self: ...


_Tscalar = TypeVar("_Tscalar", bound=scalar_p)
_Tscalar_contra = TypeVar("_Tscalar_contra", bound=scalar_p, contravariant=True)


class poly_p(elem_p, Protocol[_Tscalar]):
    """FLINT univariate polynomial Protocol."""
    def str(
        self, ascending: bool = False, var: str = "x", *args: Any, **kwargs: Any
    ) -> str: ...
    def __iter__(self) -> Iterator[_Tscalar]: ...
    def __getitem__(self, index: int, /) -> _Tscalar: ...
    def __setitem__(self, index: int, value: _Tscalar | int, /) -> None: ...
    def __len__(self) -> int: ...
    def length(self) -> int: ...
    def degree(self) -> int: ...
    def coeffs(self) -> list[_Tscalar]: ...
    @overload
    def __call__(self, other: _Tscalar | ifmpz, /) -> _Tscalar: ...
    @overload
    def __call__(self, other: Self, /) -> Self: ... # pyright: ignore[reportOverlappingOverload]

    def __pos__(self) -> Self: ...
    def __neg__(self) -> Self: ...
    def __add__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __radd__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __sub__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __rsub__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __mul__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __rmul__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __truediv__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __rtruediv__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __floordiv__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __rfloordiv__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __mod__(self, other: _Tscalar | ifmpz | Self, /) -> Self: ...
    def __rmod__(self, other: _Tscalar | ifmpz, /) -> Self: ...
    def __divmod__(self, other: _Tscalar | ifmpz | Self, /) -> tuple[Self, Self]: ...
    def __rdivmod__(self, other: _Tscalar | ifmpz, /) -> tuple[Self, Self]: ...
    def __pow__(self, other: int, /) -> Self: ...

    def left_shift(self, other: int, /) -> Self: ...
    def right_shift(self, other: int, /) -> Self: ...
    def truncate(self, n: int, /) -> Self: ...

    def is_zero(self) -> bool: ...
    def is_one(self) -> bool: ...
    def is_constant(self) -> bool: ...
    def is_gen(self) -> bool: ...
    def roots(self) -> list[tuple[_Tscalar, int]]: ...
    # Should be list[arb]:
    def real_roots(self) -> list[Any]: ...
    # Should be list[acb]:
    def complex_roots(self) -> list[Any]: ...
    def derivative(self) -> Self: ...


class epoly_p(poly_p[_Tscalar], Protocol):
    """FLINT exact univariate polynomial Protocol."""
    def sqrt(self) -> Self: ...
    def gcd(self, other: Self | _Tscalar, /) -> Self: ...
    def factor(self) -> tuple[_Tscalar, list[tuple[Self, int]]]: ...
    def factor_squarefree(self) -> tuple[_Tscalar, list[tuple[Self, int]]]: ...
    def deflation(self) -> tuple[Self, int]: ...


class _series_p(elem_p, Protocol[_Tscalar]):
    """FLINT univariate power series."""

    def __iter__(self) -> Iterator[_Tscalar]: ...
    def coeffs(self) -> list[_Tscalar]: ...


class mpoly_p(elem_p, Protocol[_Tscalar]):
   def __init__(
       self,
       val: Self
       | _Tscalar
       | int
       | Mapping[tuple[int, ...], _Tscalar | int]
       | str = 0,
       ctx: Any | None = None,
   ) -> None: ...
   def str(self) -> _str: ...
   def repr(self) -> _str: ...
   def context(self) -> mpoly_context_p[Self, _Tscalar]: ...
   def degrees(self) -> tuple[int, ...]: ...
   def total_degree(self) -> int: ...
   def leading_coefficient(self) -> _Tscalar: ...
   def to_dict(self) -> dict[tuple[int, ...], _Tscalar]: ...
   def is_one(self) -> bool: ...
   def is_zero(self) -> bool: ...
   def is_constant(self) -> bool: ...
   def __len__(self) -> int: ...
   def __getitem__(self, index: tuple[int, ...]) -> _Tscalar: ...
   def __setitem__(
       self, index: tuple[int, ...], coeff: _Tscalar | int
   ) -> None: ...
   def __iter__(self) -> Iterable[tuple[int, ...]]: ...
   def __contains__(self, index: tuple[int, ...]) -> bool: ...
   def coefficient(self, i: int) -> _Tscalar: ...
   def monomial(self, i: int) -> tuple[int, ...]: ...
   def terms(self) -> Iterable[tuple[tuple[int, ...], _Tscalar]]: ...
   def monoms(self) -> list[tuple[int, ...]]: ...
   def coeffs(self) -> list[_Tscalar]: ...
   def __pos__(self) -> Self: ...
   def __neg__(self) -> Self: ...
   def __add__(self, other: Self | _Tscalar | int) -> Self: ...
   def __radd__(self, other: _Tscalar | int) -> Self: ...
   def __sub__(self, other: Self | _Tscalar | int) -> Self: ...
   def __rsub__(self, other: _Tscalar | int) -> Self: ...
   def __mul__(self, other: Self | _Tscalar | int) -> Self: ...
   def __rmul__(self, other: _Tscalar | int) -> Self: ...
   def __truediv__(self, other: Self | _Tscalar | int) -> Self: ...
   def __rtruediv__(self, other: _Tscalar | int) -> Self: ...
   def __floordiv__(self, other: Self | _Tscalar | int) -> Self: ...
   def __rfloordiv__(self, other: _Tscalar | int) -> Self: ...
   def __mod__(self, other: Self | _Tscalar | int) -> Self: ...
   def __rmod__(self, other: _Tscalar | int) -> Self: ...
   def __divmod__(
       self, other: Self | _Tscalar | int
   ) -> tuple[Self, Self]: ...
   def __rdivmod__(self, other: _Tscalar | int) -> tuple[Self, Self]: ...
   def __pow__(self, other: _Tscalar | int) -> Self: ...
   def __rpow__(self, other: _Tscalar | int) -> Self: ...
   def iadd(self, other: _Tscalar | int) -> None: ...
   def isub(self, other: _Tscalar | int) -> None: ...
   def imul(self, other: _Tscalar | int) -> None: ...
   def gcd(self, other: Self) -> Self: ...
   def term_content(self) -> Self: ...
   def factor(self) -> tuple[_Tscalar, Sequence[tuple[Self, int]]]: ...
   def factor_squarefree(self) -> tuple[_Tscalar, Sequence[tuple[Self, int]]]: ...
   def sqrt(self) -> Self: ...
   def resultant(self, other: Self, var: _str | int) -> Self: ...
   def discriminant(self, var: _str | int) -> Self: ...
   def deflation_index(self) -> tuple[list[int], list[int]]: ...
   def deflation(self) -> tuple[Self, list[int]]: ...
   def deflation_monom(self) -> tuple[Self, list[int], Self]: ...
   def inflate(self, N: list[int]) -> Self: ...
   def deflate(self, N: list[int]) -> Self: ...
   def subs(self, mapping: Mapping[_str | int, _Tscalar | int]) -> Self: ...
   def compose(self, *args: Self, ctx: Any | None = None) -> Self: ...
   def __call__(self, *args: _Tscalar) -> _Tscalar: ...
   def derivative(self, var: _str | int) -> Self: ...
   def unused_gens(self) -> tuple[_str, ...]: ...
   def project_to_context(
       self, other_ctx: Any, mapping: dict[_str | int, _str | int] | None = None
   ) -> Self: ...


_Tmpoly_p = TypeVar("_Tmpoly_p", bound=mpoly_p, covariant=True)


class mpoly_context_p(elem_p, Protocol[_Tmpoly_p, _Tscalar_contra]):
    def nvars(self) -> int: ...
    def ordering(self) -> Ordering: ...
    def gen(self, i: int, /) -> _Tmpoly_p: ...
    def from_dict(self, d: Mapping[tuple[int, ...], _Tscalar_contra | int], /) -> _Tmpoly_p: ...
    def constant(self, z: _Tscalar_contra | int, /) -> _Tmpoly_p: ...
    def name(self, i: int, /) -> str: ...
    def names(self) -> tuple[str]: ...
    def gens(self) -> tuple[_Tmpoly_p, ...]: ...
    def variable_to_index(self, var: str, /) -> int: ...
    def term(
        self, coeff: _Tscalar_contra | None = None, exp_vec: Iterable[int] | None = None
    ) -> _Tmpoly_p: ...
    def drop_gens(self, gens: Iterable[str | int], /) -> Self: ...
    def append_gens(self, gens: Iterable[str | int], /) -> Self: ...
    def infer_generator_mapping(
        self, ctx: Self, /
    ) -> dict[int, int]: ...
    @classmethod
    def from_context(
        cls,
        ctx: flint_mpoly_context, # make a protocol for this?
        names: str | Iterable[str | tuple[str, int]] | tuple[str, int] | None = None,
        ordering: Ordering | str = Ordering.lex,
    ) -> Self: ...


if TYPE_CHECKING:

    import flint as _f

    _x1: scalar_p = _f.fmpz(1)
    _x2: scalar_p = _f.fmpq(1, 2)
    _x3: scalar_p = _f.nmod(1, 2)
    _x4: scalar_p = _f.fmpz_mod(1, _f.fmpz_mod_ctx(2))
    _x5: scalar_p = _f.fq_default(1, _f.fq_default_ctx(2))
    # XXX: Add arf, acf, arb, acb, ...

    _Tscalar0 = TypeVar("_Tscalar0", bound=scalar_p)

    def _scalar(x: _Tscalar0, y: _Tscalar0) -> _Tscalar0:
        z = x + y + 1
        z = z / z + x**2
        z = 1 / z + 1**z
        return z

    _y1: poly_p[_f.fmpz] = _f.fmpz_poly([1, 2])
    _y2: poly_p[_f.fmpq] = _f.fmpq_poly([1, 2])
    _y3: poly_p[_f.nmod] = _f.nmod_poly(1, 2)
    _y4: poly_p[_f.fmpz_mod] = _f.fmpz_mod_poly(1, _f.fmpz_mod_poly_ctx(2))
    _y5: poly_p[_f.fq_default] = _f.fq_default_poly(1, _f.fq_default_poly_ctx(2))
    # XXX: Add arb_poly, acb_poly, ...

    def _poly(x: epoly_p[_Tscalar0], y: epoly_p[_Tscalar0]) -> epoly_p[_Tscalar0]:
        z = x + y + 1
        z = z / z + x**2
        z = 1 / z + z.coeffs()[0]
        return z

    _z1: epoly_p[_f.fmpz] = _f.fmpz_poly([1, 2])
    _z2: epoly_p[_f.fmpq] = _f.fmpq_poly([1, 2])
    _z3: epoly_p[_f.nmod] = _f.nmod_poly(1, 2)
    _z4: epoly_p[_f.fmpz_mod] = _f.fmpz_mod_poly(1, _f.fmpz_mod_poly_ctx(2))
    _z5: epoly_p[_f.fq_default] = _f.fq_default_poly(1, _f.fq_default_poly_ctx(2))

    def _epoly(x: epoly_p[_Tscalar0], y: epoly_p[_Tscalar0]) -> epoly_p[_Tscalar0]:
        return x.gcd(y).factor()[1][0][0]

    _ctx1: mpoly_context_p[_f.fmpz_mpoly, _f.fmpz] = _f.fmpz_mpoly_ctx.get(['x', 'y'])
    _ctx2: mpoly_context_p[_f.fmpq_mpoly, _f.fmpq] = _f.fmpq_mpoly_ctx.get(['x', 'y'])
    _ctx3: mpoly_context_p[_f.nmod_mpoly, _f.nmod] = _f.nmod_mpoly_ctx.get(['x', 'y'], modulus=7)
    _ctx4: mpoly_context_p[_f.fmpz_mod_mpoly, _f.fmpz_mod] = _f.fmpz_mod_mpoly_ctx.get(['x', 'y'], modulus=7)

    _p1: mpoly_p[_f.fmpz] = _ctx1.gens()[0]
    _p2: mpoly_p[_f.fmpq] = _ctx2.gens()[0]
    _p3: mpoly_p[_f.nmod] = _ctx3.gens()[0]
    _p4: mpoly_p[_f.fmpz_mod] = _ctx4.gens()[0]

    _Tmpoly0 = TypeVar("_Tmpoly0", bound=mpoly_p)

    def _mpoly(x: mpoly_p[_Tscalar], y: mpoly_p[_Tscalar], ctx: mpoly_context_p[_Tmpoly0, _Tscalar]) -> mpoly_p[_Tscalar]:
        z = x.gcd(y).factor()[1][0][0] + ctx.gens()[0]
        t = z.compose(z, ctx=ctx) + z.context().gens()[0]
        return t
