import builtins
import collections
import functools
import inspect
import types
import typing

import forge._immutable as immutable
from forge._counter import CreationOrderMeta
from forge._marker import _void, empty, void

## Parameter
POSITIONAL_ONLY = inspect.Parameter.POSITIONAL_ONLY
POSITIONAL_OR_KEYWORD = inspect.Parameter.POSITIONAL_OR_KEYWORD
VAR_POSITIONAL = inspect.Parameter.VAR_POSITIONAL
KEYWORD_ONLY = inspect.Parameter.KEYWORD_ONLY
VAR_KEYWORD = inspect.Parameter.VAR_KEYWORD

_PARAMETER_KIND_STRINGS = {
    inspect.Parameter.POSITIONAL_ONLY: 'positional only',
    inspect.Parameter.POSITIONAL_OR_KEYWORD: 'positional or keyword',
    inspect.Parameter.VAR_POSITIONAL: 'variable positional',
    inspect.Parameter.KEYWORD_ONLY: 'keyword only',
    inspect.Parameter.VAR_KEYWORD: 'variable keyword',
}
_get_pk_string = _PARAMETER_KIND_STRINGS.__getitem__


class Factory(immutable.Immutable):
    """
    A Factory object is a wrapper around a callable that gets called to generate
    a default value everytime a function is invoked.

    :param factory: a callable which is invoked without argument to generate
        a default value.
    """
    __slots__ = ('factory',)

    def __init__(self, factory: typing.Callable[[], typing.Any]) -> None:
        # pylint: disable=C0102, blacklisted-name
        super().__init__(factory=factory)

    def __repr__(self) -> str:
        return '<{} {}>'.format(type(self).__name__, self.factory.__qualname__)

    def __call__(self) -> typing.Any:
        return self.factory()


# Common type hints for FParameter
_TYPE_FP_CTX_CALLABLE = typing.Callable[
    [typing.Any, str, typing.Any],
    typing.Any,
]
_TYPE_FP_KIND = inspect._ParameterKind # pylint: disable=C0103, invalid-name
_TYPE_FP_BOUND = bool # pylint: disable=C0103, invalid-name
_TYPE_FP_CONTEXTUAL = bool # pylint: disable=C0103, invalid-name
_TYPE_FP_NAME = typing.Optional[str]
_TYPE_FP_DEFAULT = typing.Any
_TYPE_FP_FACTORY = typing.Callable[[], typing.Any]
_TYPE_FP_TYPE = typing.Any
_TYPE_FP_CONVERTER = typing.Optional[
    typing.Union[
        _TYPE_FP_CTX_CALLABLE,
        typing.Iterable[_TYPE_FP_CTX_CALLABLE]
    ]
]
_TYPE_FP_VALIDATOR = typing.Optional[
    typing.Union[
        _TYPE_FP_CTX_CALLABLE,
        typing.Iterable[_TYPE_FP_CTX_CALLABLE]
    ]
]
_TYPE_FP_METADATA = typing.Mapping


class FParameter(immutable.Immutable, metaclass=CreationOrderMeta):
    """
    An immutable representation of a signature parameter that encompasses its
    public name, its interface name, transformations to be applied, and
    associated meta-data that defines its behavior in a signature.

    .. note::

        This class doesn't need to be invoked directly. Use one of the
        constructor methods instead:

        - :func:`~forge.pos` for :term:`positional-only` \
        :class:`~forge.FParameter`
        - :func:`~forge.pok` *or* :func:`~forge.arg` for \
        :term:`positional-or-keyword` :class:`~forge.FParameter`
        - :func:`~forge.vpo` for :term:`var-positional` \
        :class:`~forge.FParameter`
        - :func:`~forge.kwo` *or* :func:`~forge.kwarg` for \
        :term:`keyword-only` :class:`~forge.FParameter`
        - :func:`~forge.vkw` for :term:`var-keyword` :class:`~forge.FParameter`

    :param kind: the :term:`parameter kind`, which detemrines the position
        of the parameter in a callable signature.
    :param name: the public name of the parameter.
        For example, in :code:`f(x)` -> :code:`g(y)`, ``name`` is ``x``.
    :param interface_name: the name of mapped-to the parameter.
        For example, in :code:`f(x)` -> :code:`g(y)`,
        ``interface_name`` is ``y``.
    :param default: the default value for the parameter.
        Cannot be supplied alongside a ``factory`` argument.
        For example, to achieve :code:`f(x=3)`, specify :code`default=3`.
    :param factory: a function that generates a default for the parameter
        Cannot be supplied alongside a ``default`` argument.
        For example, to achieve :code:`f(x=<Factory now>)`,
        specify :code:`factory=default.now` (notice: without parentheses).
    :param type: the type annotation of the parameter.
        For example, to achieve :code:`f(x: int)`, ``type`` is ``int``.
    :param converter: a callable or iterable of callables that receive a
        ``ctx`` argument, a ``name`` argument and a ``value`` argument
        for transforming inputs.
    :param validator: a callable that receives a ``ctx`` argument,
        a ``name`` argument and a ``value`` argument for validating inputs.
    :param bound: whether the parameter is visible in the signature
        (requires ``default`` or ``factory`` if True)
    :param contextual: whether the parameter will be passed to
        ``converter`` and ``validator`` callables as the context
        (only the first parameter in a :class:`~forge.FSignature` can be
        contextual)
    :param metadata: optional, extra meta-data that describes the parameter

    :cvar POSITIONAL_ONLY: the :term:`positional-only`
        :term:`parameter kind` constant
        :attr:`inspect.Parameter.POSITIONAL_ONLY`
    :cvar POSITIONAL_OR_KEYWORD: the :term:`positional-or-keyword`
        :term:`parameter kind` constant
        :attr:`inspect.Parameter.POSITIONAL_OR_KEYWORD`
    :cvar VAR_POSITIONAL: the :term:`var-positional` constant
        :term:`parameter kind` constant
        :attr:`inspect.Parameter.VAR_POSITIONAL`
    :cvar KEYWORD_ONLY: the :term:`keyword-only` constant
        :term:`parameter kind` constant
        :attr:`inspect.Parameter.KEYWORD_ONLY`
    :cvar VAR_KEYWORD: the :term:`var-keyword` constant
        :term:`parameter kind` constant
        :attr:`inspect.Parameter.VAR_KEYWORD`
    """

    __slots__ = (
        '_creation_order',
        'kind',
        'name',
        'interface_name',
        'default',
        'type',
        'converter',
        'validator',
        'bound',
        'contextual',
        'metadata',
    )

    empty = empty
    POSITIONAL_ONLY = POSITIONAL_ONLY
    POSITIONAL_OR_KEYWORD = POSITIONAL_OR_KEYWORD
    VAR_POSITIONAL = VAR_POSITIONAL
    KEYWORD_ONLY = KEYWORD_ONLY
    VAR_KEYWORD = VAR_KEYWORD

    def __init__(
            self,
            kind: _TYPE_FP_KIND,
            name: _TYPE_FP_NAME = None,
            interface_name: _TYPE_FP_NAME = None,
            default: _TYPE_FP_DEFAULT = empty,
            factory: _TYPE_FP_FACTORY = empty,
            type: _TYPE_FP_TYPE = empty,
            converter: _TYPE_FP_CONVERTER = None,
            validator: _TYPE_FP_VALIDATOR = None,
            bound: _TYPE_FP_BOUND = False,
            contextual: _TYPE_FP_CONTEXTUAL = False,
            metadata: typing.Optional[_TYPE_FP_METADATA] = None
        ) -> None:
        # pylint: disable=W0622, redefined-builtin
        # pylint: disable=R0913, too-many-arguments
        if name is not None and not isinstance(name, str):
            # Do enough validation of name to enable the Sequence functionality
            # of FSignature
            raise TypeError(
                'name must be a str, not a {}'.format(name),
            )

        if interface_name is not None and not isinstance(interface_name, str):
            raise TypeError(
                'interface_name must be a str, not a {}'.format(interface_name)
            )

        if factory is not empty:
            if default is not empty:
                raise TypeError(
                    'expected either "default" or "factory", received both'
                )
            default = Factory(factory)

        if bound and default is empty:
            raise TypeError('bound arguments must have a default value')

        super().__init__(
            kind=kind,
            name=name or interface_name,
            interface_name=interface_name or name,
            default=default,
            type=type,
            converter=converter,
            validator=validator,
            contextual=contextual,
            bound=bound,
            metadata=types.MappingProxyType(metadata or {}),
        )

    def __str__(self) -> str:
        """
        Generates a string representation of the :class:`~forge.FParameter`
        """
        if self.kind == self.VAR_POSITIONAL:
            prefix = '*'
        elif self.kind == self.VAR_KEYWORD:
            prefix = '**'
        else:
            prefix = ''

        mapped = \
            '{prefix}{name}'.format(
                prefix=prefix,
                name=self.name or '<missing>',
            ) if self.name == self.interface_name \
            else '{prefix}{name}->{prefix}{interface_name}'.format(
                prefix=prefix,
                name=self.name or '<missing>',
                interface_name=self.interface_name or '<missing>',
            )

        annotated = mapped \
            if self.type is empty \
            else '{mapped}:{annotation}'.format(
                mapped=mapped,
                annotation=self.type.__name__ \
                    if inspect.isclass(self.type) \
                    else str(self.type),
            )

        return annotated \
            if self.default is empty \
            else '{annotated}={default}'.format(
                annotated=annotated,
                default=self.default,
            )

    def __repr__(self) -> str:
        return '<{} "{}">'.format(type(self).__name__, str(self))

    def apply_default(self, value: typing.Any) -> typing.Any:
        """
        Return the argument value (if not :class:`~forge.empty`), or the value
        from :paramref:`~forge.FParmeter.default` (if not an instance of
        :class:`~forge.Factory`), or the value obtained by calling
        :paramref:`~forge.FParameter.default` (if an instance of
        :class:`~forge.Factory`).

        :param value: the argument value for this parameter
        :returns: the input value or a default value
        """
        if value is not empty:
            return value() if isinstance(value, Factory) else value
        return self.default

    def apply_conversion(
            self,
            ctx: typing.Any,
            value: typing.Any,
        ) -> typing.Any:
        """
        Apply a transform or series of transforms against the argument value
        with the callables from :paramref:`~forge.FParameter.converter`.

        :param ctx: the context of this parameter as provided by the
            :class:`~forge.FSignature` (typically self or ctx).
        :param value: the argument value for this parameter
        :returns: the converted value
        """
        # pylint: disable=W0621, redefined-outer-name
        if self.converter is None:
            return value
        elif isinstance(self.converter, typing.Iterable):
            return functools.reduce(
                lambda val, func: func(ctx, self.name, val),
                [value, *self.converter],
            )
        return self.converter(ctx, self.name, value)

    def apply_validation(
            self,
            ctx: typing.Any,
            value: typing.Any,
        ) -> typing.Any:
        """
        Apply a validation or series of validations against the argument value
        with the callables from :paramref:`~forge.FParameter.validator`.

        :param ctx: the context of this parameter as provided by the
            :class:`~forge.FSignature` (typically self or ctx).
        :param value: the value the user has supplied or a default value
        :returns: the (unchanged) validated value
        """
        # pylint: disable=W0621, redefined-outer-name
        if isinstance(self.validator, typing.Iterable):
            for validate in self.validator:
                validate(ctx, self.name, value)
        elif self.validator is not None:
            self.validator(ctx, self.name, value)
        return value

    def __call__(
            self,
            ctx: typing.Any,
            value: typing.Any
        ) -> typing.Any:
        """
        Can be called after defaults have been applied (if not a ``bound``
        :class:`~forge.FParameter`) or without a value (i.e.
        :class:`inspect.Parameter.emtpy`) in the case of a ``bound``
        :class:`~forge.FParameter`.

        Process:

        - conditionally apply the :class:`~forge.Factory`,
        - convert the resulting value with the \
        :paramref:`~forge.FParameter.converter`, and then
        - validate the resulting value with the \
        :forge:`~forge.FParameter.validator`.

        :param ctx: the context of this parameter as provided by the
            :class:`~forge.FSignature` (typically self or ctx).
        :param value: the user-supplied (or default) value
        """
        # pylint: disable=W0621, redefined-outer-name
        defaulted = self.apply_default(value)
        converted = self.apply_conversion(ctx, defaulted)
        return self.apply_validation(ctx, converted)

    @property
    def native(self) -> inspect.Parameter:
        """
        A native representation of this :class:`~forge.FParameter` as an
        :class:`inspect.Parameter`, fit for an instance of
        :class:`inspect.Signature`
        """
        if not self.name:
            raise TypeError('Cannot generate an unnamed parameter')
        return inspect.Parameter(
            name=self.name,
            kind=self.kind,
            default=empty.ccoerce_native(self.default),
            annotation=empty.ccoerce_native(self.type),
        )

    def replace(
            self,
            *,
            kind=_void,
            name=_void,
            interface_name=_void,
            default=_void,
            factory=_void,
            type=_void,
            converter=_void,
            validator=_void,
            bound=_void,
            contextual=_void,
            metadata=_void
        ):
        """
        An evolution method that generates a new :class:`~forge.FParameter`
        derived from this instance and the provided updates.

        :param kind: see :paramref:`~forge.FParameter.kind`
        :param name: see :paramref:`~forge.FParameter.name`
        :param interface_name: see :paramref:`~forge.FParameter.interface_name`
        :param default: see :paramref:`~forge.FParameter.default`
        :param factory: see :paramref:`~forge.FParameter.factory`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param bound: see :paramref:`~forge.FParameter.bound`
        :param contextual: see :paramref:`~forge.FParameter.contextual`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        :returns: an instance of `~forge.FParameter`
        """
        # pylint: disable=E1120, no-value-for-parameter
        # pylint: disable=W0622, redefined-builtin
        # pylint: disable=R0913, too-many-arguments
        if factory is not _void and default is _void:
            default = empty

        return immutable.replace(self, **{
            k: v for k, v in {
                'kind': kind,
                'name': name,
                'interface_name': interface_name,
                'default': default,
                'factory': factory,
                'type': type,
                'converter': converter,
                'validator': validator,
                'bound': bound,
                'contextual': contextual,
                'metadata': metadata,
            }.items() if v is not _void
        })

    @classmethod
    def from_native(cls, native: inspect.Parameter) -> 'FParameter':
        """
        A factory method for creating :class:`~forge.FParameter` instances from
        :class:`inspect.Parameter` instances.

        Parameter descriptions are a subset of those defined on
        :class:`~forge.FParameter`

        :param native: an instance of :class:`inspect.Parameter`, used as a
            template for creating a new :class:`~forge.FParameter`
            :returns: a new instance of :class:`~forge.FParameter`, using
            :paramref:`~forge.FParameter.from_native.native` as a template
        """
        return cls(  # type: ignore
            kind=native.kind,
            name=native.name,
            interface_name=native.name,
            default=cls.empty.ccoerce_synthetic(native.default),
            type=cls.empty.ccoerce_synthetic(native.annotation),
        )

    @classmethod
    def create_positional_only(
            cls,
            name=None,
            interface_name=None,
            *,
            default=empty,
            factory=empty,
            type=empty,
            converter=None,
            validator=None,
            bound=False,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`positional-only`
        :class:`~forge.FParameter` instances.

        :param name: see :paramref:`~forge.FParameter.name`
        :param interface_name: see :paramref:`~forge.FParameter.interface_name`
        :param default: see :paramref:`~forge.FParameter.default`
        :param factory: see :paramref:`~forge.FParameter.factory`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param bound: see :paramref:`~forge.FParameter.bound`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.POSITIONAL_ONLY,
            name=name,
            interface_name=interface_name,
            default=default,
            factory=factory,
            type=type,
            converter=converter,
            validator=validator,
            bound=bound,
            metadata=metadata,
        )

    @classmethod
    def create_positional_or_keyword(
            cls,
            name=None,
            interface_name=None,
            *,
            default=empty,
            factory=empty,
            type=empty,
            converter=None,
            validator=None,
            bound=False,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`positional-or-keyword`
        :class:`~forge.FParameter` instances.

        :param name: see :paramref:`~forge.FParameter.name`
        :param interface_name: see :paramref:`~forge.FParameter.interface_name`
        :param default: see :paramref:`~forge.FParameter.default`
        :param factory: see :paramref:`~forge.FParameter.factory`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param bound: see :paramref:`~forge.FParameter.bound`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.POSITIONAL_OR_KEYWORD,
            name=name,
            interface_name=interface_name,
            default=default,
            factory=factory,
            type=type,
            converter=converter,
            validator=validator,
            bound=bound,
            metadata=metadata,
        )

    @classmethod
    def create_contextual(
            cls,
            name=None,
            interface_name=None,
            *,
            type=empty,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`positional-or-keyword`
        :class:`~forge.FParameter` instances that are ``contextual`` (this value
        is passed to other :class:`~forge.FParameter`s ``converter`` and
        ``validator`` functions.)

        :param name: see :paramref:`~forge.FParameter.name`
        :param interface_name: see :paramref:`~forge.FParameter.interface_name`
        :param type: see :paramref:`~forge.FParameter.type`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.POSITIONAL_OR_KEYWORD,
            name=name,
            interface_name=interface_name,
            type=type,
            contextual=True,
            metadata=metadata,
        )

    @classmethod
    def create_var_positional(
            cls,
            name,
            *,
            type=empty,
            converter=None,
            validator=None,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`var-positional`
        :class:`~forge.FParameter` instances.

        :param name: see :paramref:`~forge.FParameter.name`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.VAR_POSITIONAL,
            name=name,
            type=type,
            converter=converter,
            validator=validator,
            metadata=metadata,
        )

    @classmethod
    def create_keyword_only(
            cls,
            name=None,
            interface_name=None,
            *,
            default=empty,
            factory=empty,
            type=empty,
            converter=None,
            validator=None,
            bound=False,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`keyword-only`
        :class:`~forge.FParameter` instances.

        :param name: see :paramref:`~forge.FParameter.name`
        :param interface_name: see :paramref:`~forge.FParameter.interface_name`
        :param default: see :paramref:`~forge.FParameter.default`
        :param factory: see :paramref:`~forge.FParameter.factory`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param bound: see :paramref:`~forge.FParameter.bound`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.KEYWORD_ONLY,
            name=name,
            interface_name=interface_name,
            default=default,
            factory=factory,
            type=type,
            converter=converter,
            validator=validator,
            bound=bound,
            metadata=metadata,
        )

    @classmethod
    def create_var_keyword(
            cls,
            name,
            *,
            type=empty,
            converter=None,
            validator=None,
            metadata=None
        ) -> 'FParameter':
        """
        A factory method for creating :term:`var-keyword`
        :class:`~forge.FParameter` instances.

        :param name: see :paramref:`~forge.FParameter.name`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        """
        # pylint: disable=W0622, redefined-builtin
        return cls(  # type: ignore
            kind=cls.VAR_KEYWORD,
            name=name,
            type=type,
            converter=converter,
            validator=validator,
            metadata=metadata,
        )

# Convenience
pos = FParameter.create_positional_only
arg = pok = FParameter.create_positional_or_keyword
kwarg = kwo = FParameter.create_keyword_only
ctx = FParameter.create_contextual
vpo = FParameter.create_var_positional
vkw = FParameter.create_var_keyword
self_ = ctx('self')
cls_ = ctx('cls')


class VarPositional(collections.abc.Iterable):
    """
    A psuedo-sequence that unpacks as a :term:`var-positional`
    :class:`~forge.FParameter`.

    Can also be called with arguments to generate another instance.

    Typical usage::

        >>> import forge
        >>> fsig = forge.FSignature(*forge.args)
        >>> print(fsig)
        <FSignature (*args)>

        >>> import forge
        >>> fsig = forge.FSignature(*forge.args(name='vars'))
        >>> print(fsig)
        <FSignature (*vars)>

    While ``name`` can be supplied (by default it's ``args``),
    ``interface_name`` is unavailable.
    This is because when :class:`~forge.FSignature` maps parameters, the mapping
    between :term:`var-positional` parameters is 1:1, so the interface name for
    :term:`var-positional` is auto-discovered.

    Implements :class:`collections.abc.Iterable`, with provided: ``__iter__``.
    Inherits method: ``__next__``.

    :param name: see :paramref:`~forge.FParameter.name`
    :param type: see :paramref:`~forge.FParameter.type`
    :param converter: see :paramref:`~forge.FParameter.converter`
    :param validator: see :paramref:`~forge.FParameter.validator`
    :param metadata: see :paramref:`~forge.FParameter.metadata`
    """
    _default_name = 'args'

    def __init__(
            self,
            name: _TYPE_FP_NAME = None,
            *,
            type: _TYPE_FP_TYPE = empty,
            converter: _TYPE_FP_CONVERTER = None,
            validator: _TYPE_FP_VALIDATOR = None,
            metadata: typing.Optional[_TYPE_FP_METADATA] = None
        ) -> None:
        # pylint: disable=W0622, redefined-builtin
        self.name = name or self._default_name
        self.type = type
        self.converter = converter
        self.validator = validator
        self.metadata = metadata

    @property
    def fparameter(self) -> FParameter:
        """
        :returns: a representation of this
            :class:`~forge._parameter.VarPositional` as a
            :class:`~forge.FParameter` of :term:`parameter kind`
            :term:`var-positional`, with attributes ``name``, ``converter``,
            ``validator`` and ``metadata`` from the instance.
        """
        # pylint: disable=E1101, no-member
        return FParameter.create_var_positional(
            name=self.name,
            type=self.type,
            converter=self.converter,
            validator=self.validator,
            metadata=self.metadata,
        )

    def __iter__(self) -> typing.Iterator:
        """
        Concrete method for :class:`collections.abc.Iterable`

        :returns: an iterable consisting of one item: the representation of this
            :class:`~forge._parameter.VarPositional` as a
            :class:`~forge.FParameter` via
            :attr:`~forge._parameter.VarPositional.fparameter`.
        """
        return iter((self.fparameter,))

    def __call__(
            self,
            name: _TYPE_FP_NAME = None,
            *,
            type: _TYPE_FP_TYPE = empty,
            converter: _TYPE_FP_CONVERTER = None,
            validator: _TYPE_FP_VALIDATOR = None,
            metadata: typing.Optional[_TYPE_FP_METADATA] = None
        ) -> 'VarPositional':
        """
        A factory method which creates a new
        :class:`~forge._parameter.VarPositional` instance.
        Convenient for use like::

            *args(converter=lambda ctx, name, value: value[::-1])

        :param name: see :paramref:`~forge.FParameter.name`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        :returns: a new instance of :class:`~forge._parameter.VarPositional`
        """
        # pylint: disable=W0622, redefined-builtin
        return builtins.type(self)(
            name=name,
            type=type,
            converter=converter,
            validator=validator,
            metadata=metadata,
        )

# Convenience
args = VarPositional()


class VarKeyword(collections.abc.Mapping):
    """
    A psuedo-collection that unpacks as a :term:`var-keyword`
    :class:`~forge.FParameter`.

    Can also be called with arguments to generate another instance.

    Typical usage::

        >>> import forge
        >>> fsig = forge.FSignature(**forge.kwargs)
        >>> print(fsig)
        <FSignature (**kwargs)>

        >>> import forge
        >>> fsig = forge.FSignature(**forge.kwargs(name='items'))
        >>> print(fsig)
        <FSignature (**items)>

    While ``name`` can be supplied (by default it's ``kwargs``),
    ``interface_name`` is unavailable.
    This is because when :class:`~forge.FSignature` maps parameters, the mapping
    between :term:`var-keyword` parameters is 1:1, so the interface name for
    :term:`var-keyword` is auto-discovered.

    Implements :class:`collections.abc.Mapping`, with provided: ``__getitem__``,
    ``__iter__`` and ``__len__``. Inherits methods: ``__contains__``, ``keys``,
    ``items``, ``values``, ``get``, ``__eq__`` and ``__ne__``.

    :param name: see :paramref:`~forge.FParameter.name`
    :param type: see :paramref:`~forge.FParameter.type`
    :param converter: see :paramref:`~forge.FParameter.converter`
    :param validator: see :paramref:`~forge.FParameter.validator`
    :param metadata: see :paramref:`~forge.FParameter.metadata`
    """
    _default_name = 'kwargs'

    def __init__(
            self,
            name: _TYPE_FP_NAME = None,
            *,
            type: _TYPE_FP_TYPE = empty,
            converter: _TYPE_FP_CONVERTER = None,
            validator: _TYPE_FP_VALIDATOR = None,
            metadata: typing.Optional[_TYPE_FP_METADATA] = None
        ) -> None:
        # pylint: disable=W0622, redefined-builtin
        self.name = name or self._default_name
        self.type = type
        self.converter = converter
        self.validator = validator
        self.metadata = metadata

    @property
    def fparameter(self) -> FParameter:
        """
        :returns: a representation of this :class:`~forge._parameter.VarKeyword`
            as a :class:`~forge.FParameter` of :term:`parameter kind`
            :term:`var-keyword`, with attributes ``name``, ``converter``,
            ``validator`` and ``metadata`` from the instance.
        """
        # pylint: disable=E1101, no-member
        return FParameter.create_var_keyword(
            name=self.name,
            type=self.type,
            converter=self.converter,
            validator=self.validator,
            metadata=self.metadata,
        )

    def __getitem__(self, key: str) -> FParameter:
        """
        Concrete method for :class:`collections.abc.Mapping`

        :key: only retrieves for :paramref:`.VarKeyword.name`
        :raise: KeyError (if ``key`` is not
            :paramref:`~forge._parameter.VarKeyword.name`)
        :returns: a representation of this
            :class:`~forge._parameter.VarKeyword` as a
            :class:`~forge.FParameter` via
            :attr:`~forge._parameter.VarKeyword.fparameter`.
        """
        if self.name == key:
            return self.fparameter
        raise KeyError(key)

    def __iter__(self) -> typing.Iterator[str]:
        """
        Concrete method for :class:`collections.abc.Mapping`

        :returns: an iterable consisting of one item: the representation of this
            :class:`~forge._parameter.VarKeyword` as a
            :class:`~forge.FParameter` via
            :attr:`~forge._parameter.VarKeyword.fparameter`.
        """
        return iter({self.name: self.fparameter})

    def __len__(self) -> int:
        """
        Concrete method for :class:`collections.abc.Mapping`

        :returns: 1
        """
        return 1

    def __call__(
            self,
            name: _TYPE_FP_NAME = None,
            *,
            type: _TYPE_FP_TYPE = empty,
            converter: _TYPE_FP_CONVERTER = None,
            validator: _TYPE_FP_VALIDATOR = None,
            metadata: typing.Optional[_TYPE_FP_METADATA] = None
        ) -> 'VarKeyword':
        """
        A factory method which creates a new
        :class:`~forge._parameter.VarKeyword` instance.
        Convenient for use like::

            **kwargs(
                converter=lambda ctx, name, value:
                    {'_' + k: v for k, v in value.items()},
            )

        :param name: see :paramref:`~forge.FParameter.name`
        :param type: see :paramref:`~forge.FParameter.type`
        :param converter: see :paramref:`~forge.FParameter.converter`
        :param validator: see :paramref:`~forge.FParameter.validator`
        :param metadata: see :paramref:`~forge.FParameter.metadata`
        :returns: a new instance of :class:`~forge._parameter.VarKeyword`
        """
        # pylint: disable=W0622, redefined-builtin
        return builtins.type(self)(
            name=name,
            type=type,
            converter=converter,
            validator=validator,
            metadata=metadata,
        )

# Convenience
kwargs = VarKeyword()

_T_PARAM = typing.TypeVar('_T_PARAM', inspect.Parameter, FParameter)
_TYPE_FINDITER_PARAMETERS = typing.Iterable[_T_PARAM]
_TYPE_FINDITER_SELECTOR = typing.Union[
    str,
    typing.Iterable[str],
    typing.Callable[[_T_PARAM], bool],
]

def findparam(
        parameters: _TYPE_FINDITER_PARAMETERS,
        selector: _TYPE_FINDITER_SELECTOR
    ) -> typing.Iterator[_T_PARAM]:
    """
    Return an iterator yielding those parameters (of type
    :class:`inspect.Parameter` or :class:`~forge.FParameter`) that are
    mached by the selector.

    :paramref:`~forge.findparam.selector` is used differently based on what is
    supplied:

    - str: a parameter is found if its :attr:`name` attribute is contained
    - Iterable[str]: a parameter is found if its :attr:`name` attribute is
        contained
    - callable: a parameter is found if the callable (which receives the
        parameter), returns a truthy value.

    :param parameters: an iterable of :class:`inspect.Parameter` or
        :class:`~forge.FParameter`
    :param selector: an identifier which is used to determine whether a
        parameter matches.
    :returns: an iterator yield parameters
    """
    if isinstance(selector, str):
        return filter(lambda param: param.name == selector, parameters)
    elif isinstance(selector, typing.Iterable):
        selector = list(selector)
        return filter(
            lambda param: param.name in selector,  # type: ignore
            parameters,
        )
    return filter(selector, parameters) # else: callable(selector)


def get_context_parameter(parameters: typing.Iterable[FParameter]):
    """
    Get the first context parameter from the provided parameters.

    :param parameters: parameters to search for a ``contextual`` parameter
    :returns: the first :term:`var-keyword` parameter from
        :paramref:`get_var_keyword_parameters.parameters` if it exists,
        else ``None``.
    """
    try:
        return next(findparam(parameters, lambda p: p.contextual))
    except StopIteration:
        return None


def get_var_keyword_parameter(parameters: _TYPE_FINDITER_PARAMETERS):
    """
    Get the first :term:`var-keyword` :term:`parameter kind` from the provided
    parameters.

    :param parameters: parameters to search for :term:`var-keyword`
        :term:`parameter kind`.
    :returns: the first :term:`var-keyword` parameter from
        :paramref:`get_var_keyword_parameters.parameters` if it exists,
        else ``None``.
    """
    try:
        return next(findparam(parameters, lambda p: p.kind is VAR_KEYWORD))
    except StopIteration:
        return None


def get_var_positional_parameter(parameters: _TYPE_FINDITER_PARAMETERS):
    """
    Get the first :term:`var-positional` :term:`parameter kind` from the
    provided parameters.

    :param parameters: parameters to search for :term:`var-positional`
        :term:`parameter kind`.
    :returns: the first :term:`var-positional` parameter from
        :paramref:`get_var_positional_parameters.parameters` if it exists,
        else ``None``.
    """
    try:
        return next(findparam(parameters, lambda p: p.kind is VAR_POSITIONAL))
    except StopIteration:
        return None


class FSignature(immutable.Immutable, collections.abc.Sequence):
    """
    An immutable, validated representation of a signature composed of
    :class:`~forge.FParameter` instances, and a return type annotation.

    Sequence methods are supported and ``__getitem__`` is overloaded to provide
    access to parameters by index, name, or a slice.
    Described in further detail: :meth:`~forge.FSignature.__getitem__`

    :param parameters: an iterable of :class:`~forge.FParameter` that makes up
        the signature
    :param return_annotation: the return type annotation for the signature
    :param __validate_parameters__: whether the sequence of provided parameters
        should be validated
    """
    __slots__ = ('_data', 'return_annotation')

    def __init__(
            self,
            parameters: typing.Optional[typing.Iterable[FParameter]] = None,
            *,
            return_annotation: typing.Any = empty.native,
            __validate_parameters__: bool = False
        ) -> None:
        super().__init__(
            _data=list(parameters or ()),
            return_annotation=return_annotation,
        )
        if __validate_parameters__:
            self.validate()

    def __len__(self):
        return len(self._data)

    @typing.overload
    def __getitem__(self, index: int) -> FParameter:
        pass # pragma: no cover

    @typing.overload
    def __getitem__(self, index: slice) -> typing.List[FParameter]:
        # pylint: disable=E0102, function-redefined
        pass # pragma: no cover

    @typing.overload
    def __getitem__(self, index: str) -> FParameter:
        # pylint: disable=E0102, function-redefined
        pass # pragma: no cover

    def __getitem__(self, index):
        """
        Depending on the type of ``index`` (integer, string, or slice), this
        method returns :class:`~forge.FParameter <parameters>` using the
        following strategies:

        - ``index`` as ``str``: the first parameter (and if the signature is \
        validated, the *only* parameter) with ``index`` as a ``name`` is
        returned. \
        If no parameter is found, then a :class:`KeyError` is raised.

        - ``index`` as ``int``: the parameter at the ``index`` is returned. \
        If no parameter is found, then an :class:`IndexError` is raised.

        - ``index`` as a ``str`` slice: when accessing parameters using str \
        slice notation, e.g. ``fsignature['a':'c']``, all parameters \
        (beginning with the parameter with name 'a', and ending *inclusively* \
        with the parameter with name 'c', will be returned. \
        The ``step`` value of ``slice`` must not be provided.

        - ``index`` as an ``int`` slice: when accessing parameters using int \
        slice notation, e.g. ``fsignature[0:3]``, all parameters \
        (beginning with the parameter at index 0, and ending with the
        parameter before index 3, will be returned. \
        The ``step`` value of ``slice`` can be provided.

        :param index: a parameter index, name, or slice of indices or names
        :raises KeyError: if an instance of :class:`~forge.FParameter` with
            :paramref:`~forge.FParameter.name` doesn't exist on this
            :class:`~forge.FSignature`.
        :returns: the instance of :class:`~forge.FParameter.name` for which
            :paramref:`~forge.FSignature.__getitem__.index` corresponds.
        """
        # pylint: disable=E0102, function-redefined
        if isinstance(index, slice):
            typemap = dict(
                start=type(index.start),
                stop=type(index.stop),
                step=type(index.step),
            )
            if set([int, type(None)]) >= set(typemap.values()):
                # slice with ints
                return self._data[index]

            if set([str, type(None)]) >= set(typemap.values()):
                # slice with strings
                if getattr(index, 'step', None):
                    raise TypeError('string slices cannot have a step')

                params = []
                visited_start = not bool(index.start)
                for param in self._data:
                    if param.name == index.start:
                        visited_start = True
                        params.append(param)
                    elif param.name == index.stop:
                        params.append(param)
                        break
                    elif visited_start:
                        params.append(param)
                return params

            raise TypeError(
                'slice arguments must all be integers or all be strings'
            )

        if isinstance(index, int):
            return self._data[index]

        if isinstance(index, str):
            for param in self._data:
                if param.name == index:
                    return param
            raise KeyError(index)

        raise TypeError(
            "indices must be integers, strings or slices, not {}".\
            format(getattr(type(index), '__name__', repr(index)))
        )

    def __str__(self) -> str:
        components = []
        if self:
            pos_param = next(
                findparam(self, lambda p: p.kind is POSITIONAL_ONLY),
                None,
            )
            has_positional = bool(pos_param)
            vpo_param = get_var_positional_parameter(self)
            has_var_positional = bool(vpo_param)

            for i, param in enumerate(self):
                last_ = self[i - 1] if (i > 0) else None
                next_ = self[i + 1] if (len(self) > i + 1) else None

                if (
                        not has_var_positional and
                        self[i].kind is KEYWORD_ONLY and
                        (not last_ or last_.kind is not KEYWORD_ONLY)
                    ):
                    components.append('*')

                components.append(str(param))
                if (
                        has_positional and
                        self[i].kind is POSITIONAL_ONLY and
                        (not next_ or next_.kind is not POSITIONAL_ONLY)
                    ):
                    components.append('/')

        ra_str = ' -> {}'.format(
            inspect.formatannotation(self.return_annotation)
        ) if self.return_annotation is not empty.native else ''

        return '({}){}'.format(', '.join(components), ra_str)

    def __repr__(self) -> str:
        return '<{} {}>'.format(type(self).__name__, self)

    @classmethod
    def from_native(cls, signature: inspect.Signature) -> 'FSignature':
        """
        A factory method that creates an instance of
        :class:`~forge.FSignature` from an instance of
        :class:`inspect.Signature`.
        Calls down to :class:`~forge.FParameter` to map the
        :attr:`inspect.Signature.parameters` to :class:`inspect.Parameter`
        instances.

        The ``return type`` annotation from the provided signature is not
        retained, as :meth:`~forge.FSignature.from_native` doesn't provide
        this functionality.

        :param signature: an instance of :class:`inspect.Signature` from which
            to derive the :class:`~forge.FSignature`
        :returns: an instance of :class:`~forge.FSignature` derived from the
            :paramref:`~forge.FSignature.from_native.signature` argument.
        """
        # pylint: disable=E1101, no-member
        return cls([
            FParameter.from_native(native)
            for native in signature.parameters.values()
        ], return_annotation=signature.return_annotation)

    @classmethod
    def from_callable(cls, callable: typing.Callable) -> 'FSignature':
        """
        A factory method that creates an instance of
        :class:`~forge.FSignature` from a callable. Calls down to
        :meth:`~forge.FSignature.from_native` to do the heavy loading.

        :param callable: a callable from which to derive the
            :class:`~forge.FSignature`
        :returns: an instance of :class:`~forge.FSignature` derived from the
            :paramref:`~forge.FSignature.from_callable.callable` argument.
        """
        # pylint: disable=W0622, redefined-builtin
        return cls.from_native(inspect.signature(callable))

    @property
    def native(self) -> inspect.Signature:
        """
        Provides a representation of this :class:`~forge.FSignature` as an
        instance of :class:`inspect.Signature`
        """
        return inspect.Signature(
            [param.native for param in self if not param.bound],
            return_annotation=self.return_annotation,
        )

    def replace(
            self,
            *,
            parameters=void,
            return_annotation=void,
            __validate_parameters__=True
        ) -> 'FSignature':
        """
        Returns a copy of this :class:`~forge.FSignature` with replaced
        attributes.

        :param parameters: see :paramref:`~forge.FSignature.parameters`
        :param return_annotation: see
            :paramref:`~forge.FSignature.return_annotation`
        :param __validate_parameters__: see
            :paramref:`~forge.FSignature.__validate_parameters__`
        :returns: a new copy of :class:`~forge.FSignature` revised with
            replacements
        """
        return type(self)(  # type: ignore
            parameters=parameters \
                if parameters is not void \
                else self._data,
            return_annotation=return_annotation \
                if return_annotation is not void \
                else self.return_annotation,
            __validate_parameters__=__validate_parameters__,
        )

    @property
    def parameters(self) -> types.MappingProxyType:
        """
        The signature's :class:`~forge.FParameter <parameters>`
        """
        return types.MappingProxyType(
            collections.OrderedDict([(p.name, p) for p in self._data])
        )

    def validate(self):
        """
        Validation ensures:

        - the appropriate order of parameters by kind:

            #. (optional) :term:`positional-only`, followed by
            #. (optional) :term:`positional-or-keyword`, followed by
            #. (optional) :term:`var-positional`, followed by
            #. (optional) :term:`keyword-only`, followed by
            #. (optional) :term:`var-keyword`

        - that non-default :term:`positional-only` or
            :term:`positional-or-keyword` parameters don't follow their
            respective similarly-kinded parameters with defaults,

            .. note::

                Python signatures allow non-default :term:`keyword-only`
                parameters to follow default :term:`keyword-only` parameters.

        - that at most there is one :term:`var-positional` parameter,

        - that at most there is one :term:`var-keyword` parameter,

        - that at most there is one ``context`` parameter, and that it
            is the first parameter (if it is provided.)

        - that no two instances of :class:`~forge.FParameter` share the same
            :paramref:`~forge.FParameter.name` or
            :paramref:`~forge.FParameter.interface_name`.
        """
        # pylint: disable=R0912, too-many-branches
        name_set = set()  # type: typing.Set[str]
        iname_set = set()  # type: typing.Set[str]
        for i, current in enumerate(self._data):
            if not isinstance(current, FParameter):
                raise TypeError(
                    "Received non-FParameter '{}'".\
                    format(current)
                )
            elif not (current.name and current.interface_name):
                raise ValueError(
                    "Received unnamed parameter: '{}'".\
                    format(current)
                )
            elif current.contextual:
                if i > 0:
                    raise TypeError(
                        'Only the first parameter can be contextual'
                    )

            if current.name in name_set:
                raise ValueError(
                    "Received multiple parameters with name '{}'".\
                    format(current.name)
                )
            name_set.add(current.name)

            if current.interface_name in iname_set:
                raise ValueError(
                    "Received multiple parameters with interface_name '{}'".\
                    format(current.interface_name)
                )
            iname_set.add(current.interface_name)

            last = self._data[i-1] if i > 0 else None
            if not last:
                continue

            elif current.kind < last.kind:
                raise SyntaxError(
                    "'{current}' of kind '{current.kind.name}' follows "
                    "'{last}' of kind '{last.kind.name}'".\
                    format(current=current, last=last)
                )
            elif current.kind is last.kind:
                if current.kind is FParameter.VAR_POSITIONAL:
                    raise TypeError(
                        'Received multiple variable-positional parameters'
                    )
                elif current.kind is FParameter.VAR_KEYWORD:
                    raise TypeError(
                        'Received multiple variable-keyword parameters'
                    )
                elif current.kind in (
                        FParameter.POSITIONAL_ONLY,
                        FParameter.POSITIONAL_OR_KEYWORD
                    ) \
                    and last.default is not empty \
                    and current.default is empty:
                    raise SyntaxError(
                        'non-default parameter follows default parameter'
                    )

fsignature = FSignature.from_callable  # Convenience
