"""Collection of subdivisions for polygonal chains."""
from numpy import zeros

from ..internal import _get_vertices_information


def jareks_subdivision(vertices, is_closed):
    """Return subdivided vertices for given `vertices` using Jarek's scheme."""
    vertices, number_of_vertices, number_of_dimensions = _get_vertices_information(
        vertices=vertices
    )
    if number_of_vertices < 3:
        raise ValueError(
            "Subdivision requires at least three vertices ({} given).".format(
                number_of_vertices
            )
        )
    if is_closed:
        subdivided_vertices = zeros([2 * number_of_vertices, number_of_dimensions])

        subdivided_vertices[0] = 0.0625 * (
            vertices[-1] + 14 * vertices[0] + vertices[1]
        )
        subdivided_vertices[2:-2:2] = 0.0625 * (
            vertices[:-2] + 14 * vertices[1:-1] + vertices[2:]
        )
        subdivided_vertices[-2] = 0.0625 * (
            vertices[-2] + 14 * vertices[-1] + vertices[0]
        )

        subdivided_vertices[1] = 0.03125 * (
            -vertices[-1] + 17 * (vertices[0] + vertices[1]) - vertices[2]
        )
        subdivided_vertices[3:-3:2] = 0.03125 * (
            -vertices[0:-3] + 17 * (vertices[1:-2] + vertices[2:-1]) - vertices[3:]
        )
        subdivided_vertices[-3] = 0.03125 * (
            -vertices[-3] + 17 * (vertices[-2] + vertices[-1]) - vertices[0]
        )
        subdivided_vertices[-1] = 0.03125 * (
            -vertices[-2] + 17 * (vertices[-1] + vertices[0]) - vertices[1]
        )
    else:
        subdivided_vertices = zeros([2 * number_of_vertices - 3, number_of_dimensions])

        subdivided_vertices[1:-1:2] = 0.0625 * (
            vertices[:-2] + 14 * vertices[1:-1] + vertices[2:]
        )

        subdivided_vertices[0] = 0.03125 * (
            -vertices[-1] + 17 * (vertices[0] + vertices[1]) - vertices[2]
        )
        subdivided_vertices[2:-1:2] = 0.03125 * (
            -vertices[0:-3] + 17 * (vertices[1:-2] + vertices[2:-1]) - vertices[3:]
        )
        subdivided_vertices[-1] = 0.03125 * (
            -vertices[-3] + 17 * (vertices[-2] + vertices[-1]) - vertices[0]
        )
    return subdivided_vertices


def modified_jareks_subdivision(vertices, is_closed):
    """
    Return subdivided vertices for given `vertices` using a modified Jarek's scheme.

    The modification applies for open polygonal chains only. Instead of introducing a
    phantom edge between the initial and terminal points, phantom edges are generated by
    prepending and appending the first and last edges defined by the polygonal chain,
    respectively.

    """
    vertices, number_of_vertices, number_of_dimensions = _get_vertices_information(
        vertices=vertices
    )
    if number_of_vertices < 3:
        raise ValueError(
            "Subdivision requires at least three vertices ({} given).".format(
                number_of_vertices
            )
        )
    if is_closed:
        subdivided_vertices = zeros([2 * number_of_vertices, number_of_dimensions])

        subdivided_vertices[0] = 0.0625 * (
            vertices[-1] + 14 * vertices[0] + vertices[1]
        )
        subdivided_vertices[2:-2:2] = 0.0625 * (
            vertices[:-2] + 14 * vertices[1:-1] + vertices[2:]
        )
        subdivided_vertices[-2] = 0.0625 * (
            vertices[-2] + 14 * vertices[-1] + vertices[0]
        )

        subdivided_vertices[1] = 0.03125 * (
            -vertices[-1] + 17 * (vertices[0] + vertices[1]) - vertices[2]
        )
        subdivided_vertices[3:-3:2] = 0.03125 * (
            -vertices[0:-3] + 17 * (vertices[1:-2] + vertices[2:-1]) - vertices[3:]
        )
        subdivided_vertices[-3] = 0.03125 * (
            -vertices[-3] + 17 * (vertices[-2] + vertices[-1]) - vertices[0]
        )
        subdivided_vertices[-1] = 0.03125 * (
            -vertices[-2] + 17 * (vertices[-1] + vertices[0]) - vertices[1]
        )
    else:
        subdivided_vertices = zeros([2 * number_of_vertices - 1, number_of_dimensions])

        subdivided_vertices[0] = vertices[0]
        subdivided_vertices[2:-1:2] = 0.0625 * (
            vertices[:-2] + 14 * vertices[1:-1] + vertices[2:]
        )
        subdivided_vertices[-1] = vertices[-1]

        subdivided_vertices[1] = 0.03125 * (
            17 * vertices[0] + 16 * vertices[1] - vertices[2]
        )
        subdivided_vertices[3:-2:2] = 0.03125 * (
            -vertices[0:-3] + 17 * (vertices[1:-2] + vertices[2:-1]) - vertices[3:]
        )
        subdivided_vertices[-2] = 0.03125 * (
            -vertices[-3] + 16 * vertices[-2] + 17 * vertices[-1]
        )

    return subdivided_vertices
