from unittest import mock

import pytest

from gerritclient.tests.unit.cli import clibase
from gerritclient.tests.utils import fake_server


class TestConfigServerCommand(clibase.BaseCLITest):
    """Tests for gerrit config server * commands."""

    @pytest.fixture(autouse=True)
    def setup_server_mocks(self, setup_client_mock):
        """Set up server-specific mocks."""
        fake_cache_info_list = fake_server.get_fake_caches_info(5)
        self.m_client.get_caches.return_value = fake_cache_info_list
        fake_cache_info = fake_server.get_fake_cache_info()
        self.m_client.get_cache.return_value = fake_cache_info

    def test_server_version_show(self):
        args = "server version"
        self.m_client.get_version.return_value = "2.14"
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_version.assert_called_once_with()

    @mock.patch("json.dump")
    def test_server_configuration_download_json(self, m_dump):
        file_format = "json"
        directory = "/tmp"
        test_data = fake_server.get_fake_config()
        args = f"server configuration download -f {file_format} -d {directory}"
        expected_path = f"{directory}/configuration.{file_format}"

        self.m_client.get_config.return_value = test_data

        m_open = mock.mock_open()
        with mock.patch("gerritclient.commands.server.open", m_open, create=True):
            self.exec_command(args)

        m_open.assert_called_once_with(expected_path, "w")
        m_dump.assert_called_once_with(test_data, mock.ANY, indent=4)
        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_config.assert_called_once_with()

    @mock.patch("yaml.safe_dump")
    def test_server_configuration_download_yaml(self, m_safe_dump):
        file_format = "yaml"
        directory = "/tmp"
        test_data = fake_server.get_fake_config()
        args = f"server configuration download -f {file_format} -d {directory}"
        expected_path = f"{directory}/configuration.{file_format}"

        self.m_client.get_config.return_value = test_data

        m_open = mock.mock_open()
        with mock.patch("gerritclient.commands.server.open", m_open, create=True):
            self.exec_command(args)

        m_open.assert_called_once_with(expected_path, "w")
        m_safe_dump.assert_called_once_with(
            test_data, mock.ANY, default_flow_style=False
        )
        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_config.assert_called_once_with()

    @mock.patch("json.dump")
    def test_capabilities_download_json(self, m_dump):
        file_format = "json"
        directory = "/tmp"
        test_data = fake_server.get_fake_capabilities()
        args = f"server capabilities download -f {file_format} -d {directory}"
        expected_path = f"{directory}/capabilities.{file_format}"

        self.m_client.get_capabilities.return_value = test_data

        m_open = mock.mock_open()
        with mock.patch("gerritclient.commands.server.open", m_open, create=True):
            self.exec_command(args)

        m_open.assert_called_once_with(expected_path, "w")
        m_dump.assert_called_once_with(test_data, mock.ANY, indent=4)
        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_capabilities.assert_called_once_with()

    @mock.patch("yaml.safe_dump")
    def test_capabilities_download_yaml(self, m_safe_dump):
        file_format = "yaml"
        directory = "/tmp"
        test_data = fake_server.get_fake_capabilities()
        args = f"server capabilities download -f {file_format} -d {directory}"
        expected_path = f"{directory}/capabilities.{file_format}"

        self.m_client.get_capabilities.return_value = test_data

        m_open = mock.mock_open()
        with mock.patch("gerritclient.commands.server.open", m_open, create=True):
            self.exec_command(args)

        m_open.assert_called_once_with(expected_path, "w")
        m_safe_dump.assert_called_once_with(
            test_data, mock.ANY, default_flow_style=False
        )
        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_capabilities.assert_called_once_with()

    def test_server_cache_list(self):
        args = "server cache list"
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_caches.assert_called_once_with()

    def test_server_cache_show(self):
        fake_cache = "fake_cache"
        args = f"server cache show {fake_cache}"
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_cache.assert_called_once_with(fake_cache)

    def test_server_cache_flush_all(self):
        args = "server cache flush --all"
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.flush_caches.assert_called_once_with(is_all=True, names=None)

    def test_server_cache_flush_several(self):
        fake_caches = ["fake_cache_1", "fake_cache_2"]
        args = "server cache flush --name {}".format(" ".join(fake_caches))
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.flush_caches.assert_called_once_with(
            is_all=False, names=fake_caches
        )

    @mock.patch("sys.stderr")
    def test_server_cache_flush_w_mutually_exclusive_params_fail(self, m_stderr):
        args = "server cache flush --name fake_cache1 fake_cache_2 --all"
        with pytest.raises(SystemExit):
            self.exec_command(args)
        assert "not allowed" in m_stderr.write.call_args_list[-1][0][0]

    @mock.patch("sys.stderr")
    def test_server_cache_flush_wo_params_fail(self, m_stderr):
        args = "server cache flush"
        with pytest.raises(SystemExit):
            self.exec_command(args)
        assert "error: one of the arguments" in m_stderr.write.call_args_list[-1][0][0]

    def test_server_get_summary_state_show(self):
        args = "server state show --max-width 110"
        fake_summary_state = fake_server.get_fake_summary_state()
        self.m_client.get_summary_state.return_value = fake_summary_state
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_summary_state.assert_called_once_with(False, False)

    def test_server_get_summary_state_show_w_jvm_gc(self):
        args = "server state show --jvm --gc --max-width 110"
        fake_summary_state = fake_server.get_fake_summary_state()
        self.m_client.get_summary_state.return_value = fake_summary_state
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_summary_state.assert_called_once_with(True, True)

    def test_server_task_list(self):
        args = "server task list"
        fake_task_list = fake_server.get_fake_tasks(5)
        self.m_client.get_tasks.return_value = fake_task_list
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_tasks.assert_called_once_with()

    def test_server_task_show(self):
        task_id = "62dc1cee"
        args = f"server task show {task_id}"
        fake_task = fake_server.get_fake_task(task_id=task_id)
        self.m_client.get_tasks.return_value = fake_task
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.get_task.assert_called_once_with(task_id)

    def test_server_task_delete(self):
        task_id = "62dc1cee"
        args = f"server task delete {task_id}"
        self.exec_command(args)

        self.m_get_client.assert_called_once_with("server", mock.ANY)
        self.m_client.delete_task.assert_called_once_with(task_id)
