"""Gitea User resource."""

from __future__ import annotations

from typing import Any, cast

from requests import Response

from gitea.resource.resource import Resource
from gitea.user.base import BaseUser
from gitea.utils.response import process_response


class User(BaseUser, Resource):
    """Gitea User resource."""

    def _get_user(self, username: str | None = None, **kwargs: Any) -> Response:
        """Get user information.

        Args:
            username: The username of the user to retrieve. If None, retrieves the authenticated user.
            **kwargs: Additional arguments for the request.

        Returns:
            The authenticated user's information as a Response object.
        """
        endpoint, kwargs = self._get_user_helper(username=username, **kwargs)
        return self._get(endpoint=endpoint, **kwargs)

    def get_user(self, username: str | None = None, **kwargs: Any) -> tuple[dict[str, Any], int]:
        """Get user information.

        Args:
            username: The username of the user to retrieve. If None, retrieves the authenticated user.
            **kwargs: Additional arguments for the request.

        Returns:
            A tuple containing the user information as a dictionary and the status code.
        """
        response = self._get_user(username=username, **kwargs)
        data, status_code = process_response(response)
        return cast(dict[str, Any], data), status_code

    def _update_user_settings(  # noqa: PLR0913
        self,
        diff_view_style: str | None = None,
        full_name: str | None = None,
        hide_activity: bool | None = None,
        hide_email: bool | None = None,
        language: str | None = None,
        location: str | None = None,
        theme: str | None = None,
        website: str | None = None,
        **kwargs: Any,
    ) -> Response:
        endpoint, payload, kwargs = self._update_user_settings_helper(
            diff_view_style=diff_view_style,
            full_name=full_name,
            hide_activity=hide_activity,
            hide_email=hide_email,
            language=language,
            location=location,
            theme=theme,
            website=website,
            **kwargs,
        )
        return self._patch(endpoint=endpoint, json=payload, **kwargs)

    def update_user_settings(  # noqa: PLR0913
        self,
        diff_view_style: str | None = None,
        full_name: str | None = None,
        hide_activity: bool | None = None,
        hide_email: bool | None = None,
        language: str | None = None,
        location: str | None = None,
        theme: str | None = None,
        website: str | None = None,
        **kwargs: Any,
    ) -> tuple[dict[str, Any], int]:
        """Update user settings.

        Args:
            diff_view_style: The preferred diff view style.
            full_name: The full name of the user.
            hide_activity: Whether to hide the user's activity.
            hide_email: Whether to hide the user's email.
            language: The preferred language.
            location: The location of the user.
            theme: The preferred theme.
            website: The user's website.
            **kwargs: Additional arguments for the request.

        Returns:
            A tuple containing the updated user settings as a dictionary and the status code.
        """
        response = self._update_user_settings(
            diff_view_style=diff_view_style,
            full_name=full_name,
            hide_activity=hide_activity,
            hide_email=hide_email,
            language=language,
            location=location,
            theme=theme,
            website=website,
            **kwargs,
        )
        data, status_code = process_response(response)
        return cast(dict[str, Any], data), status_code
