# GitLab MCP Server - Product Requirements Document

**Version:** 1.0
**Date:** October 22, 2025
**Status:** Draft

---

## 1. Executive Summary

### 1.1 Overview
This document outlines the requirements for developing a Model Context Protocol (MCP) server that enables AI tools like Claude Code to interact seamlessly with self-hosted GitLab repositories. The server will provide a standardized interface for AI assistants to perform repository management, issue tracking, merge request operations, and CI/CD pipeline management through natural language interactions.

### 1.2 Objectives
- Enable Claude Code and other MCP-compatible AI tools to access GitLab data and perform operations
- Provide a comprehensive set of tools covering repository operations, issues, merge requests, and CI/CD pipelines
- Implement secure authentication using Personal Access Tokens (PAT)
- Deliver a local Python-based server that's easy to install and configure
- Model functionality after the GitHub MCP server while adapting to GitLab's unique features

### 1.3 Scope

**In Scope:**
- Local MCP server implementation in Python
- Personal Access Token (PAT) authentication
- Single GitLab instance configuration
- Repository operations (browse, search, commits, branches, tags)
- Issues and Merge Requests management
- CI/CD pipeline operations
- Security scanning integration
- Wikis, snippets, and project management features
- Comprehensive error handling and logging

**Out of Scope (Future Enhancements):**
- OAuth 2.0 authentication
- Remote hosted deployment model
- Multi-instance support
- Semantic/AI-powered code search (GitLab Duo integration)
- Real-time WebSocket notifications

---

## 2. Project Background

### 2.1 Model Context Protocol (MCP)
The Model Context Protocol is an open standard introduced by Anthropic that standardizes how AI systems integrate with external data sources and tools. MCP uses a client-server architecture where:
- **MCP Host**: Applications like Claude Code that initiate connections
- **MCP Client**: Connectors within the host that maintain connections to servers
- **MCP Server**: Services that expose tools, resources, and prompts through the protocol

### 2.2 GitLab Integration Need
GitLab is a comprehensive DevSecOps platform offering repository management, CI/CD, issue tracking, and security features. Many organizations use self-hosted GitLab instances for their development workflows. An MCP server for GitLab will:
- Enable AI-assisted code reviews and analysis
- Automate issue and merge request management
- Provide intelligent CI/CD pipeline monitoring
- Facilitate natural language interactions with GitLab data

### 2.3 Reference Implementation
The GitHub MCP server serves as the primary reference, offering:
- Organized toolset categories (repos, issues, pull_requests, actions, etc.)
- Comprehensive API coverage
- Proven patterns for authentication and error handling
- Resource and prompt support for enhanced AI context

---

## 3. Goals & Success Metrics

### 3.1 Goals
1. **Functional Completeness**: Implement 90%+ of common GitLab operations used in daily development workflows
2. **Ease of Use**: Users can install and configure the server in under 5 minutes
3. **Reliability**: 99%+ successful API call rate with proper error handling
4. **Performance**: Response times under 2 seconds for 95% of operations
5. **Security**: Secure credential storage and API communication

### 3.2 Success Metrics
- Number of successful tool invocations
- User adoption rate among development teams
- Error rate and types
- Average response time per tool
- User satisfaction score (post-launch survey)

---

## 4. User Personas & Use Cases

### 4.1 User Personas

**Persona 1: Software Developer (Sarah)**
- Uses Claude Code for daily coding tasks
- Works with self-hosted GitLab for version control
- Needs to quickly search code, review MRs, and check CI/CD status
- Values efficiency and natural language interactions

**Persona 2: DevOps Engineer (David)**
- Manages CI/CD pipelines and infrastructure
- Monitors pipeline failures and deployment status
- Needs to quickly diagnose build failures and trigger reruns
- Requires comprehensive pipeline and job information

**Persona 3: Engineering Manager (Maria)**
- Tracks team progress through issues and merge requests
- Reviews project status and milestone progress
- Needs quick access to issue statistics and MR review status
- Values high-level overviews and summaries

### 4.2 Use Cases

**UC-1: Code Search and Review**
- User asks Claude Code: "Find all functions that handle user authentication"
- Server searches repository files using GitLab Search API
- Returns relevant code snippets with file locations

**UC-2: Merge Request Analysis**
- User asks: "What changed in MR #123?"
- Server fetches MR details, diffs, commits, and pipeline status
- Provides comprehensive summary of changes and CI/CD results

**UC-3: CI/CD Pipeline Monitoring**
- User asks: "Why did the latest pipeline fail?"
- Server retrieves pipeline jobs, identifies failed jobs, and fetches logs
- Presents failure reasons and suggests potential fixes

**UC-4: Issue Management**
- User asks: "Create an issue to fix the authentication bug with high priority"
- Server creates issue with appropriate labels, milestone, and assignee
- Confirms creation and provides issue URL

**UC-5: Repository Operations**
- User asks: "Show me recent commits on the main branch"
- Server fetches commit history with author, date, and messages
- Presents formatted commit log

---

## 5. Technical Requirements

### 5.1 Technology Stack

**Primary Language**: Python 3.9+

**Key Dependencies**:
- `mcp` - Official Python MCP SDK
- `python-gitlab` - GitLab API client library
- `pydantic` - Data validation and settings management
- `python-dotenv` - Environment variable management
- `pytest` - Testing framework
- `httpx` - Async HTTP client (if async operations needed)

### 5.2 Architecture

```
┌─────────────────┐
│   Claude Code   │
│   (MCP Host)    │
└────────┬────────┘
         │ MCP Protocol
         │ (JSON-RPC)
┌────────▼────────┐
│  GitLab MCP     │
│     Server      │
│   (Python)      │
└────────┬────────┘
         │ REST API
         │ (HTTPS)
┌────────▼────────┐
│  Self-Hosted    │
│ GitLab Instance │
└─────────────────┘
```

### 5.3 Configuration

**Environment Variables**:
```
GITLAB_URL=https://gitlab.example.com
GITLAB_TOKEN=glpat-xxxxxxxxxxxxxxxxxxxx
GITLAB_DEFAULT_PROJECT=group/project-name (optional)
LOG_LEVEL=INFO (optional, default: INFO)
```

**Configuration File** (.gitlab-mcp.json):
```json
{
  "gitlab_url": "https://gitlab.example.com",
  "default_project": "group/project-name",
  "timeout": 30,
  "verify_ssl": true
}
```

### 5.4 Authentication
- **Method**: Personal Access Token (PAT)
- **Scopes Required**: `api`, `read_repository`, `write_repository` (optional for write operations)
- **Storage**: Environment variables or secure configuration file
- **Validation**: Token validation on server startup

### 5.5 Deployment Model
- **Type**: Local server running on user's machine
- **Installation**: pip package or standalone executable
- **Execution**: Started via command line or configured in Claude Code settings
- **Communication**: stdio transport (standard input/output) with JSON-RPC 2.0

---

## 6. Feature Requirements

### 6.1 Phase 1: MVP (Minimum Viable Product)

#### 6.1.1 Core Infrastructure
- **MCP-001**: MCP server initialization with proper protocol handshake
- **MCP-002**: Tool discovery and listing
- **MCP-003**: Error handling with informative error messages
- **MCP-004**: Logging with configurable levels
- **MCP-005**: Configuration management (env vars + config file)
- **MCP-006**: GitLab API client initialization and authentication

#### 6.1.2 Context Tools
- **CTX-001**: `get_current_user` - Get authenticated user information
- **CTX-002**: `get_server_version` - Get MCP server version
- **CTX-003**: `get_gitlab_version` - Get GitLab instance version

#### 6.1.3 Repository Operations
- **REPO-001**: `search_code` - Search for code across repositories
- **REPO-002**: `get_file_contents` - Get contents of a specific file
- **REPO-003**: `list_repository_tree` - List files and directories in a repository
- **REPO-004**: `get_commit` - Get details of a specific commit
- **REPO-005**: `list_commits` - List commits for a branch or repository
- **REPO-006**: `list_branches` - List repository branches
- **REPO-007**: `get_branch` - Get details of a specific branch
- **REPO-008**: `create_branch` - Create a new branch
- **REPO-009**: `delete_branch` - Delete a branch
- **REPO-010**: `list_tags` - List repository tags
- **REPO-011**: `get_tag` - Get details of a specific tag
- **REPO-012**: `create_tag` - Create a new tag
- **REPO-013**: `compare_branches` - Compare two branches or commits
- **REPO-014**: `get_repository` - Get repository details and metadata
- **REPO-015**: `create_file` - Create a new file in repository with commit
- **REPO-016**: `update_file` - Update existing file content with commit
- **REPO-017**: `delete_file` - Delete a file from repository with commit

#### 6.1.4 Issues
- **ISSUE-001**: `create_issue` - Create a new issue
- **ISSUE-002**: `get_issue` - Get details of a specific issue
- **ISSUE-003**: `update_issue` - Update an existing issue
- **ISSUE-004**: `list_issues` - List issues with filters (state, labels, assignee, etc.)
- **ISSUE-005**: `close_issue` - Close an issue
- **ISSUE-006**: `reopen_issue` - Reopen a closed issue
- **ISSUE-007**: `add_issue_comment` - Add a comment to an issue
- **ISSUE-008**: `list_issue_comments` - List comments on an issue
- **ISSUE-009**: `search_issues` - Search issues across projects

#### 6.1.5 Merge Requests
- **MR-001**: `create_merge_request` - Create a new merge request
- **MR-002**: `get_merge_request` - Get details of a specific merge request
- **MR-003**: `update_merge_request` - Update an existing merge request
- **MR-004**: `list_merge_requests` - List merge requests with filters
- **MR-005**: `get_merge_request_changes` - Get diff/changes in a merge request
- **MR-006**: `get_merge_request_commits` - Get commits in a merge request
- **MR-007**: `get_merge_request_pipelines` - Get CI/CD pipelines for a merge request
- **MR-008**: `merge_merge_request` - Merge a merge request
- **MR-009**: `approve_merge_request` - Approve a merge request
- **MR-010**: `unapprove_merge_request` - Remove approval from a merge request
- **MR-011**: `add_merge_request_comment` - Add a comment to a merge request
- **MR-012**: `list_merge_request_comments` - List comments and discussions
- **MR-013**: `close_merge_request` - Close a merge request
- **MR-014**: `reopen_merge_request` - Reopen a closed merge request

#### 6.1.6 CI/CD Pipelines
- **PIPE-001**: `list_pipelines` - List pipelines for a project
- **PIPE-002**: `get_pipeline` - Get details of a specific pipeline
- **PIPE-003**: `create_pipeline` - Trigger a new pipeline
- **PIPE-004**: `retry_pipeline` - Retry a failed pipeline
- **PIPE-005**: `cancel_pipeline` - Cancel a running pipeline
- **PIPE-006**: `delete_pipeline` - Delete a pipeline
- **PIPE-007**: `list_pipeline_jobs` - List jobs in a pipeline
- **PIPE-008**: `get_job` - Get details of a specific job
- **PIPE-009**: `get_job_trace` - Get job execution log
- **PIPE-010**: `retry_job` - Retry a failed job
- **PIPE-011**: `cancel_job` - Cancel a running job
- **PIPE-012**: `play_job` - Start a manual job
- **PIPE-013**: `download_job_artifacts` - Download job artifacts
- **PIPE-014**: `list_pipeline_variables` - List pipeline variables

### 6.2 Phase 2: Advanced Features

#### 6.2.1 Project Management
- **PROJ-001**: `list_projects` - List accessible projects
- **PROJ-002**: `get_project` - Get project details
- **PROJ-003**: `search_projects` - Search for projects
- **PROJ-004**: `list_project_members` - List project members
- **PROJ-005**: `get_project_statistics` - Get project statistics
- **PROJ-006**: `list_milestones` - List project milestones
- **PROJ-007**: `get_milestone` - Get milestone details
- **PROJ-008**: `create_milestone` - Create a milestone
- **PROJ-009**: `update_milestone` - Update a milestone

#### 6.2.2 Labels
- **LABEL-001**: `list_labels` - List project labels
- **LABEL-002**: `create_label` - Create a new label
- **LABEL-003**: `update_label` - Update a label
- **LABEL-004**: `delete_label` - Delete a label

#### 6.2.3 Wikis
- **WIKI-001**: `list_wiki_pages` - List wiki pages
- **WIKI-002**: `get_wiki_page` - Get wiki page content
- **WIKI-003**: `create_wiki_page` - Create a wiki page
- **WIKI-004**: `update_wiki_page` - Update a wiki page
- **WIKI-005**: `delete_wiki_page` - Delete a wiki page

#### 6.2.4 Snippets
- **SNIP-001**: `list_snippets` - List project snippets
- **SNIP-002**: `get_snippet` - Get snippet details
- **SNIP-003**: `create_snippet` - Create a snippet
- **SNIP-004**: `update_snippet` - Update a snippet
- **SNIP-005**: `delete_snippet` - Delete a snippet

#### 6.2.5 Security & Compliance
- **SEC-001**: `list_vulnerabilities` - List project vulnerabilities
- **SEC-002**: `get_vulnerability` - Get vulnerability details
- **SEC-003**: `list_security_reports` - List security scan reports
- **SEC-004**: `get_dependency_list` - Get project dependencies
- **SEC-005**: `list_license_scanning_reports` - Get license compliance reports

#### 6.2.6 Releases
- **REL-001**: `list_releases` - List project releases
- **REL-002**: `get_release` - Get release details
- **REL-003**: `create_release` - Create a new release
- **REL-004**: `update_release` - Update a release
- **REL-005**: `delete_release` - Delete a release

#### 6.2.7 Users & Groups
- **USER-001**: `get_user` - Get user details
- **USER-002**: `search_users` - Search for users
- **USER-003**: `list_user_projects` - List user's projects
- **GROUP-001**: `list_groups` - List accessible groups
- **GROUP-002**: `get_group` - Get group details
- **GROUP-003**: `list_group_members` - List group members

---

## 7. MCP Tools Specification

### 7.1 Tool Organization

Tools are organized into toolsets, similar to the GitHub MCP server:

**Core Toolsets**:
1. `context` - Server and user context
2. `repos` - Repository operations
3. `issues` - Issue management
4. `merge_requests` - Merge request operations
5. `pipelines` - CI/CD pipeline and job management
6. `projects` - Project management
7. `labels` - Label operations
8. `wikis` - Wiki management
9. `snippets` - Snippet operations
10. `security` - Security scanning and vulnerability management
11. `releases` - Release management
12. `users` - User operations
13. `groups` - Group operations

### 7.2 Tool Schema Example

Each tool follows the MCP tool schema:

```python
{
    "name": "get_merge_request",
    "description": "Get detailed information about a specific merge request including status, approvals, and metadata",
    "inputSchema": {
        "type": "object",
        "properties": {
            "project": {
                "type": "string",
                "description": "Project path (e.g., 'group/project') or ID"
            },
            "merge_request_iid": {
                "type": "integer",
                "description": "Internal ID of the merge request"
            },
            "include_diverged_commits_count": {
                "type": "boolean",
                "description": "Include count of commits diverged from target branch",
                "default": false
            },
            "render_html": {
                "type": "boolean",
                "description": "Render description and title in HTML",
                "default": false
            }
        },
        "required": ["project", "merge_request_iid"]
    }
}
```

### 7.3 Response Format

All tools return responses in a consistent format:

```python
{
    "success": true,
    "data": {
        # Tool-specific response data
    },
    "metadata": {
        "rate_limit_remaining": 1500,
        "rate_limit_reset": "2025-10-22T15:30:00Z",
        "execution_time_ms": 245
    }
}
```

Error responses:

```python
{
    "success": false,
    "error": {
        "code": "GITLAB_API_ERROR",
        "message": "Merge request not found",
        "details": {
            "project": "mygroup/myproject",
            "merge_request_iid": 999,
            "http_status": 404
        }
    }
}
```

---

## 8. GitLab API Integration

### 8.1 API Endpoints Mapping

#### Repository Operations
- `GET /projects/:id/repository/tree` - List repository tree
- `GET /projects/:id/repository/files/:file_path` - Get file contents
- `POST /projects/:id/repository/files/:file_path` - Create file
- `PUT /projects/:id/repository/files/:file_path` - Update file
- `DELETE /projects/:id/repository/files/:file_path` - Delete file
- `GET /projects/:id/repository/commits` - List commits
- `GET /projects/:id/repository/commits/:sha` - Get commit details
- `GET /projects/:id/repository/branches` - List branches
- `GET /projects/:id/repository/branches/:branch` - Get branch details
- `POST /projects/:id/repository/branches` - Create branch
- `DELETE /projects/:id/repository/branches/:branch` - Delete branch
- `GET /projects/:id/repository/tags` - List tags
- `POST /projects/:id/repository/tags` - Create tag
- `GET /projects/:id/repository/compare` - Compare branches
- `GET /search` - Global search (scope: blobs for code)

#### Issues
- `GET /projects/:id/issues` - List issues
- `GET /projects/:id/issues/:issue_iid` - Get issue
- `POST /projects/:id/issues` - Create issue
- `PUT /projects/:id/issues/:issue_iid` - Update issue
- `GET /projects/:id/issues/:issue_iid/notes` - List issue comments
- `POST /projects/:id/issues/:issue_iid/notes` - Add comment

#### Merge Requests
- `GET /projects/:id/merge_requests` - List merge requests
- `GET /projects/:id/merge_requests/:merge_request_iid` - Get MR
- `POST /projects/:id/merge_requests` - Create MR
- `PUT /projects/:id/merge_requests/:merge_request_iid` - Update MR
- `GET /projects/:id/merge_requests/:merge_request_iid/changes` - Get MR changes
- `GET /projects/:id/merge_requests/:merge_request_iid/commits` - Get MR commits
- `GET /projects/:id/merge_requests/:merge_request_iid/pipelines` - Get MR pipelines
- `PUT /projects/:id/merge_requests/:merge_request_iid/merge` - Merge MR
- `POST /projects/:id/merge_requests/:merge_request_iid/approve` - Approve MR
- `GET /projects/:id/merge_requests/:merge_request_iid/notes` - List MR comments

#### Pipelines & Jobs
- `GET /projects/:id/pipelines` - List pipelines
- `GET /projects/:id/pipelines/:pipeline_id` - Get pipeline
- `POST /projects/:id/pipeline` - Create pipeline
- `POST /projects/:id/pipelines/:pipeline_id/retry` - Retry pipeline
- `POST /projects/:id/pipelines/:pipeline_id/cancel` - Cancel pipeline
- `DELETE /projects/:id/pipelines/:pipeline_id` - Delete pipeline
- `GET /projects/:id/pipelines/:pipeline_id/jobs` - List pipeline jobs
- `GET /projects/:id/jobs/:job_id` - Get job
- `GET /projects/:id/jobs/:job_id/trace` - Get job log
- `POST /projects/:id/jobs/:job_id/retry` - Retry job
- `POST /projects/:id/jobs/:job_id/cancel` - Cancel job
- `POST /projects/:id/jobs/:job_id/play` - Play manual job
- `GET /projects/:id/jobs/:job_id/artifacts` - Download artifacts

#### Projects
- `GET /projects` - List projects
- `GET /projects/:id` - Get project
- `GET /projects/:id/members` - List members
- `GET /projects/:id/statistics` - Get statistics
- `GET /projects/:id/milestones` - List milestones
- `POST /projects/:id/milestones` - Create milestone

### 8.2 Rate Limiting
- Monitor rate limit headers: `RateLimit-Remaining`, `RateLimit-Reset`
- Implement exponential backoff for rate limit errors (HTTP 429)
- Log rate limit consumption for monitoring
- Allow configurable request throttling

### 8.3 Error Handling
- Handle common HTTP status codes:
  - 401: Unauthorized (invalid token)
  - 403: Forbidden (insufficient permissions)
  - 404: Not found
  - 422: Unprocessable entity (validation errors)
  - 429: Rate limit exceeded
  - 500: Internal server error
- Provide user-friendly error messages
- Include troubleshooting suggestions where applicable

---

## 9. Configuration & Setup

### 9.1 Installation

**Via pip**:
```bash
pip install gitlab-mcp-server
```

**From source**:
```bash
git clone https://github.com/yourusername/gitlab-mcp-server.git
cd gitlab-mcp-server
pip install -e .
```

### 9.2 Configuration Steps

1. **Create Personal Access Token**:
   - Navigate to GitLab → User Settings → Access Tokens
   - Create token with scopes: `api`, `read_repository`, `write_repository`
   - Copy token securely

2. **Configure Environment**:
   ```bash
   export GITLAB_URL="https://gitlab.example.com"
   export GITLAB_TOKEN="glpat-xxxxxxxxxxxxxxxxxxxx"
   ```

3. **Optional: Create Config File**:
   ```bash
   cat > ~/.gitlab-mcp.json << EOF
   {
     "gitlab_url": "https://gitlab.example.com",
     "default_project": "mygroup/myproject",
     "timeout": 30,
     "verify_ssl": true
   }
   EOF
   ```

4. **Configure Claude Code**:
   Add to Claude Code MCP settings:
   ```json
   {
     "mcpServers": {
       "gitlab": {
         "command": "gitlab-mcp-server",
         "env": {
           "GITLAB_URL": "https://gitlab.example.com",
           "GITLAB_TOKEN": "glpat-xxxxxxxxxxxxxxxxxxxx"
         }
       }
     }
   }
   ```

### 9.3 Verification

Run server in standalone mode for testing:
```bash
gitlab-mcp-server --test
```

Expected output:
```
GitLab MCP Server v1.0.0
✓ Connected to GitLab (v16.5.0)
✓ Authenticated as: john.doe
✓ Available tools: 47
✓ Server ready
```

---

## 10. Security Considerations

### 10.1 Credential Management
- **Never log tokens**: Ensure tokens are never written to logs
- **Environment variables**: Prefer environment variables over config files for tokens
- **File permissions**: Config files should have 600 permissions (owner read/write only)
- **Token validation**: Validate token on startup and provide clear error messages
- **Scope validation**: Warn users if token lacks required scopes

### 10.2 API Security
- **HTTPS only**: Enforce HTTPS for GitLab API communication
- **SSL verification**: Enable SSL certificate verification by default (configurable for self-signed certs)
- **Timeout**: Set reasonable timeouts to prevent hanging requests
- **Input validation**: Validate all user inputs before making API calls
- **Path traversal**: Prevent path traversal attacks in file operations

### 10.3 Data Privacy
- **Sensitive data**: Redact sensitive information in logs and error messages
- **Local storage**: No persistent storage of GitLab data without explicit user consent
- **Transmission**: All data transmitted via secure channels

### 10.4 Error Handling
- **Safe error messages**: Error messages should not expose sensitive system information
- **Graceful degradation**: Continue operation when non-critical operations fail
- **Audit logging**: Log all API calls for audit purposes (optional, configurable)

---

## 11. Testing Strategy

### 11.1 Unit Tests
- Test each tool function independently
- Mock GitLab API responses
- Test error handling paths
- Validate input/output schemas
- Test configuration loading and validation
- Code coverage target: 80%+

### 11.2 Integration Tests
- Test actual GitLab API interactions (against test instance)
- Test authentication flow
- Test rate limiting handling
- Test pagination for large result sets
- Test concurrent requests

### 11.3 End-to-End Tests
- Test MCP protocol communication
- Test integration with MCP client
- Test complete user workflows (e.g., create MR from issue)
- Test error scenarios with real GitLab instance

### 11.4 Test Data
- Use dedicated test GitLab instance or project
- Create fixtures for common API responses
- Document test data setup requirements

### 11.5 Testing Tools
- `pytest` - Test framework
- `pytest-cov` - Code coverage
- `pytest-asyncio` - Async test support (if needed)
- `responses` or `httpx-mock` - HTTP mocking
- `mcp-test-client` - MCP protocol testing

---

## 12. Documentation Requirements

### 12.1 User Documentation

**README.md**:
- Overview and features
- Quick start guide
- Installation instructions
- Configuration guide
- Usage examples
- Troubleshooting
- Contributing guidelines
- License

**User Guide**:
- Detailed setup instructions
- Claude Code integration guide
- Tool reference with examples
- Common workflows
- Best practices
- FAQ

**Configuration Reference**:
- All environment variables
- Configuration file schema
- GitLab token scope requirements
- Advanced configuration options

### 12.2 Developer Documentation

**Architecture Overview**:
- System architecture diagram
- Component descriptions
- Data flow diagrams
- Technology stack details

**API Reference**:
- Tool catalog with schemas
- Response formats
- Error codes and handling
- Rate limiting information

**Development Guide**:
- Development environment setup
- Running tests
- Adding new tools
- Code style and conventions
- Release process

### 12.3 API Documentation
- OpenAPI/Swagger spec for tools (if applicable)
- MCP protocol conformance documentation
- GitLab API version compatibility

---

## 13. Detailed Feature Specifications

### 13.1 File Editing Operations (REPO-015, REPO-016, REPO-017)

**Priority**: HIGH - Enables AI-assisted code editing workflows
**Phase**: Phase 2 (after Issues tools completion) or early Phase 3
**Implementation Session**: Estimated Session 018-019

#### 13.1.1 REPO-015: create_file

**Purpose**: Create a new file in a repository with a commit message, enabling AI to generate and save new files directly.

**API Endpoint**: `POST /api/v4/projects/:id/repository/files/:file_path`

**Parameters**:
- `project_id` (required): Project ID or path
- `file_path` (required): Full path to new file (URL-encoded)
- `branch` (required): Target branch name
- `content` (required): File content
- `commit_message` (required): Commit message
- `start_branch` (optional): Create new branch from this branch
- `author_email` (optional): Override commit author email
- `author_name` (optional): Override commit author name
- `encoding` (optional): 'text' or 'base64' (default: 'text')
- `execute_filemode` (optional): Make file executable (boolean)

**Response Format**:
```python
{
    "file_path": str,
    "branch": str,
    "commit_id": str,
    "web_url": str,
    "message": str  # Success message
}
```

**Use Cases**:
- Generate new source files based on specifications
- Create configuration files
- Add documentation files
- Generate test files

**Error Handling**:
- 400: File already exists
- 400: Invalid file path (directory traversal attempt)
- 404: Project or branch not found
- 403: Insufficient permissions

#### 13.1.2 REPO-016: update_file

**Purpose**: Update existing file content with a commit, enabling AI to make edits, fixes, and refactorings.

**API Endpoint**: `PUT /api/v4/projects/:id/repository/files/:file_path`

**Parameters**:
- `project_id` (required): Project ID or path
- `file_path` (required): Full path to file (URL-encoded)
- `branch` (required): Target branch name
- `content` (required): New file content
- `commit_message` (required): Commit message
- `start_branch` (optional): Create new branch from this branch
- `last_commit_id` (optional): Last known commit ID for conflict detection
- `author_email` (optional): Override commit author email
- `author_name` (optional): Override commit author name
- `encoding` (optional): 'text' or 'base64' (default: 'text')
- `execute_filemode` (optional): Make file executable (boolean)

**Response Format**:
```python
{
    "file_path": str,
    "branch": str,
    "commit_id": str,
    "web_url": str,
    "message": str  # Success message
}
```

**Use Cases**:
- Apply bug fixes suggested by AI
- Refactor code based on analysis
- Update configuration values
- Fix linting or formatting issues
- Apply security patches

**Conflict Detection**:
- If `last_commit_id` provided and doesn't match current, return 400 error
- Prevents overwriting changes made by others
- Client should re-fetch file before retry

**Error Handling**:
- 400: File doesn't exist
- 400: Conflict detected (last_commit_id mismatch)
- 400: Content identical to existing version
- 404: Project or branch not found
- 403: Insufficient permissions

#### 13.1.3 REPO-017: delete_file

**Purpose**: Delete a file from repository with commit message.

**API Endpoint**: `DELETE /api/v4/projects/:id/repository/files/:file_path`

**Parameters**:
- `project_id` (required): Project ID or path
- `file_path` (required): Full path to file (URL-encoded)
- `branch` (required): Target branch name
- `commit_message` (required): Commit message
- `start_branch` (optional): Create new branch from this branch
- `last_commit_id` (optional): Last known commit ID for conflict detection
- `author_email` (optional): Override commit author email
- `author_name` (optional): Override commit author name

**Response Format**:
```python
{
    "file_path": str,
    "branch": str,
    "commit_id": str,
    "message": str  # Success message
}
```

**Use Cases**:
- Remove deprecated files
- Clean up temporary files
- Remove unused code
- Delete obsolete configurations

**Error Handling**:
- 400: File doesn't exist
- 400: Conflict detected
- 404: Project or branch not found
- 403: Insufficient permissions

#### 13.1.4 Implementation Notes

**File Path Encoding**:
- All file paths must be URL-encoded (e.g., `lib/class.rb` → `lib%2Fclass.rb`)
- Handle special characters properly
- Prevent directory traversal attacks

**Content Encoding**:
- Support both 'text' and 'base64' encoding
- Use base64 for binary files
- Default to text for code files

**Branch Strategy**:
- If `start_branch` provided, creates new branch from it
- Enables "edit on new branch" workflow
- Useful for AI-suggested changes that need review

**Commit Attribution**:
- Default: Use authenticated user
- Optional: Override with `author_name` and `author_email`
- Useful for attributing AI-assisted changes appropriately

**Testing Strategy**:
- Unit tests with mocked GitLab API
- Integration tests with real GitLab instance
- Test conflict detection scenarios
- Test various encodings (text, base64)
- Test special characters in paths
- Test error conditions

**Security Considerations**:
- Validate file paths to prevent directory traversal
- Sanitize file content
- Respect GitLab permissions (protected branches)
- Audit file modifications
- Consider protected file patterns

**MCP Tool Design**:
```python
async def create_file(
    client: GitLabClient,
    project_id: Union[str, int],
    file_path: str,
    branch: str,
    content: str,
    commit_message: str,
    start_branch: Optional[str] = None,
    encoding: str = "text",
) -> dict[str, Any]:
    """Create a new file in repository."""
    pass

async def update_file(
    client: GitLabClient,
    project_id: Union[str, int],
    file_path: str,
    branch: str,
    content: str,
    commit_message: str,
    last_commit_id: Optional[str] = None,
    start_branch: Optional[str] = None,
    encoding: str = "text",
) -> dict[str, Any]:
    """Update existing file in repository."""
    pass

async def delete_file(
    client: GitLabClient,
    project_id: Union[str, int],
    file_path: str,
    branch: str,
    commit_message: str,
    last_commit_id: Optional[str] = None,
    start_branch: Optional[str] = None,
) -> dict[str, Any]:
    """Delete a file from repository."""
    pass
```

**Recommended Implementation Timeline**:
- **Session 018-019**: After Issues tools complete, before Merge Requests
- **Alternative**: Early Phase 3 alongside Merge Request operations
- **Rationale**: High value for AI workflows, builds on repository tool patterns

---

## 14. Future Enhancements

### 14.1 Phase 3: OAuth 2.0 Support
- Implement OAuth 2.0 authentication flow
- Support dynamic client registration
- Handle token refresh automatically
- Provide OAuth setup documentation

### 14.2 Phase 4: Remote Hosting
- Design remote server architecture
- Implement authentication middleware
- Add multi-user support
- Deploy to cloud platform (AWS, GCP, Azure)
- Provide hosted service option

### 14.3 Phase 5: Multi-Instance Support
- Support multiple GitLab instances in single server
- Instance selection via tool parameters
- Separate credential management per instance
- Instance discovery and listing

### 14.4 Phase 6: Advanced Features
- **Semantic Code Search**: Integrate GitLab Duo semantic search
- **Real-time Updates**: WebSocket support for live notifications
- **GraphQL API**: Use GitLab GraphQL API for complex queries
- **Caching**: Implement intelligent caching for frequently accessed data
- **Webhooks**: Listen to GitLab webhooks for proactive updates
- **Batch Operations**: Support bulk operations (e.g., update multiple issues)
- **Templates**: MCP prompts for common workflows
- **Analytics**: Usage analytics and insights dashboard

### 14.5 Phase 7: Integration Enhancements
- **Jira Integration**: Link GitLab issues with Jira
- **Slack/Teams**: Post updates to communication platforms
- **Custom Webhooks**: User-configurable webhook triggers
- **Export/Import**: Export GitLab data in various formats

---

## 14. Timeline & Milestones

### 14.1 Development Phases

**Phase 1: Foundation (Weeks 1-2)**
- [ ] Project setup and repository creation
- [ ] MCP SDK integration and basic server structure
- [ ] GitLab API client setup with python-gitlab
- [ ] Authentication and configuration management
- [ ] Basic error handling and logging
- [ ] Unit test framework setup

**Phase 2: Core Tools - Repository & Issues (Weeks 3-4)**
- [ ] Implement all repository operation tools (REPO-001 to REPO-014)
- [ ] Implement all issue management tools (ISSUE-001 to ISSUE-009)
- [ ] Unit tests for repository and issue tools
- [ ] Integration tests with test GitLab instance

**Phase 3: Merge Requests & Pipelines (Weeks 5-6)**
- [ ] Implement all merge request tools (MR-001 to MR-014)
- [ ] Implement all pipeline and job tools (PIPE-001 to PIPE-014)
- [ ] Unit and integration tests
- [ ] Performance optimization

**Phase 4: Advanced Features (Weeks 7-8)**
- [ ] Implement project management tools
- [ ] Implement labels, wikis, and snippets
- [ ] Implement security and compliance tools
- [ ] Implement releases and user/group tools
- [ ] Comprehensive testing

**Phase 5: Documentation & Polish (Week 9)**
- [ ] Complete user documentation
- [ ] Complete developer documentation
- [ ] Tool reference documentation
- [ ] Example workflows and tutorials
- [ ] Code cleanup and refactoring

**Phase 6: Testing & Release (Week 10)**
- [ ] End-to-end testing with Claude Code
- [ ] Performance testing and optimization
- [ ] Security audit
- [ ] Beta testing with select users
- [ ] Bug fixes and improvements
- [ ] v1.0.0 release

### 14.2 Release Criteria

**Version 1.0.0 Release Criteria**:
- All Phase 1 (MVP) features implemented and tested
- Code coverage ≥80%
- All integration tests passing
- Documentation complete
- Successful testing with Claude Code
- No critical or high-priority bugs
- Performance meets success metrics

---

## 15. Dependencies & Risks

### 15.1 Dependencies

**External Dependencies**:
- GitLab API availability and stability
- GitLab API version compatibility
- MCP protocol specification stability
- Claude Code MCP client implementation
- Python ecosystem packages

**Internal Dependencies**:
- Development resources and expertise
- Test GitLab instance availability
- Documentation resources

### 15.2 Risks & Mitigation

| Risk | Severity | Probability | Mitigation |
|------|----------|-------------|------------|
| GitLab API changes break compatibility | High | Medium | Pin to specific API version, implement version detection, maintain compatibility matrix |
| MCP protocol changes | Medium | Low | Monitor MCP spec updates, participate in community, implement version negotiation |
| Performance issues with large repos | Medium | Medium | Implement pagination, caching, and request throttling |
| Security vulnerabilities in dependencies | High | Medium | Regular dependency updates, security scanning, use dependabot |
| Insufficient GitLab token permissions | Medium | High | Clear documentation, token validation on startup, helpful error messages |
| Rate limiting impacts usability | Medium | Medium | Implement smart rate limit handling, request queuing, user notifications |
| Complex MR diffs cause timeouts | Low | Medium | Implement configurable timeouts, streaming responses, diff size limits |

---

## 16. Open Questions

1. **Caching Strategy**: Should we implement caching for frequently accessed data? What should the cache TTL be?

2. **Pagination Defaults**: What should be the default page size for list operations? How should we handle very large result sets?

3. **Async vs Sync**: Should we implement async/await for all API calls, or is synchronous sufficient for MVP?

4. **Resource Support**: Should we implement MCP resources (for providing context) in addition to tools? What resources would be most useful?

5. **Prompt Support**: Should we provide MCP prompts for common workflows (e.g., "create a release", "review open MRs")?

6. **Diff Handling**: How should we handle very large diffs that might exceed token limits in Claude Code?

7. **Project Resolution**: Should project parameters accept both IDs and paths? How do we handle ambiguity?

8. **Error Verbosity**: What level of error detail should we return to users vs. log internally?

---

## 17. Appendices

### Appendix A: GitLab API Version Compatibility

Target compatibility:
- GitLab CE/EE version: 15.0+
- GitLab API version: v4
- Tested versions: 15.x, 16.x, 17.x

### Appendix B: MCP Protocol Version

- MCP Protocol Version: 2025-06-18 specification
- Transport: stdio (JSON-RPC 2.0)
- SDK: Official Python MCP SDK (latest)

### Appendix C: Related Projects

- GitHub MCP Server: https://github.com/github/github-mcp-server
- GitLab Official MCP Server: https://docs.gitlab.com/user/gitlab_duo/model_context_protocol/mcp_server/
- python-gitlab: https://python-gitlab.readthedocs.io/
- MCP Python SDK: https://github.com/modelcontextprotocol/python-sdk

### Appendix D: Glossary

- **MCP**: Model Context Protocol
- **PAT**: Personal Access Token
- **MR**: Merge Request (GitLab equivalent of GitHub Pull Request)
- **IID**: Internal ID (GitLab's project-scoped ID)
- **CI/CD**: Continuous Integration/Continuous Deployment
- **DevSecOps**: Development, Security, and Operations
- **SDK**: Software Development Kit
- **API**: Application Programming Interface
- **JSON-RPC**: JSON Remote Procedure Call

---

## Document History

| Version | Date | Author | Changes |
|---------|------|--------|---------|
| 1.0 | 2025-10-22 | Initial Draft | Complete PRD based on user requirements |

---

## Approval & Sign-off

| Role | Name | Date | Signature |
|------|------|------|-----------|
| Product Owner | | | |
| Technical Lead | | | |
| Engineering Manager | | | |

---

**End of Document**
