"""A parser for HCL2 implemented using the Lark parser"""
from pathlib import Path

from hcl2.transformer import DictTransformer

THIS_DIR = Path(__file__).absolute().resolve().parent
PARSER_FILE = THIS_DIR / "lark_parser.py"


def create_parser_file(parser_file: Path = PARSER_FILE) -> None:
    """
    Parsing the Lark grammar takes about 0.5 seconds. In order to improve performance we can cache the parser
    file. The below code caches the entire python file which is generated by Lark's standalone parser feature
    See: https://github.com/lark-parser/lark/blob/master/lark/tools/standalone.py

    Lark also supports serializing the parser config but the deserialize function did not work for me.
    The lark state contains dicts with numbers as keys which is not supported by json so the serialized
    state can't be written to a json file. Exporting to other file types would have required
    adding additional dependencies or writing a lot more code. Lark's standalone parser
    feature works great but it expects to be run as a separate shell command
    The below code copies some of the standalone parser generator code in a way that we can use
    """
    # This function is needed only if standalone parser is not yet compiled.
    # pylint: disable=import-outside-toplevel
    from lark import Lark
    from lark.tools.standalone import gen_standalone

    lark_file = THIS_DIR / "hcl2.lark"
    with open(parser_file, "w", encoding="utf-8") as parser_file_stream:
        lark_inst = Lark(lark_file.read_text(), parser="lalr", lexer="contextual")
        parser_file_stream.write("# mypy: ignore-errors\n")
        gen_standalone(lark_inst, out=parser_file_stream)


if not PARSER_FILE.exists():
    create_parser_file(PARSER_FILE)

# pylint: disable=wrong-import-position
# Lark_StandAlone needs to be imported after the above block of code because lark_parser.py might not exist
from hcl2.lark_parser import Lark_StandAlone

hcl2 = Lark_StandAlone(transformer=DictTransformer())
