import os
from functools import wraps
from typing import Any, List, Optional

import duckdb
import pandas as pd
from loguru import logger

from .exceptions import ConnectionError, QueryError


def attach_motherduck(func):
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        self.execute("ATTACH 'md:'")
        return func(self, *args, **kwargs)

    return wrapper


class HdDB:
    def __init__(self, motherduck_token="", read_only=False):
        try:
            if motherduck_token:
                os.environ["motherduck_token"] = motherduck_token
            self.conn = duckdb.connect(":memory:", read_only=read_only)
        except duckdb.Error as e:
            raise ConnectionError(f"Failed to connect to database: {e}")

    def set_motherduck_token(self, motherduck_token: str):
        os.environ["motherduck_token"] = motherduck_token

    def execute(
        self, query: str, parameters: Optional[List[Any]] = None
    ) -> duckdb.DuckDBPyConnection:
        return self.conn.execute(query, parameters)

    def create_database(self, dataframes: List[pd.DataFrame], names: List[str]):
        """
        Create in-memory database and create tables from a list of dataframes.

        :param dataframes: List of pandas DataFrames to create tables from
        :param names: List of names for the tables to be created
        :raises ValueError: If the number of dataframes doesn't match the number of table names
        :raises QueryError: If there's an error executing a query
        """
        if len(dataframes) != len(names):
            raise ValueError(
                "The number of dataframes must match the number of table names"
            )

        try:
            # self.conn.execute("CREATE TABLE hd_fields ();")
            for df, table_name in zip(dataframes, names):
                query = f"CREATE TABLE {table_name} AS SELECT * FROM df"
                self.execute(query)
            self.create_hd_tables()
            self.create_hd_fields()
        except duckdb.Error as e:
            raise QueryError(f"Error executing query: {e}")

    def create_hd_tables(self):
        try:
            self.execute(
                "CREATE TABLE hd_tables AS SELECT table_name AS id, table_name AS label, estimated_size AS nrow, column_count AS ncol from duckdb_tables();"
            )
        except duckdb.Error as e:
            logger.error(f"Error creating hd_tables: {e}")
            raise QueryError(f"Error creating hd_tables: {e}")

    # TODO: map duckdb data types to datasketch types
    def create_hd_fields(self):
        try:
            self.execute(
                "CREATE TABLE hd_fields AS SELECT  column_name AS id, column_name AS label, table_name AS table, data_type AS type from information_schema.columns WHERE table_name IN (SELECT id from hd_tables);"
            )
        except duckdb.Error as e:
            logger.error(f"Error creating hd_fields: {e}")
            raise QueryError(f"Error creating hd_fields: {e}")

    @attach_motherduck
    def upload_to_motherduck(self, org: str, db: str):
        """
        Upload the current database to Motherduck
        """
        if not os.environ.get("motherduck_token"):
            raise ValueError("Motherduck token has not been set")

        try:
            # https://motherduck.com/docs/key-tasks/loading-data-into-motherduck/loading-duckdb-database/
            self.execute(
                f"CREATE OR REPLACE DATABASE \"{org}__{db}\" from CURRENT_DATABASE();",
            )
        except duckdb.Error as e:
            logger.error(f"Error uploading database to MotherDuck: {e}")
            raise ConnectionError(f"Error uploading database to MotherDuck: {e}")

    @attach_motherduck
    def drop_database(self, org: str, db: str):
        """
        Delete a database stored in Motherduck

        :param org: Organization name
        :param db: Database name
        :raises ValueError: If Motherduck token is not set
        :raises ConnectionError: If there's an error deleting the database from Motherduck
        """
        if not os.environ.get("motherduck_token"):
            raise ValueError("Motherduck token has not been set")

        try:
            self.execute(f"DROP DATABASE {org}__{db};")
            logger.info(f"Database {org}__{db} successfully deleted from Motherduck")
        except duckdb.Error as e:
            logger.error(f"Error deleting database from MotherDuck: {e}")
            raise ConnectionError(f"Error deleting database from MotherDuck: {e}")

    def close(self):
        try:
            self.conn.close()
            logger.info("Database connection closed")
        except duckdb.Error as e:
            logger.error(f"Error closing connection: {e}")
