from typing import Optional, NamedTuple


class Condition(NamedTuple):
    """
    Helper class for declaring GCP IAM policy conditions

    Represents a condition attached to a binding with:
    - title (String): A short, descriptive title for the condition
    - expression (String) : A CEL (Common Expression Language) expression defining the condition logic
    - description (String, Optional): Optional human-readable description of the condition

    Example:
        Condition(
            title="TimeBoundAccess",
            expression="request.time < timestamp('2024-12-31T23:59:59Z')",
            description="Temporary access until end of year"
        )._asdict()
    """
    title: str
    expression: str
    description: str = ''


class GCPIAMPolicyBuilder:
    """
    A builder for GCP IAM policy

    [Usage]
    iam_policy = GCPIAMPolicyBuilder(version=3)

    iam_policy.add_binding(
        role="roles/viewer",
        members=[
            "user:alice@example.com",
            "serviceAccount:compute@example.com"
        ],
        condition=Condition(
            title="TimeBoundAccess",
            expression=|request.time < timestamp('2024-12-31T23:59:59Z')",
            description="Temporary access until end of year"
        )._asdict()
    )

    print(iam_policy.build())

    [Output]
    {
        "bindings": [
            {
                "role": "roles/viewer",
                "members": [
                    "user:alice@example.com",
                    "serviceAccount:compute@example.com"
                ],
                "condition": {
                    "title": "TimeBoundAccess",
                    "expression": "request.time < timestamp(\"2024-12-31T23:59:59Z\")",
                    "description": "Temporary access until end of year"
                }
            }
        ],
        "version": 3
    }

    [Notes]
        - Multiple conditions per binding are not supported, as GCP IAM currently allows only one condition per binding (as of May 2025).
        - auditConfigs (Used for configuring audit logging) are not supported
        - Role names, member identifiers, and condition expressions are not validated
    """

    def __init__(self, version: Optional[int] = None):
        """
        GCP IAM policy builder constructor

        :param version (Int): GCP IAM policy versions are one of the lesser-known but important aspects of using IAM conditions properly
                              Submit an IAM policy with a condition but no version will result in an error
        """
        self._bindings = []
        self._version = version
        self._etag = None

    def build(self) -> dict:
        policy = {'bindings': self._bindings}

        if self._etag is not None:
            policy['etag'] = self._etag

        if self._version is not None:
            policy['version'] = self._version

        return policy

    def add_binding(self, role: str, members: Optional[list[str]] = None, condition: Optional[dict] = None) -> 'GCPIAMPolicyBuilder':
        """ Add a new binding (Role + Optional Members + Optional Condition) """
        binding = {'role': role}
        if members:
            binding['members'] = members
        if condition:
            binding['condition'] = condition
        self._bindings.append(binding)
        return self

    def set_etag(self, etag: str) -> 'GCPIAMPolicyBuilder':
        """
        etag is a base64-encoded string used for optimistic concurrency control
        It ensures you're not unintentionally overwriting someone else's changes when updating a policy
        etag is automatically generated by GCP when creating an IAM policy

        Sets the etag value used for optimistic concurrency when updating policies

        :param etag: A string etag value returned from an existing policy
        """
        self._etag = etag
        return self
